package com.sap.tc.webdynpro.clientserver.uielib.graphics.api;

import java.net.URL;
import java.util.List;

/**
 * This interface provides geo routing services.
 * <p>A <b>route</b> denotes the complete itinerary from start to end. 
 * A <b>part<b> is a fragment of the itenary between two stops, 
 * a <b>segment</> is a fragment of a path that follows a street or road.
 * GEO ROUTING IS ONLY FUNCTIONAL IF AN IGS WITH ADDITIONAL GEO-SERVICES IS INSTALLED.
 * 
 * @author SAP
 * @SAPWebDynproPart 2
 */
public interface IWDGeoRouter
{
  /**
   * Set the URL of the IGS that provides the GIS services 
   */
  public void setIgsUrl(URL igsURL);
  
  /**
   * Clears all paramters of the geo router.
   */
  public void clearAll();
  
  /**
   * Calculates the route.
   */
  public void execute();
  
  /**
   * Returns any errors that occured during the processing of the route.
   */
  public String getErrors();
  
  /**
   * Adds a stop to the route, e.g. the start or end.
   */
  public void addStop(String id, String description, WDGeoPosition wdPosition);
  
  /**
   * Adds a stop to the route, e.g. the start or end.
   */
  public void addStop(WDGeoRouterStop wdStop);
  
  /**
   * Adds a property to the route. A property stands for extra information that
   * the geo service might need to process the route. The two most popular properties
   * are already covered by setQuickest and setShortest.
   */
  public void addProperty(String name, String value);
  
  /**
   * Calculate the quickest route.
   */
  public void setQuickest();
  
  /**
   * Calculate the shortest route.
   */
  public void setShortest();
  
  /**
   * Sets the desired language for descriptions.
   * @param ISO 2-letter country code, e.g. "de"
   */
  public void setLanguage(String language);
    
  /**
   * Returns the part count of the route, usually the number of stops minus one.
   */
  public int getPartCount();
  
  /**
   * Returns the geo border of a part.
   */
  public WDGeoBorder getPartBorder(int partIndex);
  
  /**
   * Returns the distance of a part.
   */
  public double getPartDistance(int partIndex);
  
  /**
   * Returns the duration of a part.
   */
  public double getPartDuration(int partIndex);
  
  /**
   * Returns the description of a part in the language determined by setLanguage.
   * @see #setLanguage
   */
  public String getPartDescription(int partIndex);
  
  /**
   * Returns the path of WDGeoPositions of a part.
   * @return a list WDGeoPositions
   */
  public List getPartPath(int partIndex);
  
  /**
   * Returns the number of segments of a part.
   */
  public int getSegmentCount(int partIndex);
  
  /**
   * Returns the distance of a segment.
   */
  public double getSegmentDistance(int partIndex, int segmentIndex);
  
  /**
   * Returns the duration of a segment.
   */
  public double getSegmentDuration(int partIndex, int segmentIndex);
  
  /**
   * Returns the description of a segment in the language determined by setLanguage.
   * @see #setLanguage
   */
  public String getSegmentDescription(int partIndex, int segmentIndex);
  
  /**
   * Returns the path of WDGeoPositions of a segment.
   * @return a list WDGeoPositions
   */
  public List getSegmentPath(int partIndex, int segmentIndex);
  
  /**
   * Returns the border of the whole route.
   */
  public WDGeoBorder getRouteBorder();
  
  /**
   * Returns the distance of the whole route.
   * @return distance
   */
  public double getRouteDistance();
  
  /**
   * Returns the duration of the whole route.
   * @return duration
   */
  public double getRouteDuration();
  
  /**
   * Returns the description of the whole route.
   * Note: this is not the concatenated description ot the parts but an optional
   * total description. To obtain a detailed description of the route, request the 
   * descriptions of the parts.
   * @see #getPartDescription
   */
  public String getRouteDescription();
  
  /**
   * Returns the path of WDGeoPositions of the whole route.
   * @return a list of WDGeoPositions
   */
  public List getRoutePath();
}
