/*
 * SAP Copyright (c) 2002
 * All rights reserved
 *
 * @version $Id: //tc/WebDynproRuntime/630_VAL_REL/src/_webdynpro_clientserver/java/com/sap/tc/webdynpro/clientserver/navigation/api/WDPortalNavigation.java#8 $
 */
package com.sap.tc.webdynpro.clientserver.navigation.api;

import com.sap.security.api.IUser;
import com.sap.tc.webdynpro.services.api.WDHelperFactory;

/**
 * <code>WDPortalNavigation</code> provides access to portal navigation functionality.
 * 
 * @author SAP
 * @SAPWebDynproPart 2
 */
public class WDPortalNavigation {

  private static final String OBJECT_VALUE_NAME = "ObjectValue";

  /**
   * Defines the name of the parameter containing the system of the business object.
   */
  public static final String OBN_SYSTEM = "obn.system";

  /**
   * Defines the name of the parameter containing the business object type.
   */
  public static final String OBN_BO_TYPE = "obn.bo_type";

  /**
   * Defines the name of the parameter containing the triggered operation.
   */
  public static final String OBN_OPERATION = "obn.operation";

  private static IWDPortalNavigation instance = (IWDPortalNavigation) WDHelperFactory.getInstance(IWDPortalNavigation.class);

  private WDPortalNavigation() {
  }

  /**
   * ****************************************************************************************************************
   * ABSOLUTE page navigation
   * ****************************************************************************************************************
   */

  /**
   * Performs a navigation to a absolute defined navigation target.
   *  
   * @param navigationTarget      the absolute navigation url to navigate to
   * @param mode                  the mode in which the navigation will be performed
   * @param windowFeatures        additional client windows properties (for external window only)
   * @param windowName            the name of the client window (for external window only)
   * @param historyMode           the used history mode (i.e. defined how the navigation target occurs in the navigation history)
   * @param targetTitle           the displayed title in the page titel bar (optional)
   * @param contextUrl            the used navigation context (optional)
   * @param businessParameters    optional (business) parameters like customerID=4711 
   * @param launcherParameters    optional (launcher) parameters like WebDynproNamespace=myNamespace
   * @param postParameters        defines whether or not the defined parameters are POST or GET parameters
   * @param useSAPLauncher        should be TRUE if the navigation destination uses the SAP application launcher                             (i.e. if the navigation destination is implemented using BSP, Web Dynpro, ITS or SAPGui). 
   */
  public static void navigateAbsolute(
    String navigationTarget,
    WDPortalNavigationMode mode,
    String windowFeatures,
    String windowName,
    WDPortalNavigationHistoryMode historyMode,
    String targetTitle,
    String contextUrl,
    String businessParameters,
    String launcherParameters,
    boolean postParameters,
    boolean useSAPLauncher) {

    instance.navigateAbsolute(
      navigationTarget,
      mode,
      windowFeatures,
      windowName,
      historyMode,
      targetTitle,
      contextUrl,
      businessParameters,
      launcherParameters,
      postParameters,
      useSAPLauncher);
  }

  /**
   * Performs a navigation to a absolute defined navigation target.
   * All defined parameters are handled as GET parameters.
   *  
   * @param navigationTarget the absolute navigation url to navigate to
   * @param mode             the mode in which the navigation will be performed
   * @param windowFeatures   additional client windows properties (for external window only)
   * @param windowName       the name of the client window (for external window only)
   * @param historyMode      the used history mode (i.e. defined how the navigation target occurs in the navigation history)
   * @param targetTitle      the displayed title in the page titel bar (optional)
   * @param contextUrl       the used navigation context (optional)
   * @param businessParameters  optional (business) parameters like customerID=4711 
   * @param useSAPLauncher   should be TRUE if the navigation destination uses the SAP application launcher  														(i.e. if the navigation destination is implemented using BSP, Web Dynpro, ITS or SAPGui). 
   */
  public static void navigateAbsolute(
    String navigationTarget,
    WDPortalNavigationMode mode,
    String windowFeatures,
    String windowName,
    WDPortalNavigationHistoryMode historyMode,
    String targetTitle,
    String contextUrl,
    String businessParameters,
    boolean useSAPLauncher) {
    navigateAbsolute(
      navigationTarget,
      mode,
      windowFeatures,
      windowName,
      historyMode,
      targetTitle,
      contextUrl,
      businessParameters,
      null,
      false,
			useSAPLauncher);
  }

  /**
   * Performs a navigation to a absolute defined navigation target.
   * The assumption is that this navigation target uses the SAP application launcher.
   * All defined parameters are handled as GET parameters.
   *  
   * @param navigationTarget the absolute navigation url to navigate to
   * @param mode             the mode in which the navigation will be performed
   * @param windowFeatures   additional client windows properties (for external window only)
   * @param windowName       the name of the client window (for external window only)
   * @param historyMode      the used history mode (i.e. defined how the navigation target occurs in the navigation history)
   * @param targetTitle      the displayed title in the page titel bar (optional)
   * @param contextUrl       the used navigation context (optional)
   * @param businessParameters  optional (business) parameters like customerID=4711 
   */
  public static void navigateAbsolute(
    String navigationTarget,
    WDPortalNavigationMode mode,
    String windowFeatures,
    String windowName,
    WDPortalNavigationHistoryMode historyMode,
    String targetTitle,
    String contextUrl,
    String businessParameters) {
    navigateAbsolute(
      navigationTarget,
      mode,
      windowFeatures,
      windowName,
      historyMode,
      targetTitle,
      contextUrl,
      businessParameters,
      null,
      false,
      true);
  }

  /**
   * Performs a navigation to a absolute defined navigation target.
   * The assumption is that this navigation target uses the SAP application launcher.
   * All defined parameters are handled as GET parameters.
   *  
   * @param navigationTarget the absolute navigation url to navigate to
   * @param mode             the mode in which the navigation will be performed
   * @param windowFeatures   additional client windows properties (for external window only)
   * @param windowName       the name of the client window (for external window only)
   * @param businessParameters  optional (business) parameters like customerID=4711 
   * @deprecated This method may be withdrawn with the first new NetWeaver release in 2006.
   *   Use one of the similar methods in this class that accept a parameter of type 
   *   <code>WDPortalNavigationHistoryMode</code> to define the needed history mode. 
   */
  public static void navigateAbsolute(
    String navigationTarget,
    WDPortalNavigationMode mode,
    String windowFeatures,
    String windowName,
    String businessParameters) {
    navigateAbsolute(
      navigationTarget,
      mode,
      windowFeatures,
      windowName,
      WDPortalNavigationHistoryMode.NO_HISTORY,
      null,
      null,
      businessParameters,
      null,
      false,
      true);
  }

  /**
   * Performs a navigation to a absolute defined navigation target.
   * The assumption is that this navigation target uses the SAP application launcher.
   * All defined parameters are handled as GET parameters.
   *  
   * @param navigationTarget the absolute navigation url to navigate to
   * @param mode             the mode in which the navigation will be performed
   * @param windowFeatures   additional client windows properties (for external window only)
   * @param windowName       the name of the client window (for external window only)
   * @param businessParameters  optional (business) parameters like customerID=4711 
   * @param useSAPLauncher   should be TRUE if the navigation destination uses the SAP application launcher 
   * 														(i.e. if the navigation destination is implemented using BSP, Web Dynpro, ITS or SAPGui). 
   * @deprecated This method may be withdrawn with the first new NetWeaver release in 2006.
   *   Use one of the similar methods in this class that accept a parameter of type 
   *   <code>WDPortalNavigationHistoryMode</code> to define the needed history mode. 
   */
  public static void navigateAbsolute(
    String navigationTarget,
    WDPortalNavigationMode mode,
    String windowFeatures,
    String windowName,
    String businessParameters,
    boolean useSAPLauncher) {
    navigateAbsolute(
      navigationTarget,
      mode,
      windowFeatures,
      windowName,
      WDPortalNavigationHistoryMode.NO_HISTORY,
      null,
      null,
      businessParameters,
      null,
      false,
      useSAPLauncher);
  }

  /**
   * Performs a navigation to a absolute defined navigation target.
   * The assumption is that this navigation target uses the SAP application launcher.
   * All defined parameters are handled as GET parameters.
   *  
   * @param navigationTarget the absolute navigation url to navigate to
   * @param mode            the mode in which the navigation will be performed
   * @param historyMode      the used history mode (i.e. defined how the navigation target occurs in the navigation history)
   * @param businessParameters  optional (business) parameters like customerID=4711 
   */
  public static void navigateAbsolute(
    String navigationTarget,
    WDPortalNavigationMode mode,
    WDPortalNavigationHistoryMode historyMode,
    String businessParameters) {
    navigateAbsolute(navigationTarget, mode, null, null, historyMode, null, null, businessParameters, null, false, true);
  }

  /**
   * ****************************************************************************************************************
   * RELATIVE page navigation
   * ****************************************************************************************************************
   */

  /**
   * Performs a navigation to a relative defined navigation target.
   *  
   * @param baseUrl               the base url (i.e. the start point for the relative navigation). Could be NULL - in this case the 
   *                                  start url is computed automatically (the current position of the navigation trigger is used).
   * @param levelsUp              the number of levels up
   * @param path                  a list of the children to navigate to
   * @param mode                  the mode in which the navigation will be performed
   * @param windowFeatures        additional client windows properties (for external window only)
   * @param windowName            the name of the client window (for external window only)
   * @param historyMode           the used history mode (i.e. defined how the navigation target occurs in the navigation history)
   * @param targetTitle           the displayed title in the page titel bar (optional)
   * @param contextUrl            the used navigation context (optional)
   * @param businessParameters    optional (business) parameters like customerID=4711 
   * @param launcherParameters    optional (launcher) parameters like WebDynproNamespace=myNamespace
   * @param postParameters        defines whether or not the defined parameters are POST or GET parameters      
   * @param useSAPLauncher        should be TRUE if the navigation destination uses the SAP application launcher 
   *                                (i.e. if the navigation destination is implemented using BSP, Web Dynpro, ITS or SAPGui). 
   */
  public static void navigateRelative(
    String baseUrl,
    int levelsUp,
    String[] path,
    WDPortalNavigationMode mode,
    String windowFeatures,
    String windowName,
    WDPortalNavigationHistoryMode historyMode,
    String targetTitle,
    String contextUrl,
    String businessParameters,
    String launcherParameters,
    boolean postParameters,
    boolean useSAPLauncher) {

    instance.navigateRelative(
      baseUrl,
      levelsUp,
      path,
      mode,
      windowFeatures,
      windowName,
      historyMode,
      targetTitle,
      contextUrl,
      businessParameters,
      launcherParameters,
      postParameters,
      useSAPLauncher);
  }

  /**
   * Performs a navigation to a relative defined navigation target.
   *  
   * @param baseUrl          the base url (i.e. the start point for the relative navigation). Could be NULL - in this case the 
   *                         	 start url is computed automatically (the current position of the navigation trigger is used).
   * @param levelsUp         the number of levels up
   * @param path             a list of the children to navigate to
   * @param mode             the mode in which the navigation will be performed
   * @param windowFeatures   additional client windows properties (for external window only)
   * @param windowName       the name of the client window (for external window only)
   * @param historyMode      the used history mode (i.e. defined how the navigation target occurs in the navigation history)
   * @param targetTitle      the displayed title in the page titel bar (optional)
   * @param contextUrl       the used navigation context (optional)
   * @param businessParameters  optional (business) parameters like customerID=4711 
   * @param useSAPLauncher   should be TRUE if the navigation destination uses the SAP application launcher 
   * 														(i.e. if the navigation destination is implemented using BSP, Web Dynpro, ITS or SAPGui). 
   */
  public static void navigateRelative(
    String baseUrl,
    int levelsUp,
    String[] path,
    WDPortalNavigationMode mode,
    String windowFeatures,
    String windowName,
    WDPortalNavigationHistoryMode historyMode,
    String targetTitle,
    String contextUrl,
    String businessParameters,
    boolean useSAPLauncher) {
    navigateRelative(
      baseUrl,
      levelsUp,
      path,
      mode,
      windowFeatures,
      windowName,
      historyMode,
      targetTitle,
      contextUrl,
      businessParameters,
      null,
      false,
      true);
  }

  /**
   * Performs a navigation to a relative defined navigation target.
   * The assumption is that this navigation target uses the SAP application launcher.
   *  
   * @param baseUrl          the base url (i.e. the start point for the relative navigation). Could be NULL - in this case the 
   *                         	 start url is computed automatically (the current position of the navigation trigger is used).
   * @param levelsUp         the number of levels up
   * @param path             a list of the children to navigate to
   * @param mode             the mode in which the navigation will be performed
   * @param windowFeatures   additional client windows properties (for external window)
   * @param windowName       the name of the client window (for external window)
   * @param historyMode      the used history mode (i.e. defined how the navigation target occurs in the navigation history)
   * @param targetTitle      the displayed title in the page titel bar (optional)
   * @param contextUrl       the used navigation context (optional)
   * @param businessParameters  optional (business) parameters like customerID=4711 
   */
  public static void navigateRelative(
    String baseUrl,
    int levelsUp,
    String[] path,
    String navigationTarget,
    WDPortalNavigationMode mode,
    String windowFeatures,
    String windowName,
    WDPortalNavigationHistoryMode historyMode,
    String targetTitle,
    String contextUrl,
    String businessParameters) {
    navigateRelative(
      baseUrl,
      levelsUp,
      path,
      mode,
      windowFeatures,
      windowName,
      historyMode,
      targetTitle,
      contextUrl,
      businessParameters,
      null,
      false,
      true);
  }

  /**
   * Performs a navigation to a relative defined navigation target.
   * The assumption is that this navigation target uses the SAP application launcher.
   *  
  * @param baseUrl          the base url (i.e. the start point for the relative navigation). Could be NULL - in this case the 
   *                         	 start url is computed automatically (the current position of the navigation trigger is used).
   * @param levelsUp         the number of levels up
   * @param path             a list of the children to navigate to
   * @param mode             the mode in which the navigation will be performed
   * @param historyMode      the used history mode (i.e. defined how the navigation target occurs in the navigation history)
   * @param businessParameters  optional (business) parameters like customerID=4711 
   */
  public static void navigateRelative(
    String baseUrl,
    int levelsUp,
    String[] path,
    WDPortalNavigationMode mode,
    WDPortalNavigationHistoryMode historyMode,
    String businessParameters) {
    navigateRelative(baseUrl, levelsUp, path, mode, null, null, historyMode, null, null, businessParameters, null, false, true);
  }

  /**
   * ****************************************************************************************************************
   * OBJECT BASED navigation
   * ****************************************************************************************************************
   */

  /**
   * Performs a navigation to the default operation of the defined businees object.
   * 
   * @param system              the system which defines the business object
   * @param businessobjectType  the business object type
   * @param objValue            the businees object value
   * @param businessParameters  optional (business) parameters like customerID=4711 
   * @param useSAPLauncher      should be TRUE if the navigation destination uses the SAP application launcher 
   * 															(i.e. if the navigation destination is implemented using BSP, Web Dynpro, ITS or SAPGui). 
   * @deprecated Do not use this method any longer, it may be withdrawn with the first new NetWeaver release in 2006. 
   *   Use <code>navigateToObject</code> without parameter <code>useSAPLauncher</code> instead.
   **/
  public static void navigateToObject(
    String system,
    String businessobjType,
    String objValue,
    String businessParameters,
    boolean useSAPLauncher) {

    instance.navigateToObject(system, businessobjType, objValue, businessParameters, useSAPLauncher);
  }

  /**
   * Performs a navigation to the default operation of the defined businees object.
   * 
   * @param system              the system which defines the business object
   * @param businessobjectType  the business object type
   * @param objValue            the business object value
   * @param businessParameters  optional (business) parameters like customerID=4711 
   */
  public static void navigateToObject(String system, String businessobjType, String objValue, String businessParameters) {
    navigateToObject(system, businessobjType, objValue, OBJECT_VALUE_NAME, businessParameters, false);
  }

  /**
   * Performs a navigation to the default operation of the defined businees object.
   * 
   * @param system              the system which defines the business object
   * @param businessobjectType  the business object type
   * @param objValue            the business object value
   * @param objValueName        the used name for the object value
   * @param businessParameters  optional (business) parameters like customerID=4711 #
   * @param forwardOBNMetaData  should be TRUE if the meta data of the OBN should be also forwarded.
   */
  public static void navigateToObject(
    String system,
    String businessobjType,
    String objValue,
    String objValueName,
    String businessParameters,
    boolean forwardOBNMetaData) {
    instance.navigateToObject(system, businessobjType, objValue, objValueName, businessParameters, forwardOBNMetaData);
  }

  /**
   * Performs a navigation to the specified operation of the defined businees object.
   * 
   * @param system              the system which defines the business object
   * @param businessobjectType  the business object type
   * @param objValue            the business object value
   * @param operation           the operation
   * @param businessParameters  optional (business) parameters like customerID=4711 
   */
  public static void navigateToObjectWithSpecificOperation(
    String system,
    String businessobjType,
    String objValue,
    String operation,
    String businessParameters) {
    navigateToObject(system, businessobjType, objValue, OBJECT_VALUE_NAME, businessParameters, false);
  }

  /**
   * Performs a navigation to the specific operation of the defined businees object.
   * 
   * @param system              the system which defines the business object
   * @param businessobjectType  the business object type
   * @param objValue            the business object value
   * @param objValueName        the used name for the object value
   * @param operation           the operation
   * @param businessParameters  optional (business) parameters like customerID=4711 
   * @param forwardOBNMetaData  should be TRUE if the meta data of the OBN should be also forwarded.
   */
  public static void navigateToObjectWithSpecificOperation(
    String system,
    String businessobjType,
    String objValue,
    String objValueName,
    String operation,
    String businessParameters,
    boolean forwardOBNMetaData) {
    instance.navigateToObject(system, businessobjType, objValue, objValueName, businessParameters, forwardOBNMetaData);
  }

/*

  /**
   * Checks whether or not there is a valid target defined for the default operation of the specified business object.
   * 
   * @param system           the system which defines the business object
   * @param businessobjType  the business object type
   * @param user             the user - if the user is NULL the current user is used automatically.
   * @return                 TRUE if there is a valid target for the default operation; FALSE otherwise.
   *
  public static boolean targetExistsForDefaultOperation(String system, String businessobjType, IUser user) {
    return instance.targetExistsForDefaultOperation(system, businessobjType, user);
  }

  /**
   * Checks whether or not there is a valid target defined for the specified operation of the specified business object.
   * 
   * @param system           the system which defines the business object
   * @param businessobjType  the business object type
   * @param operation        the operation
   * @param user             the user - if the user is NULL the current user is used automatically.
   * @return                 TRUE if there is a valid target for the specific operation; FALSE otherwise.
   *
  public static boolean targetExistsForOperation(String system, String businessobjType, String operation, IUser user) {
    return instance.targetExistsForOperation(system, businessobjType, operation, user);
  }

  /**
   * Returns the list of operations of the specified business object with a valid target.
   * @param system           the system which defines the business object
   * @param businessobjType  the business object type
   * @param user             the user - if the user is NULL the current user is used automatically.
   * @return                 an array containing the names of the operations with a valid target. These names could be used to 
   *                           trigger a navigation with <code>navigateToObjectWithSpecificOperation</code> 
   *
  public static String[] getListOfOperationsWithValidTarget(String system, String businessobjType, IUser user) {
    return instance.getListOfOperationsWithValidTarget(system, businessobjType, user);
  }
*/


  public interface IWDPortalNavigation {
    void navigateAbsolute(
      String navigationTarget,
      WDPortalNavigationMode mode,
      String windowFeatures,
      String windowName,
      WDPortalNavigationHistoryMode historyMode,
      String targetTitle,
      String contextUrl,
      String businessParameters,
      String applicationParameters,
      boolean postParameters,
      boolean useSAPLauncher);

    void navigateRelative(
      String baseUrl,
      int levelsUp,
      String[] path,
      WDPortalNavigationMode mode,
      String windowFeatures,
      String windowName,
      WDPortalNavigationHistoryMode historyMode,
      String targetTitle,
      String contextUrl,
      String businessParameters,
      String applicationParameters,
      boolean postParameters,
      boolean useSAPLauncher);

    void navigateToObject(String system, String businessobjType, String objValue, String parameter, boolean useSAPLauncher);
    void navigateToObject(
      String system,
      String businessobjType,
      String objValue,
      String objValueName,
      String parameter,
      boolean forwardOBNMetaData);

    /*
    boolean targetExistsForDefaultOperation(String system, String businessobjType, IUser user);
    boolean targetExistsForOperation(String system, String businessobjType, String operation, IUser user);

    String[] getListOfOperationsWithValidTarget(String system, String businessobjType, IUser user);
    */

    void navigateToObjectWithSpecificOperation(
      String system,
      String businessobjType,
      String objValue,
      String objValueName,
      String operation,
      String parameter,
      boolean forwardOBNMetaData);
  }

}
