/*******************************************************************************
 * Copyright (c) 2000, 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.pde.core;

import java.io.InputStream;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.*;
/**
 * A generic model. Classes that implement this
 * interface are expected to be able to:
 * <ul>
 * <li>Load from an input stream
 * <li>Reload (reset, load, fire 'world change')
 * <li>Dispose (clear all the data and reset)
 * <li>Be associated with a resource (optional)
 * </ul>
 * If a model is not created from a workspace
 * resource file, its underlying resource will
 * be <samp>null</samp>.
 * <p>
 * <b>Note:</b> This interface is part of an interim API that is still under development and expected to
 * change significantly before reaching stability. It is being made available at this early stage to solicit feedback
 * from pioneering adopters on the understanding that any code that uses this API will almost certainly be broken
 * (repeatedly) as the API evolves.
 * </p>
 */
public interface IModel extends IAdaptable {
	/**
	 * Releases all the data in this model and
	 * clears the state. A disposed model
	 * can be returned to the normal state
	 * by reloading.
	 * <p>
	 * <b>Note:</b> This method is part of an interim API that is still under development and expected to
	 * change significantly before reaching stability. It is being made available at this early stage to solicit feedback
	 * from pioneering adopters on the understanding that any code that uses this API will almost certainly be broken
	 * (repeatedly) as the API evolves.
	 * </p>
	 */
	void dispose();
	/**
	 * Returns a string found in the resource
	 * bundle associated with this model
	 * for the provided key.
	 *
	 * @param key the name to use for bundle lookup
	 * @return the string for the key in the resource bundle,
	 * or the key itself if not found
	 * <p>
	 * <b>Note:</b> This method is part of an interim API that is still under development and expected to
	 * change significantly before reaching stability. It is being made available at this early stage to solicit feedback
	 * from pioneering adopters on the understanding that any code that uses this API will almost certainly be broken
	 * (repeatedly) as the API evolves.
	 * </p>
	 */
	String getResourceString(String key);
	/**
	 * Returns a workspace resource that this model
	 * is created from. Load/reload operations are
	 * not directly connected with the resource
	 * (although they can be). In some cases,
	 * models will load from a buffer (an editor
	 * document) rather than a resource. However,
	 * the buffer will eventually be synced up
	 * with this resource.
	 * <p>With the caveat of stepped loading, 
	 * all other properties of
	 * the underlying resource could
	 * be used directly (path, project etc.).
	 *
	 * @return a workspace resource (file)
	 * that this model is associated with,
	 * or <samp>null</samp> if the model
	 * is not created from a resource.
	 * <p>
	 * <b>Note:</b> This method is part of an interim API that is still under development and expected to
	 * change significantly before reaching stability. It is being made available at this early stage to solicit feedback
	 * from pioneering adopters on the understanding that any code that uses this API will almost certainly be broken
	 * (repeatedly) as the API evolves.
	 * </p>
	 */
	public IResource getUnderlyingResource();
	/**
	 * Tests if this model has been disposed.
	 * Disposed model cannot be used until
	 * it is loaded/reloaded.
	 * @return true if the model has been disposed
	 * <p>
	 * <b>Note:</b> This method is part of an interim API that is still under development and expected to
	 * change significantly before reaching stability. It is being made available at this early stage to solicit feedback
	 * from pioneering adopters on the understanding that any code that uses this API will almost certainly be broken
	 * (repeatedly) as the API evolves.
	 * </p>
	 */
	boolean isDisposed();
	/**
	 * Tests if this model can be modified. Modification
	 * of a model that is not editable will result
	 * in CoreException being thrown.
	 * @return true if this model can be modified
	 * <p>
	 * <b>Note:</b> This method is part of an interim API that is still under development and expected to
	 * change significantly before reaching stability. It is being made available at this early stage to solicit feedback
	 * from pioneering adopters on the understanding that any code that uses this API will almost certainly be broken
	 * (repeatedly) as the API evolves.
	 * </p>
	 */
	boolean isEditable();

	/**
	 * Tests if this model is loaded and can be used.
	 * @return true if the model has been loaded
	 * <p>
	 * <b>Note:</b> This method is part of an interim API that is still under development and expected to
	 * change significantly before reaching stability. It is being made available at this early stage to solicit feedback
	 * from pioneering adopters on the understanding that any code that uses this API will almost certainly be broken
	 * (repeatedly) as the API evolves.
	 * </p>
	 */
	boolean isLoaded();

	/**
	 * Tests if this model is in sync with the storage
	 * object it was loaded from. Models loaded
	 * from resources are in sync if underlying
	 * resources are in sync. Models loaded
	 * from files on the file systems are in sync
	 * if the time stamp matches the model time stamp.
	 * @return true if the model is in sync
	 * with the file system.
	 * <p>
	 * <b>Note:</b> This method is part of an interim API that is still under development and expected to
	 * change significantly before reaching stability. It is being made available at this early stage to solicit feedback
	 * from pioneering adopters on the understanding that any code that uses this API will almost certainly be broken
	 * (repeatedly) as the API evolves.
	 * </p>
	 */
	boolean isInSync();
	
	/**
	 * Tests if this model valid. When models
	 * are loaded from the file, they may pass the
	 * syntax error checking and load all the model objects.
	 * However, some of the objects may contain invalid
	 * values that make the model unusable.
	 * This method should return <samp>true</samp> only
	 * if the model can be safely used in all
	 * computations.
	 * <p>
	 * <b>Note:</b> This method is part of an interim API that is still under development and expected to
	 * change significantly before reaching stability. It is being made available at this early stage to solicit feedback
	 * from pioneering adopters on the understanding that any code that uses this API will almost certainly be broken
	 * (repeatedly) as the API evolves.
	 * </p>
	 */
	boolean isValid();	

	/**
	 * Returns the last modification time stamp.
	 * The model itself does not have the time stamp.
	 * It is 'borrowed' from the underlying physical
	 * object.
	 * @return the time stamp of the underlying physical object.
	 * <p>
	 * <b>Note:</b> This method is part of an interim API that is still under development and expected to
	 * change significantly before reaching stability. It is being made available at this early stage to solicit feedback
	 * from pioneering adopters on the understanding that any code that uses this API will almost certainly be broken
	 * (repeatedly) as the API evolves.
	 * </p>
	 */
	long getTimeStamp();
	/**
	 * Loads the model directly from an underlying
	 * resource. This method does nothing if
	 * this model has no underlying resource
	 * or if there is a buffer stage between
	 * the model and the resource.
	 * <p>This method will throw a CoreException
	 * if errors are encountered during
	 * the loading.
	 * <p>
	 * <b>Note:</b> This method is part of an interim API that is still under development and expected to
	 * change significantly before reaching stability. It is being made available at this early stage to solicit feedback
	 * from pioneering adopters on the understanding that any code that uses this API will almost certainly be broken
	 * (repeatedly) as the API evolves.
	 * </p>
	 */
	public void load() throws CoreException;
	/**
	 * Loads the model from the provided input stream.
	 * This method throws a CoreException if
	 * errors are encountered during the loading.
	 * Upon succesful load, 'isLoaded()' should
	 * return <samp>true</samp>.
	 *
	 * @param source an input stream that should
	 * be parsed to load the model
	 * @param outOfSync if true, time stamp will not be
	 * updated to maintain out-of-sync state of the model.
	 * <p>
	 * <b>Note:</b> This method is part of an interim API that is still under development and expected to
	 * change significantly before reaching stability. It is being made available at this early stage to solicit feedback
	 * from pioneering adopters on the understanding that any code that uses this API will almost certainly be broken
	 * (repeatedly) as the API evolves.
	 * </p>
	 */
	public void load(InputStream source, boolean outOfSync) throws CoreException;
	/**
	 * Reload is a version of 'load' operation that has
	 * the following steps:
	 * <ul>
	 * <li>Reset the model
	 * <li>Load the model
	 * <li>Fire "world changed" event
	 * </ul>
	 * Reload operation is used when a model that
	 * is already in use is invalidated by a change
	 * in the underlying buffer or resource.
	 * Since we don't know the extent of the
	 * change, the only safe thing to do is
	 * to reparse the buffer to sync up. The event that is
	 * subsequently fired should be used by listeners to
	 * discard all caches and/or fully refresh views
	 * that shows any portion of the model.
	 * 
	 * @param source an input stream that should
	 * be parsed to load the model.
	 * @param outOfSync if true, time stamp will not be updated
	 * to maintain out-of-sync state of the model.
	 * <p>
	 * <b>Note:</b> This method is part of an interim API that is still under development and expected to
	 * change significantly before reaching stability. It is being made available at this early stage to solicit feedback
	 * from pioneering adopters on the understanding that any code that uses this API will almost certainly be broken
	 * (repeatedly) as the API evolves.
	 * </p>
	 */
	public void reload(InputStream source, boolean outOfSync) throws CoreException;
}
