/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_modules/_jom/_core/src/com/sap/tc/jtools/jlint/jom/util/InheritanceTool.java#2 $
 */

package com.sap.tc.jtools.jlint.jom.util;

import com.sap.tc.jtools.jlint.jom.interfaces.IReferenceTypeBinding;

/**
 * @author $Author: p4trans $
 */
public class InheritanceTool {

  private InheritanceTool() {
  }

  /**
   * recursively check the inheritance tree of a type binding
   * to see if it implements an interface
   *  
   * @param bnd the binding to be checked 
   * @param interfaceName a fully qualified interface name in dot-notation
   * 
   * @return true if bnd implements interfaceName
   */

  public static boolean bndImplements(IReferenceTypeBinding bnd, String interfaceName) {
    if (bnd == null || interfaceName == null)
      return false;
    if (bindingIsIntf(bnd, interfaceName))
      return true;
    // 1: recursively check all interfaces up the inheritance tree
    IReferenceTypeBinding[] intfs = bnd.getInterfaces();
    if (intfs != null) {
      for (int i = 0; i < intfs.length; i++) {
        if (bndImplements(intfs[i], interfaceName))
          return true;
      }
    }
    // 2: recursively check super classes up the inheritance tree
    if (bnd.getSuperclass() != null) {
      return bndImplements(bnd.getSuperclass(), interfaceName);
    }
    return false;
  }

  /**
   * recursively check the inheritance tree of a type binding
   * to see if it extends a class
   *  
   * @param bnd the binding to be checked 
   * @param superClassName a fully qualified class name in dot-notation
   * 
   * @return true if bnd implements interfaceName
   */
  public static boolean bndExtends(IReferenceTypeBinding bnd, String superClassName) {
    if (bnd == null || superClassName == null)
      return false;
    if (bindingIsClass(bnd, superClassName))
      return true;
    // 2: recursively check super classes up the inheritance tree
    if (bnd.getSuperclass() != null) {
      return bndExtends(bnd.getSuperclass(), superClassName);
    }
    return false;
  }

  private static boolean bindingIsClass(IReferenceTypeBinding bnd, String className) {
    return bnd.isClass() && className.equals(NameTool.getFullClassName(bnd));
  }

  private static boolean bindingIsIntf(
    IReferenceTypeBinding bnd,
    String interfaceName) {
    return bnd.isInterface()
      && interfaceName.equals(NameTool.getFullClassName(bnd));
  }

}
