/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_modules/_jom/_core/src/com/sap/tc/jtools/jlint/jom/metrics/MetricInfo.java#3 $
 */

package com.sap.tc.jtools.jlint.jom.metrics;

import com.sap.tc.jtools.jtci.interfaces.ResultInterface;

/**
 * This class is used to identify a certain float measurement
 * value for which statistics such as average, sum, max, min,
 * standard deviation will be computed.
 * 
 * @author D037913
 */
public class MetricInfo {

  /*
   * Some int constants used to specify the statistic values 
   * which will be computed for this metric.
   * Use bitwise or to combine.
   */
    /**
     * int constant representing the sum of a measured quantity.
     */
  public static final int SUM = 0x1;
  /**
   * int constant representing the average of a measured quantity.
   */
  public static final int AVERAGE = 0x2;
  /**
   * int constant representing the maximum of a measured quantity.
   */
  public static final int MAX = 0x4;
  /**
   * int constant representing the minimum of a measured quantity.
   */
  public static final int MIN = 0x8;
  /**
   * int constant representing the standard deviation of a measured quantity.
   */
  public static final int STD_DEV = 0x10;

  private String name;
  private int statistics;
  private int priority;

  /**
   * @param name (metric-wide) unique name of the float metric.
   *        @see JomMetricVisitor#addMetricValue(String, float, Position)
   * @param statistics bitwise ORed statistic int constants indicating 
   *        which statistical values will be computed. 
   *        @see MetricInfo#SUM
   *        @see MetricInfo#AVERAGE
   *        @see MetricInfo#MAX
   *        @see MetricInfo#MIN
   *        @see MetricInfo#STD_DEV
   */
  public MetricInfo(String name, int statistics) {
    this(name, statistics, ResultInterface.SEVERITY_INFO);
  }

  /**
   * @param name (metric-wide) unique name of the float metric.
   *        @see JomMetricVisitor#addMetricValue(String, float, Position)
   * @param statistics bitwise ORed statistic int constants indicating 
   *        which statistical values will be computed. 
   *        @see MetricInfo#SUM
   *        @see MetricInfo#AVERAGE
   *        @see MetricInfo#MAX
   *        @see MetricInfo#MIN
   *        @see MetricInfo#STD_DEV
   * @param prio message priority 
   */
  public MetricInfo(String name, int statistics, int prio) {
    if (statistics < SUM
      || statistics > (SUM | AVERAGE | MAX | MIN | STD_DEV)) {
      throw new IllegalArgumentException("unknown int statistic constant: " + statistics); //$NON-NLS-1$
    }
    if (name == null || name.length() == 0)
      throw new IllegalArgumentException("illegal metricinfo name: " + name); //$NON-NLS-1$
    this.name = name;
    this.statistics = statistics;
    this.priority = prio;
  }

  public String getName() {
    return name;
  }

  public int getStatistics() {
    return statistics;
  }

  public int getPriority() {
    return priority;
  }

}
