/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_modules/_jom/_core/src/com/sap/tc/jtools/jlint/jom/metrics/Metric.java#2 $
 */

package com.sap.tc.jtools.jlint.jom.metrics;

import java.text.NumberFormat;
import java.util.Locale;
import java.util.Properties;

import com.sap.tc.jtools.jlint.Result;

/**
 * This class represents a float measurement quantity to
 * which measured values can be added.
 * Statistics such as average, standard deviation etc. are computed internally
 * and can be accessed via respective getter methods. 
 * 
 * @author D037913
 *
 */
public class Metric {


  public static final String PARAM_INFO_LEVEL_SUFFIX = "_INFO_LEVEL";
  public static final String PARAM_WARNING_LEVEL_SUFFIX = "_WARNING_LEVEL";
  public static final String PARAM_ERROR_LEVEL_SUFFIX = "_ERROR_LEVEL";

  public static final String PARAM_INTERNAL_DATA = "INTERNAL_DATA"; //$NON-NLS-1$
  public static final String PARAM_METRIC_NAME = "METRIC_NAME"; //$NON-NLS-1$
  public static final String PARAM_SUM = "SUM"; //$NON-NLS-1$
  public static final String PARAM_QUADRATIC_SUM = "QUADRATIC_SUM"; //$NON-NLS-1$
  public static final String PARAM_MIN = "MIN"; //$NON-NLS-1$
  public static final String PARAM_MAX = "MAX"; //$NON-NLS-1$
  public static final String PARAM_COUNT = "COUNT"; //$NON-NLS-1$
  public static final String PARAM_AVERAGE = "AVERAGE"; //$NON-NLS-1$
  public static final String PARAM_STD_DEV = "STD_DEV"; //$NON-NLS-1$
  public static final String PARAM_VALUE = "VALUE"; //$NON-NLS-1$

  private static final int MAX_FRACTION_DIGITS = 2;
  
  public static final NumberFormat formatter =
    NumberFormat.getInstance(Locale.ENGLISH);

  static {
    formatter.setGroupingUsed(false);
    formatter.setMaximumFractionDigits(MAX_FRACTION_DIGITS);
  }

  private String name;
  private int statistics;

  private float infoThreshold;
  private float warningThreshold;
  private float errorThreshold;

  private boolean infoThresholdSet = false;
  private boolean warningThresholdSet = false;
  private boolean errorThresholdSet = false;

  private int counter = 0;
  private float sum = 0f;
  private float quadraticSum = 0f;
  private float max = 0f;
  private float min = 0f;
  
  private int prio;

  public Metric(MetricInfo metricInfo) {
    this.statistics = metricInfo.getStatistics();
    this.name = metricInfo.getName();
    this.prio = metricInfo.getPriority();
  }

  public void resetValues() {
    counter = 0;
    sum = 0f;
    quadraticSum = 0f;
    max = 0f;
    min = 0f;
  }

  public String getName() {
    return name;
  }

  public int addValue(float value) {
    max = counter == 0 ? value : Math.max(value, max);
    min = counter == 0 ? value : Math.min(value, min);
    counter++;
    sum += value;
    quadraticSum += value * value;

    if (value >= errorThreshold && errorThresholdSet) {
      return Result.SEVERITY_ERROR;
    } else if (value >= warningThreshold && warningThresholdSet) {
      return Result.SEVERITY_WARNING;
    } else if (value >= infoThreshold && infoThresholdSet) {
      return Result.SEVERITY_INFO;
    } else {
      return -1;
    }
  }

  public void addValues(Properties p) {
    float sum = Float.parseFloat(p.getProperty(PARAM_SUM).trim());
    float quadraticSum =
      Float.parseFloat(p.getProperty(PARAM_QUADRATIC_SUM).trim());
    float newMin = Float.parseFloat(p.getProperty(PARAM_MIN).trim());
    float newMax = Float.parseFloat(p.getProperty(PARAM_MAX).trim());
    int count = Integer.parseInt(p.getProperty(PARAM_COUNT).trim());

    this.sum += sum;
    this.quadraticSum += quadraticSum;
    this.min = counter == 0 ? newMin : Math.min(min, newMin);
    this.max = counter == 0 ? newMax : Math.max(max, newMax);
    counter += count;
  }

  public int getStatistics() {
    return statistics;
  }

  public int count() {
    return counter;
  }

  public float getMax() {
    return max;
  }

  public float getMin() {
    return min;
  }

  public float getAverage() {
    if (counter == 0)
      return 0f;
    return sum / counter;
  }

  public float getSum() {
    return sum;
  }

  public float getStandardDeviation() {
    if (counter == 0)
      return 0f;
    float avg = getAverage();
    return (float) Math.sqrt(quadraticSum / counter - avg * avg);
  }

  public void setInfoThreshold(float infoLevel) {
    infoThreshold = infoLevel;
    infoThresholdSet = true;
  }

  public void setWarningThreshold(float warningLevel) {
    warningThreshold = warningLevel;
    warningThresholdSet = true;
  }

  public void setErrorThreshold(float errorLevel) {
    errorThreshold = errorLevel;
    errorThresholdSet = true;
  }

  public Properties getMessageProperties() {
    Properties p = new Properties();
    p.setProperty(PARAM_INTERNAL_DATA, String.valueOf(true));
    p.setProperty(PARAM_METRIC_NAME, name);
    p.setProperty(PARAM_SUM, formatter.format(sum));
    p.setProperty(PARAM_QUADRATIC_SUM, formatter.format(quadraticSum));
    p.setProperty(PARAM_MIN, formatter.format(min));
    p.setProperty(PARAM_MAX, formatter.format(max));
    p.setProperty(PARAM_COUNT, String.valueOf(counter));
    p.setProperty(PARAM_AVERAGE, formatter.format(getAverage()));
    p.setProperty(PARAM_STD_DEV, formatter.format(getStandardDeviation()));
    return p;
  }

  public int getPriority() {
    return prio;
  }

}
