/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_modules/_jom/_core/src/com/sap/tc/jtools/jlint/jom/metrics/JomMetricVisitor.java#2 $
 */

package com.sap.tc.jtools.jlint.jom.metrics;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import com.sap.tc.jtools.jlint.Result;
import com.sap.tc.jtools.jlint.java.JavaFileArrayTestObject;
import com.sap.tc.jtools.jlint.jom.JomTestVisitor;
import com.sap.tc.jtools.jlint.jom.interfaces.ICompilationUnit;
import com.sap.tc.jtools.jlint.jom.interfaces.Position;
import com.sap.tc.jtools.jtci.ParameterTool;
import com.sap.tc.jtools.jtci.TestObject;
import com.sap.tc.jtools.jtci.interfaces.ParameterInterface;
import com.sap.tc.jtools.jtci.interfaces.ResultInterface;

/**
 * This class represents a Java metric. It extends the normal
 * JomTestVisitor with additional methods for measuring any
 * float quantity metric.
 * Statistics like average, sum, min, max, standard deviation
 * will be automatically created for each metric.
 *   
 * @author D037913
 */

public abstract class JomMetricVisitor extends JomTestVisitor {

  private Metric[] metrics;
  private Map name2MetricMap = new HashMap();

  private static final String MSG_KEY_0 = "metric.0";
  private static final String MSG_KEY_CLASSDATA = "METRIC_CLASS_DATA";
  private static final String MSG_KEY_PREFIX = "metric.";

  public JomMetricVisitor() {
    MetricInfo[] metricInfos = getMetricInfos();
    if (metricInfos == null || metricInfos.length == 0)
      throw new IllegalArgumentException("getMetricInfos() must not return null or an empty array!");
    metrics = new Metric[metricInfos.length];
    for (int i = 0; i < metricInfos.length; i++) {
      metrics[i] = new Metric(metricInfos[i]);
      name2MetricMap.put(metricInfos[i].getName(), metrics[i]);
    }
  }

  /**
   * Subclasses must implement to specify their metric(s).
   * A metric is a float measurement quantity for which
   * statistics such as sum, average, standard deviation etc.
   * will be computed.
   * For each metric name, you can optionally specify info, warning
   * and/or error thresholds with default values for which individual 
   * messages will be created.
   * E.g. if your metric name is "MyMetric", you would
   * specify as the signature of your metric in the test descriptor
   * XML file:
   * 
   * <pre><code>
   *   &lt;INPUT_PARAMETER NAME="MYMETRIC_INFO_LEVEL" TYPE="FLOAT"&gt;
   *     &lt;VALUE&gt;10&lt;/VALUE&gt;
   *   &lt;/INPUT_PARAMETER&gt;
   *   &lt;INPUT_PARAMETER NAME="MYMETRIC_WARNING_LEVEL" TYPE="FLOAT"&gt;
   *     &lt;VALUE&gt;20&lt;/VALUE&gt;
   *   &lt;/INPUT_PARAMETER&gt;
   *   &lt;INPUT_PARAMETER NAME="MYMETRIC_ERROR_LEVEL" TYPE="FLOAT"&gt;
   *     &lt;VALUE&gt;30&lt;/VALUE&gt;
   *   &lt;/INPUT_PARAMETER&gt;
   *  </code></pre>
   * 
   * @see JomMetricVisitor#addMetricValue(String, float, Position)
   */
  protected abstract MetricInfo[] getMetricInfos();

  /**
   * Subclasses use this method for notifying JomMetricVisitor
   * of a single float value that has been measured for the given metric.
   * All statistics like sum, average etc. will be computed from these
   * values.
   *  
   * @param metricName name of the metric as specified in @see JomMetricVisitor#getMetricInfos()
   * @param value a single measured value for the specified metric
   * @param pos position in source (may be null)
   */
  protected final void addMetricValue(
    String metricName,
    float value,
    Position pos) {
    Metric metric = getMetric(metricName);
    int severity = metric.addValue(value);
    if (severity != -1 && metric.getPriority() <= ResultInterface.SEVERITY_INFO) {
      Properties msgProps = new Properties();
      msgProps.setProperty(Metric.PARAM_VALUE, Metric.formatter.format(value));
      msgProps.setProperty(Metric.PARAM_METRIC_NAME, metric.getName());
      addError(MSG_KEY_0, "", msgProps, severity, pos);
    }
  }

  /* internal use */
  public void setParameters(
    ParameterInterface[] parameters,
    TestObject testObject) {
    super.setParameters(parameters, testObject);
    // read threshold values for metrics (if set)
    for (int i = 0; i < metrics.length; i++) {
      String upperCaseName = metrics[i].getName().toUpperCase();
      ParameterInterface infoParam =
        getParameter(upperCaseName + Metric.PARAM_INFO_LEVEL_SUFFIX);
      if (infoParam != null
        && ParameterTool.PAR_TYPE_FLOAT.equals(infoParam.getType())) {
        metrics[i].setInfoThreshold(
          ((Float) infoParam.getValue()).floatValue());
      }
      ParameterInterface warningParam =
        getParameter(upperCaseName + Metric.PARAM_WARNING_LEVEL_SUFFIX);
      if (warningParam != null
        && ParameterTool.PAR_TYPE_FLOAT.equals(warningParam.getType())) {
        metrics[i].setWarningThreshold(
          ((Float) warningParam.getValue()).floatValue());
      }
      ParameterInterface errorParam =
        getParameter(upperCaseName + Metric.PARAM_ERROR_LEVEL_SUFFIX);
      if (errorParam != null
        && ParameterTool.PAR_TYPE_FLOAT.equals(errorParam.getType())) {
        metrics[i].setErrorThreshold(
          ((Float) errorParam.getValue()).floatValue());
      }
    }
  }

  /* internal use */
  public final void postVisit(ICompilationUnit unit) {
    for (int i = 0; i < metrics.length; i++) {
      // only add really measured data
      if (metrics[i].count() > 0) {
        addError(MSG_KEY_CLASSDATA, "",
        //$NON-NLS-1$
        metrics[i].getMessageProperties(), metrics[i].getPriority(), null);
      }
      metrics[i].resetValues();
    }
  }

  /* internal use */
  public final ResultInterface[] processErrors(ResultInterface[] results) {
    List newResults = new ArrayList();
    String myName = getName();
    // set all metric counters to zero
    for (int i = 0; i < metrics.length; i++) {
      metrics[i].resetValues();
    }
    for (int i = 0; i < results.length; i++) {
      // sanity check
      if (!myName.equals(results[i].getTestName()))
        continue;
      Properties props = results[i].getErrorParameters();
      if (props.getProperty(Metric.PARAM_INTERNAL_DATA) == null) {
        newResults.add(results[i]);
      } else {
        try {
          Metric metric =
            getMetric(props.getProperty(Metric.PARAM_METRIC_NAME));
          metric.addValues(props);
        } catch (Exception e) {
          logException(e);
        }
      }
    }
    for (int i = 0; i < metrics.length; i++) {
      // only add real measured data
      if (metrics[i].count() == 0)
        continue;
      Properties msgProps = metrics[i].getMessageProperties();
      msgProps.setProperty(
        ParameterTool.PAR_MSG_KEY,
        MSG_KEY_PREFIX + metrics[i].getStatistics());
      newResults.add(
        new Result(
          getName(),
          JavaFileArrayTestObject.JAVA_FILE_LIST,
          "",
          null,
          metrics[i].getPriority(),
          "",
          msgProps));
    }
    return (ResultInterface[]) newResults.toArray(new ResultInterface[0]);
  }

  private Metric getMetric(String name) {
    Metric m = (Metric) name2MetricMap.get(name);
    if (m == null)
      throw new IllegalArgumentException("unknown metric: " + name); //$NON-NLS-1$
    return m;
  }

}
