/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_modules/_jom/_core/src/com/sap/tc/jtools/jlint/jom/ciwrappers2_1/ReferenceTypeBinding.java#2 $
 */

package com.sap.tc.jtools.jlint.jom.ciwrappers2_1;

import org.eclipse.jdt.internal.compiler.lookup.ReferenceBinding;
import org.eclipse.jdt.internal.compiler.lookup.BaseTypeBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.IBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.IMethodBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.IPackageBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.IReferenceTypeBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.IVariableBinding;

public class ReferenceTypeBinding implements IReferenceTypeBinding {

  private ReferenceBinding binding;
  private String name = null;

  public ReferenceTypeBinding(ReferenceBinding binding) {
    this.binding = binding;
  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITypeBinding#isPrimitive()
   */
  public boolean isPrimitive() {
    return false;
  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITypeBinding#isNullType()
   */
  public boolean isNullType() {
    return binding.equals(BaseTypeBinding.NullBinding);
  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITypeBinding#isArray()
   */
  public boolean isArray() {
    return false;
  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITypeBinding#isClass()
   */
  public boolean isClass() {
    return binding.isClass();
  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITypeBinding#isInterface()
   */
  public boolean isInterface() {
    return binding.isInterface();
  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.IBinding#getName()
   */
  public String getIdentifier() {
    if (isAnonymous()) {
      if (binding.superInterfaces() == ReferenceBinding.NoSuperInterfaces)
        return ANONYMOUS_CLASS_NAME_PREFIX
          + new String(binding.superclass().readableName())
          + ANONYMOUS_CLASS_NAME_SUFFIX;
      else
        return ANONYMOUS_CLASS_NAME_PREFIX
          + new String(binding.superInterfaces()[0].readableName())
          + ANONYMOUS_CLASS_NAME_SUFFIX;

    }

    String name = new String(binding.sourceName());

    return name;
  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITypeBinding#getPackage()
   */
  public IPackageBinding getPackage() {
    return BindingFactory.getPackageBinding(binding.getPackage());
  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITypeBinding#getDeclaringClass()
   */
  public IReferenceTypeBinding getDeclaringClass() {
    return BindingFactory.getReferenceTypeBinding(binding.enclosingType());
  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITypeBinding#getSuperclass()
   */
  public IReferenceTypeBinding getSuperclass() {
    ReferenceBinding superBinding = binding.superclass();
    return BindingFactory.getReferenceTypeBinding(superBinding);
  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITypeBinding#getInterfaces()
   */
  public IReferenceTypeBinding[] getInterfaces() {
    ReferenceBinding[] superBindings = binding.superInterfaces();
    if (superBindings == null) // java.lang.Object -> null
      return null;
    IReferenceTypeBinding[] out =
      new IReferenceTypeBinding[superBindings.length];
    for (int i = 0; i < out.length; i++)
      out[i] = BindingFactory.getReferenceTypeBinding(superBindings[i]);
    return out;
  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.IBinding#getModifiers()
   */
  public int getModifiers() {
    return ((ReferenceBinding) binding).modifiers & 0x0fff; // first byte is used jdt-internally
  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITypeBinding#getDeclaredModifiers()
   */
  public int getDeclaredModifiers() {
    return 0; //todo
  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITypeBinding#isTopLevel()
   */
  public boolean isTopLevel() {
    return ((ReferenceBinding) binding).enclosingType() == null;
  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITypeBinding#isNested()
   */
  public boolean isNested() {
    return binding.isNestedType();
  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITypeBinding#isMember()
   */
  public boolean isMember() {
    return binding.isMemberType();
  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITypeBinding#isLocal()
   */
  public boolean isLocal() {
    return binding.isLocalType();
  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITypeBinding#isAnonymous()
   */
  public boolean isAnonymous() {
    return binding.isAnonymousType();
  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITypeBinding#getDeclaredTypes()
   */
  public IReferenceTypeBinding[] getDeclaredTypes() {
    ReferenceBinding[] intTypes = binding.memberTypes();
    if (intTypes == null)
      return new IReferenceTypeBinding[0];
    IReferenceTypeBinding[] extTypes =
      new IReferenceTypeBinding[intTypes.length];
    for (int i = 0; i < extTypes.length; i++) {
      extTypes[i] = BindingFactory.getReferenceTypeBinding(intTypes[i]);
    }
    return extTypes;
  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITypeBinding#getDeclaredFields()
   */
  public IVariableBinding[] getDeclaredFields() {
    org.eclipse.jdt.internal.compiler.lookup.FieldBinding[] intFields =
      binding.fields();
    if (intFields == null)
      return new IVariableBinding[0];
    VariableBinding[] extFields = new VariableBinding[intFields.length];
    for (int i = 0; i < extFields.length; i++) {
      extFields[i] = new VariableBinding(intFields[i]);
    }
    return extFields;
  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITypeBinding#getDeclaredMethods()
   */
  public IMethodBinding[] getDeclaredMethods() {
    org.eclipse.jdt.internal.compiler.lookup.MethodBinding[] intMethods =
      binding.methods();
    if (intMethods == null)
      return new IMethodBinding[0];
    MethodBinding[] extMethods = new MethodBinding[intMethods.length];
    for (int i = 0; i < extMethods.length; i++) {
      extMethods[i] = new MethodBinding(intMethods[i]);
    }
    return extMethods;
  }

  public IMethodBinding[] getAvailableMethods() {
    org.eclipse.jdt.internal.compiler.lookup.MethodBinding[] intMethods =
      binding.availableMethods();
    if (intMethods == null)
      return new IMethodBinding[0];
    MethodBinding[] extMethods = new MethodBinding[intMethods.length];
    for (int i = 0; i < extMethods.length; i++) {
      extMethods[i] = new MethodBinding(intMethods[i]);
    }
    return extMethods;
  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITypeBinding#isFromSource()
   */
  public boolean isFromSource() {
    return false; //todo
  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.IBinding#getKind()
   */
  public int getKind() {
    return IBinding.TYPE;
  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.IBinding#isDeprecated()
   */
  public boolean isDeprecated() {
    return binding.isDeprecated();
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jlint.jom.interfaces.IBinding#equals(com.sap.tc.jtools.jlint.jom.interfaces.IBinding)
   */
  public boolean equals(Object object) {
    return (object instanceof ReferenceTypeBinding)
      && (((ReferenceTypeBinding) object).binding.equals(binding));
  }

  public int hashCode() {
    return binding.hashCode();
  }

  public String getName() {
    if (name != null)
      return name;
    if (isAnonymous())
      return name = getIdentifier();
    String packageName =
      (getPackage() == null || getPackage().getName() == null)
        ? ""
        : getPackage().getName().concat(".");
    StringBuffer classNameBuf = new StringBuffer(getIdentifier());
    IReferenceTypeBinding currentBinding = this;
    while (currentBinding.getDeclaringClass() != null) {
      classNameBuf.insert(
        0,
        currentBinding.getDeclaringClass().getIdentifier().concat("."));
      currentBinding = currentBinding.getDeclaringClass();
    }
    return name = packageName.concat(classNameBuf.toString());
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITypeBinding#isReference()
   */
  public boolean isReference() {
    return true;
  }
}
