/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_modules/_jom/_core/src/com/sap/tc/jtools/jlint/jom/ciwrappers2_1/CompilationUnit.java#2 $
 */

package com.sap.tc.jtools.jlint.jom.ciwrappers2_1;

import java.util.List;
import java.util.Vector;

import org.eclipse.jdt.internal.compiler.ast.CompilationUnitDeclaration;
import org.eclipse.jdt.internal.compiler.ast.ImportReference;
import org.eclipse.jdt.internal.compiler.lookup.CompilationUnitScope;
import org.eclipse.jdt.internal.compiler.lookup.ImportBinding;
import org.eclipse.jdt.internal.compiler.lookup.ReferenceBinding;

import com.sap.tc.jtools.jlint.jom.interfaces.ICompilationUnit;
import com.sap.tc.jtools.jlint.jom.interfaces.IPackageDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.ITypeBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.Traversable;
import com.sap.tc.jtools.jlint.jom.interfaces.IVisitor;
import java.util.Iterator;

public class CompilationUnit implements ICompilationUnit {

  public CompilationUnitDeclaration cud;
  private SourceContent source;
  private CompilationUnitScope scope;

  /* cache */
  private IPackageDeclaration pack;
  private List imports;
  private List types;

  public CompilationUnit(CompilationUnitDeclaration cud, char[] source) {
    this.cud = cud;
    this.source =
      new SourceContent(source, cud.compilationResult.lineSeparatorPositions);
    this.scope = cud.scope;
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jlint.jom.interfaces.Traversable#getNodeType()
   */
  public int getNodeType() {
    return Traversable.COMPILATION_UNIT;
  }

  public int getStartPosition() {
    return cud.sourceStart;
  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ICompilationUnit#getFileName()
   */
  public String getFileName() {
    char[] token = cud.getFileName();
    if (token == null)
      return null;
    return new String(token).replace('\\', '/');
  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ICompilationUnit#getPackage()
   */
  public IPackageDeclaration getPackage() {
    if (pack == null) {
      ImportReference ir = cud.currentPackage;
      if (ir == null || ir.tokens.length == 0)
        return null;
      pack = new PackageDeclaration(ir);
    }
    return pack;

  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ICompilationUnit#imports()
   */
  public List imports() {
    if (imports == null) {
      imports = new Vector();
      ImportBinding[] importBindings = cud.scope.imports;
      ImportReference[] importArray = cud.imports;
      if (importArray != null) {
        for (int i = 0; i < importArray.length; i++) {
          int bindingIndex = -1;
          for (int j = 0;
            j < importBindings.length && bindingIndex == -1;
            j++) {
            if (charDoubleArraysAreEqual(importArray[i].tokens,
              importBindings[j].compoundName))
              bindingIndex = j;
          }
          if (bindingIndex == -1) {
            // import of the package the cu belongs to
            imports.add(
              new ImportDeclarationWithPackageBinding(
                importArray[i],
                getPackage().resolveBinding()));
          } else {
            if (importArray[i].onDemand) {
              imports.add(
                new ImportDeclarationWithImportBinding(
                  importArray[i],
                  importBindings[bindingIndex]));
            } else {
              imports.add(
                new ImportDeclarationWithTypeBinding(
                  importArray[i],
                  (ReferenceBinding) cud.scope.getType(importArray[i].tokens)));
            }

          }

        }
      }

    }
    return imports;
  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ICompilationUnit#types()
   */
  public List types() {
    if (types == null) {
      org.eclipse.jdt.internal.compiler.ast.TypeDeclaration[] temp = cud.types;
      types = new Vector();
      if (temp != null) {
        for (int i = 0; i < temp.length; i++) {
          TypeDeclaration td = new TypeDeclaration(temp[i], source);
          types.add(td);
        }
      }

    }
    return types;

  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ICompilationUnit#lineNumber(int)
   */
  public int lineNumber(int position) {
    return source.lineNumber(position);
  }

  /**
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ICompilationUnit#columnNumber(int)
   */
  public int columnNumber(int position) {
    return source.columnNumber(position);
  }

  public void traverse(IVisitor visitor) {

    if (visitor.visit(this)) {
      IPackageDeclaration pack = getPackage();
      if (pack != null)
        pack.traverse(visitor);
      List imports = imports();
      if (imports != null) {
        Iterator iterator = imports.iterator();
        while (iterator.hasNext())
           ((Traversable) iterator.next()).traverse(visitor);
      }
      List types = types();
      if (types != null) {
        Iterator iterator = types.iterator();
        while (iterator.hasNext())
           ((Traversable) iterator.next()).traverse(visitor);
      }
    }
    visitor.endVisit(this);
  }

  public ITypeBinding getBindingForName(String fullName) {

    char[] charName = fullName.toCharArray();
    return BindingFactory.getTypeBinding(scope.getType(charName));

  }

  /**
   * Method charDoubleArraysAreEqual.
   * @param c
   * @param c1
   * @return boolean
   */
  private boolean charDoubleArraysAreEqual(char[][] c, char[][] c1) {
    if (c.length != c1.length)
      return false;
    for (int i = 0; i < c.length; i++) {
      if (c[i].length != c1[i].length)
        return false;
      for (int j = 0; j < c1[i].length; j++) {
        if (c[i][j] != c1[i][j])
          return false;
      }
    }
    return true;
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ICompilationUnit#lineCount()
   */
  public int lineCount() {
    return source.getLineCount();
  }

}
