/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_modules/_jom/_core/src/com/sap/tc/jtools/jlint/jom/JomVisitor.java#2 $
 */

package com.sap.tc.jtools.jlint.jom;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Properties;
import java.util.Set;
import java.util.Vector;

import com.sap.tc.jtools.jlint.Result;
import com.sap.tc.jtools.jlint.jom.interfaces.IAnonymousClassDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.IArrayAccess;
import com.sap.tc.jtools.jlint.jom.interfaces.IArrayCreation;
import com.sap.tc.jtools.jlint.jom.interfaces.IArrayInitializer;
import com.sap.tc.jtools.jlint.jom.interfaces.IArrayType;
import com.sap.tc.jtools.jlint.jom.interfaces.IAssertStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IAssignment;
import com.sap.tc.jtools.jlint.jom.interfaces.IBlock;
import com.sap.tc.jtools.jlint.jom.interfaces.IBooleanLiteral;
import com.sap.tc.jtools.jlint.jom.interfaces.IBreakStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.ICastExpression;
import com.sap.tc.jtools.jlint.jom.interfaces.ICatchClause;
import com.sap.tc.jtools.jlint.jom.interfaces.ICharacterLiteral;
import com.sap.tc.jtools.jlint.jom.interfaces.IClassInstanceCreation;
import com.sap.tc.jtools.jlint.jom.interfaces.ICompilationUnit;
import com.sap.tc.jtools.jlint.jom.interfaces.IConditionalExpression;
import com.sap.tc.jtools.jlint.jom.interfaces.IConstructorInvocation;
import com.sap.tc.jtools.jlint.jom.interfaces.IContinueStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IDoStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IEmptyStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IExpressionStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IFieldAccess;
import com.sap.tc.jtools.jlint.jom.interfaces.IFieldDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.IForStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IIfStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IImportDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.IInfixExpression;
import com.sap.tc.jtools.jlint.jom.interfaces.IInitializer;
import com.sap.tc.jtools.jlint.jom.interfaces.IInstanceofExpression;
import com.sap.tc.jtools.jlint.jom.interfaces.ILabeledStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IMethodDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.IMethodInvocation;
import com.sap.tc.jtools.jlint.jom.interfaces.INullLiteral;
import com.sap.tc.jtools.jlint.jom.interfaces.INumberLiteral;
import com.sap.tc.jtools.jlint.jom.interfaces.IPackageDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.IPackageName;
import com.sap.tc.jtools.jlint.jom.interfaces.IPostfixExpression;
import com.sap.tc.jtools.jlint.jom.interfaces.IPrefixExpression;
import com.sap.tc.jtools.jlint.jom.interfaces.IQualifiedName;
import com.sap.tc.jtools.jlint.jom.interfaces.IReturnStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.ISimpleName;
import com.sap.tc.jtools.jlint.jom.interfaces.ISimpleType;
import com.sap.tc.jtools.jlint.jom.interfaces.IStringLiteral;
import com.sap.tc.jtools.jlint.jom.interfaces.ISuperConstructorInvocation;
import com.sap.tc.jtools.jlint.jom.interfaces.ISuperExpression;
import com.sap.tc.jtools.jlint.jom.interfaces.ISuperMethodInvocation;
import com.sap.tc.jtools.jlint.jom.interfaces.ISwitchCase;
import com.sap.tc.jtools.jlint.jom.interfaces.ISwitchStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.ISynchronizedStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor;
import com.sap.tc.jtools.jlint.jom.interfaces.IThisExpression;
import com.sap.tc.jtools.jlint.jom.interfaces.IThrowStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.ITryStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.ITypeDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.ITypeDeclarationStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.ITypeLiteral;
import com.sap.tc.jtools.jlint.jom.interfaces.IVariableDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.IVariableDeclarationStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IVisitor;
import com.sap.tc.jtools.jlint.jom.interfaces.IWhileStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.Position;
import com.sap.tc.jtools.jlint.jom.interfaces.Traversable;
import com.sap.tc.jtools.jlint.jom.metrics.JomMetricVisitor;
import com.sap.tc.jtools.jtci.FilePosition;
import com.sap.tc.jtools.jtci.ParameterTool;
import com.sap.tc.jtools.jtci.TestObjectPosition;

/**
 * ITestVisitor used by JomProjectVisitorProcessor.
 * <p>
 * It allows the execution of several tests "in parallel" (i.e. in one sweep
 * through the syntax tree).
 * 
 */
public class JomVisitor implements IVisitor {

  // HashMap containing ITestVisitor -> Traversable
  // if traversable is not null, the traversal of 
  // the <traversal> subtree will be skipped
  protected HashMap tests = new HashMap();

  protected Vector errorList = new Vector();
  protected ICompilationUnit currentCompilationUnit;

  public void addVisitor(ITestVisitor test) {
    tests.put(test, null);
  }

  public Vector getErrors() {
    return errorList;
  }

  public boolean visit(Traversable node) {
    boolean goOn = false;
    Set keys = tests.keySet();
    for (Iterator iter = keys.iterator(); iter.hasNext();) {
      ITestVisitor visitor = (ITestVisitor) iter.next();
	  Traversable skippedNode = (Traversable) tests.get(visitor);
	  if (skippedNode != null) {
		continue;
	  }
      // check if test disabled by pseudocomment
      if (visitor.isDisabledFor(node)) {
        tests.put(visitor, node);
        continue;
      }
//      Traversable skippedNode = (Traversable) tests.get(visitor);
//      if (skippedNode != null) {
//        continue;
//      }
      try {
        boolean returnValue = false;
        switch (node.getNodeType()) {
          case Traversable.ANONYMOUS_CLASS_DECLARATION :
            returnValue = visitor.visit((IAnonymousClassDeclaration) node);
            break;
          case Traversable.ARRAY_ACCESS :
            returnValue = visitor.visit((IArrayAccess) node);
            break;
          case Traversable.ARRAY_CREATION :
            returnValue = visitor.visit((IArrayCreation) node);
            break;
          case Traversable.ARRAY_INITIALIZER :
            returnValue = visitor.visit((IArrayInitializer) node);
            break;
          case Traversable.ARRAY_TYPE :
            returnValue = visitor.visit((IArrayType) node);
            break;
          case Traversable.ASSERT_STATEMENT :
            returnValue = visitor.visit((IAssertStatement) node);
            break;
          case Traversable.ASSIGNMENT :
            returnValue = visitor.visit((IAssignment) node);
            break;
          case Traversable.BLOCK :
            returnValue = visitor.visit((IBlock) node);
            break;
          case Traversable.BOOLEAN_LITERAL :
            returnValue = visitor.visit((IBooleanLiteral) node);
            break;
          case Traversable.BREAK_STATEMENT :
            returnValue = visitor.visit((IBreakStatement) node);
            break;
          case Traversable.CAST_EXPRESSION :
            returnValue = visitor.visit((ICastExpression) node);
            break;
          case Traversable.CATCH_CLAUSE :
            returnValue = visitor.visit((ICatchClause) node);
            break;
          case Traversable.CHARACTER_LITERAL :
            returnValue = visitor.visit((ICharacterLiteral) node);
            break;
          case Traversable.CLASS_INSTANCE_CREATION :
            returnValue = visitor.visit((IClassInstanceCreation) node);
            break;
          case Traversable.COMPILATION_UNIT :
            currentCompilationUnit = (ICompilationUnit) node;
            ((JomTestVisitor) visitor).setCurrentTree((ICompilationUnit) node);
            returnValue = visitor.visit((ICompilationUnit) node);
            break;
          case Traversable.CONDITIONAL_EXPRESSION :
            returnValue = visitor.visit((IConditionalExpression) node);
            break;
          case Traversable.CONSTRUCTOR_INVOCATION :
            returnValue = visitor.visit((IConstructorInvocation) node);
            break;
          case Traversable.CONTINUE_STATEMENT :
            returnValue = visitor.visit((IContinueStatement) node);
            break;
          case Traversable.DO_STATEMENT :
            returnValue = visitor.visit((IDoStatement) node);
            break;
          case Traversable.EMPTY_STATEMENT :
            returnValue = visitor.visit((IEmptyStatement) node);
            break;
          case Traversable.EXPRESSION_STATEMENT :
            returnValue = visitor.visit((IExpressionStatement) node);
            break;
          case Traversable.FIELD_ACCESS :
            returnValue = visitor.visit((IFieldAccess) node);
            break;
          case Traversable.FIELD_DECLARATION :
            returnValue = visitor.visit((IFieldDeclaration) node);
            break;
          case Traversable.FOR_STATEMENT :
            returnValue = visitor.visit((IForStatement) node);
            break;
          case Traversable.IF_STATEMENT :
            returnValue = visitor.visit((IIfStatement) node);
            break;
          case Traversable.IMPORT_DECLARATION :
            returnValue = visitor.visit((IImportDeclaration) node);
            break;
          case Traversable.INFIX_EXPRESSION :
            returnValue = visitor.visit((IInfixExpression) node);
            break;
          case Traversable.INSTANCEOF_EXPRESSION :
            returnValue = visitor.visit((IInstanceofExpression) node);
            break;
          case Traversable.INITIALIZER :
            returnValue = visitor.visit((IInitializer) node);
            break;
          case Traversable.LABELED_STATEMENT :
            returnValue = visitor.visit((ILabeledStatement) node);
            break;
          case Traversable.METHOD_DECLARATION :
            returnValue = visitor.visit((IMethodDeclaration) node);
            break;
          case Traversable.METHOD_INVOCATION :
            returnValue = visitor.visit((IMethodInvocation) node);
            break;
          case Traversable.NULL_LITERAL :
            returnValue = visitor.visit((INullLiteral) node);
            break;
          case Traversable.NUMBER_LITERAL :
            returnValue = visitor.visit((INumberLiteral) node);
            break;
          case Traversable.PACKAGE_DECLARATION :
            returnValue = visitor.visit((IPackageDeclaration) node);
            break;
          case Traversable.POSTFIX_EXPRESSION :
            returnValue = visitor.visit((IPostfixExpression) node);
            break;
          case Traversable.PREFIX_EXPRESSION :
            returnValue = visitor.visit((IPrefixExpression) node);
            break;
          case Traversable.QUALIFIED_NAME :
            returnValue = visitor.visit((IQualifiedName) node);
            break;
          case Traversable.RETURN_STATEMENT :
            returnValue = visitor.visit((IReturnStatement) node);
            break;
          case Traversable.SIMPLE_NAME :
            returnValue = visitor.visit((ISimpleName) node);
            break;
          case Traversable.SIMPLE_TYPE :
            returnValue = visitor.visit((ISimpleType) node);
            break;
          case Traversable.STRING_LITERAL :
            returnValue = visitor.visit((IStringLiteral) node);
            break;
          case Traversable.SUPER_CONSTRUCTOR_INVOCATION :
            returnValue = visitor.visit((ISuperConstructorInvocation) node);
            break;
          case Traversable.SUPER_EXPRESSION :
            returnValue = visitor.visit((ISuperExpression) node);
            break;
          case Traversable.SUPER_METHOD_INVOCATION :
            returnValue = visitor.visit((ISuperMethodInvocation) node);
            break;
          case Traversable.SWITCH_CASE :
            returnValue = visitor.visit((ISwitchCase) node);
            break;
          case Traversable.SWITCH_STATEMENT :
            returnValue = visitor.visit((ISwitchStatement) node);
            break;
          case Traversable.SYNCHRONIZED_STATEMENT :
            returnValue = visitor.visit((ISynchronizedStatement) node);
            break;
          case Traversable.THIS_EXPRESSION :
            returnValue = visitor.visit((IThisExpression) node);
            break;
          case Traversable.THROW_STATEMENT :
            returnValue = visitor.visit((IThrowStatement) node);
            break;
          case Traversable.TRY_STATEMENT :
            returnValue = visitor.visit((ITryStatement) node);
            break;
          case Traversable.TYPE_DECLARATION :
            returnValue = visitor.visit((ITypeDeclaration) node);
            break;
          case Traversable.TYPE_DECLARATION_STATEMENT :
            returnValue = visitor.visit((ITypeDeclarationStatement) node);
            break;
          case Traversable.TYPE_LITERAL :
            returnValue = visitor.visit((ITypeLiteral) node);
            break;
          case Traversable.VARIABLE_DECLARATION :
            returnValue = visitor.visit((IVariableDeclaration) node);
            break;
          case Traversable.VARIABLE_DECLARATION_STATEMENT :
            returnValue = visitor.visit((IVariableDeclarationStatement) node);
            break;
          case Traversable.WHILE_STATEMENT :
            returnValue = visitor.visit((IWhileStatement) node);
            break;
          case Traversable.PACKAGE_NAME :
            returnValue = visitor.visit((IPackageName) node);
            break;
          default :
            throw new IllegalArgumentException(
              node.getClass().getName() + ": unknown node type");
        }
        tests.put(visitor, returnValue ? null : node);
        goOn |= returnValue;
      } catch (Exception e) {
        addError((JomTestVisitor) visitor, e, node);
        tests.put(visitor, node);
      }
    }
    return goOn;
  }

  public void endVisit(Traversable node) {
    Set keys = tests.keySet();
    for (Iterator iter = keys.iterator(); iter.hasNext();) {
      ITestVisitor visitor = (ITestVisitor) iter.next();
      Traversable skippedNode = (Traversable) tests.get(visitor);
      if (skippedNode != null && skippedNode != node)
        continue;
      if (node == skippedNode && visitor.isDisabledFor(node)) {
        // reset skip info
        tests.put(visitor, null);
        continue;
      }
      try {
        switch (node.getNodeType()) {
          case Traversable.ANONYMOUS_CLASS_DECLARATION :
            visitor.endVisit((IAnonymousClassDeclaration) node);
            break;
          case Traversable.ARRAY_ACCESS :
            visitor.endVisit((IArrayAccess) node);
            break;
          case Traversable.ARRAY_CREATION :
            visitor.endVisit((IArrayCreation) node);
            break;
          case Traversable.ARRAY_INITIALIZER :
            visitor.endVisit((IArrayInitializer) node);
            break;
          case Traversable.ARRAY_TYPE :
            visitor.endVisit((IArrayType) node);
            break;
          case Traversable.ASSERT_STATEMENT :
            visitor.endVisit((IAssertStatement) node);
            break;
          case Traversable.ASSIGNMENT :
            visitor.endVisit((IAssignment) node);
            break;
          case Traversable.BLOCK :
            visitor.endVisit((IBlock) node);
            break;
          case Traversable.BOOLEAN_LITERAL :
            visitor.endVisit((IBooleanLiteral) node);
            break;
          case Traversable.BREAK_STATEMENT :
            visitor.endVisit((IBreakStatement) node);
            break;
          case Traversable.CAST_EXPRESSION :
            visitor.endVisit((ICastExpression) node);
            break;
          case Traversable.CATCH_CLAUSE :
            visitor.endVisit((ICatchClause) node);
            break;
          case Traversable.CHARACTER_LITERAL :
            visitor.endVisit((ICharacterLiteral) node);
            break;
          case Traversable.CLASS_INSTANCE_CREATION :
            visitor.endVisit((IClassInstanceCreation) node);
            break;
          case Traversable.COMPILATION_UNIT :
            visitor.endVisit((ICompilationUnit) node);
            // collect statistics per compilation unit for metric tests
            if (visitor instanceof JomMetricVisitor) {
              ((JomMetricVisitor)visitor).postVisit((ICompilationUnit)node);
            }
            break;
          case Traversable.CONDITIONAL_EXPRESSION :
            visitor.endVisit((IConditionalExpression) node);
            break;
          case Traversable.CONSTRUCTOR_INVOCATION :
            visitor.endVisit((IConstructorInvocation) node);
            break;
          case Traversable.CONTINUE_STATEMENT :
            visitor.endVisit((IContinueStatement) node);
            break;
          case Traversable.DO_STATEMENT :
            visitor.endVisit((IDoStatement) node);
            break;
          case Traversable.EMPTY_STATEMENT :
            visitor.endVisit((IEmptyStatement) node);
            break;
          case Traversable.EXPRESSION_STATEMENT :
            visitor.endVisit((IExpressionStatement) node);
            break;
          case Traversable.FIELD_ACCESS :
            visitor.endVisit((IFieldAccess) node);
            break;
          case Traversable.FIELD_DECLARATION :
            visitor.endVisit((IFieldDeclaration) node);
            break;
          case Traversable.FOR_STATEMENT :
            visitor.endVisit((IForStatement) node);
            break;
          case Traversable.IF_STATEMENT :
            visitor.endVisit((IIfStatement) node);
            break;
          case Traversable.IMPORT_DECLARATION :
            visitor.endVisit((IImportDeclaration) node);
            break;
          case Traversable.INFIX_EXPRESSION :
            visitor.endVisit((IInfixExpression) node);
            break;
          case Traversable.INSTANCEOF_EXPRESSION :
            visitor.endVisit((IInstanceofExpression) node);
            break;
          case Traversable.INITIALIZER :
            visitor.endVisit((IInitializer) node);
            break;
          case Traversable.LABELED_STATEMENT :
            visitor.endVisit((ILabeledStatement) node);
            break;
          case Traversable.METHOD_DECLARATION :
            visitor.endVisit((IMethodDeclaration) node);
            break;
          case Traversable.METHOD_INVOCATION :
            visitor.endVisit((IMethodInvocation) node);
            break;
          case Traversable.NULL_LITERAL :
            visitor.endVisit((INullLiteral) node);
            break;
          case Traversable.NUMBER_LITERAL :
            visitor.endVisit((INumberLiteral) node);
            break;
          case Traversable.PACKAGE_DECLARATION :
            visitor.endVisit((IPackageDeclaration) node);
            break;
          case Traversable.POSTFIX_EXPRESSION :
            visitor.endVisit((IPostfixExpression) node);
            break;
          case Traversable.PREFIX_EXPRESSION :
            visitor.endVisit((IPrefixExpression) node);
            break;
          case Traversable.QUALIFIED_NAME :
            visitor.endVisit((IQualifiedName) node);
            break;
          case Traversable.RETURN_STATEMENT :
            visitor.endVisit((IReturnStatement) node);
            break;
          case Traversable.SIMPLE_NAME :
            visitor.endVisit((ISimpleName) node);
            break;
          case Traversable.SIMPLE_TYPE :
            visitor.endVisit((ISimpleType) node);
            break;
          case Traversable.STRING_LITERAL :
            visitor.endVisit((IStringLiteral) node);
            break;
          case Traversable.SUPER_CONSTRUCTOR_INVOCATION :
            visitor.endVisit((ISuperConstructorInvocation) node);
            break;
          case Traversable.SUPER_EXPRESSION :
            visitor.endVisit((ISuperExpression) node);
            break;
          case Traversable.SUPER_METHOD_INVOCATION :
            visitor.endVisit((ISuperMethodInvocation) node);
            break;
          case Traversable.SWITCH_CASE :
            visitor.endVisit((ISwitchCase) node);
            break;
          case Traversable.SWITCH_STATEMENT :
            visitor.endVisit((ISwitchStatement) node);
            break;
          case Traversable.SYNCHRONIZED_STATEMENT :
            visitor.endVisit((ISynchronizedStatement) node);
            break;
          case Traversable.THIS_EXPRESSION :
            visitor.endVisit((IThisExpression) node);
            break;
          case Traversable.THROW_STATEMENT :
            visitor.endVisit((IThrowStatement) node);
            break;
          case Traversable.TRY_STATEMENT :
            visitor.endVisit((ITryStatement) node);
            break;
          case Traversable.TYPE_DECLARATION :
            visitor.endVisit((ITypeDeclaration) node);
            break;
          case Traversable.TYPE_DECLARATION_STATEMENT :
            visitor.endVisit((ITypeDeclarationStatement) node);
            break;
          case Traversable.TYPE_LITERAL :
            visitor.endVisit((ITypeLiteral) node);
            break;
          case Traversable.VARIABLE_DECLARATION :
            visitor.endVisit((IVariableDeclaration) node);
            break;
          case Traversable.VARIABLE_DECLARATION_STATEMENT :
            visitor.endVisit((IVariableDeclarationStatement) node);
            break;
          case Traversable.WHILE_STATEMENT :
            visitor.endVisit((IWhileStatement) node);
            break;
          case Traversable.PACKAGE_NAME :
            visitor.endVisit((IPackageName) node);
            break;
          default :
            throw new IllegalArgumentException(
              node.getClass().getName() + ": unknown node type");
        }

      } catch (Exception e) {
        addError((JomTestVisitor) visitor, e, node);
      } finally {
		if (skippedNode == node) {
		  // reset skip info
		  tests.put(visitor, null);
		}
      }
    }
  }

  public void addError(JomTestVisitor test, Exception e, Position position) {
    StringWriter sw = new StringWriter();
    PrintWriter pw = new PrintWriter(sw);
    e.printStackTrace(pw);
    pw.flush();
    Properties parameters = new Properties();
    parameters.setProperty(ParameterTool.PAR_EXCEPTION, sw.getBuffer().toString());
    parameters.setProperty(ParameterTool.PAR_MSG_KEY, Result.ERROR_MESSAGE_COULDNT_PERFORM);
    String relativeFileName = new String(currentCompilationUnit.getFileName());
    if (test.testObject.getBaseDir() != null)
      relativeFileName =
        relativeFileName.substring(test.testObject.getBaseDir().length() + 1);
    int offset = position.getStartPosition();
    TestObjectPosition top =
      new FilePosition(
        relativeFileName,
        currentCompilationUnit.lineNumber(offset),
        currentCompilationUnit.columnNumber(offset));
    errorList.add(
      new Result(
        test.getName(),
        test.testObject.getType(),
        test.testObject.getID(),
        top,
        Result.SEVERITY_INTERNAL_ERROR,
        Result.ERROR_MESSAGE_COULDNT_PERFORM,
        parameters)); 
  }
}
