/*
 * Copyright (c) 2004 by SAP AG, Walldorf.,
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_modules/_jom/_core/src/com/sap/tc/jtools/jlint/jom/JomTestVisitor.java#3 $
 */

package com.sap.tc.jtools.jlint.jom;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Vector;

import com.sap.tc.jtools.jlint.ProcessorInterface;
import com.sap.tc.jtools.jlint.Requirement;
import com.sap.tc.jtools.jlint.RequirementInterface;
import com.sap.tc.jtools.jlint.Result;
import com.sap.tc.jtools.jlint.ResultWrapper;
import com.sap.tc.jtools.jlint.TestComponentInterface;
import com.sap.tc.jtools.jlint.java.JavaFileArrayTestObject;
import com.sap.tc.jtools.jlint.jdt.ClasspathLocation;
import com.sap.tc.jtools.jlint.jom.interfaces.Commentable;
import com.sap.tc.jtools.jlint.jom.interfaces.IAnonymousClassDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.IArrayAccess;
import com.sap.tc.jtools.jlint.jom.interfaces.IArrayCreation;
import com.sap.tc.jtools.jlint.jom.interfaces.IArrayInitializer;
import com.sap.tc.jtools.jlint.jom.interfaces.IArrayType;
import com.sap.tc.jtools.jlint.jom.interfaces.IAssertStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IAssignment;
import com.sap.tc.jtools.jlint.jom.interfaces.IBlock;
import com.sap.tc.jtools.jlint.jom.interfaces.IBooleanLiteral;
import com.sap.tc.jtools.jlint.jom.interfaces.IBreakStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.ICastExpression;
import com.sap.tc.jtools.jlint.jom.interfaces.ICatchClause;
import com.sap.tc.jtools.jlint.jom.interfaces.ICharacterLiteral;
import com.sap.tc.jtools.jlint.jom.interfaces.IClassInstanceCreation;
import com.sap.tc.jtools.jlint.jom.interfaces.ICompilationUnit;
import com.sap.tc.jtools.jlint.jom.interfaces.IConditionalExpression;
import com.sap.tc.jtools.jlint.jom.interfaces.IConstructorInvocation;
import com.sap.tc.jtools.jlint.jom.interfaces.IContinueStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IDoStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IEmptyStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IExpressionStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IFieldAccess;
import com.sap.tc.jtools.jlint.jom.interfaces.IFieldDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.IForStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IIfStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IImportDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.IInfixExpression;
import com.sap.tc.jtools.jlint.jom.interfaces.IInitializer;
import com.sap.tc.jtools.jlint.jom.interfaces.IInstanceofExpression;
import com.sap.tc.jtools.jlint.jom.interfaces.ILabeledStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IMethodDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.IMethodInvocation;
import com.sap.tc.jtools.jlint.jom.interfaces.INullLiteral;
import com.sap.tc.jtools.jlint.jom.interfaces.INumberLiteral;
import com.sap.tc.jtools.jlint.jom.interfaces.IPackageDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.IPackageName;
import com.sap.tc.jtools.jlint.jom.interfaces.IPostfixExpression;
import com.sap.tc.jtools.jlint.jom.interfaces.IPrefixExpression;
import com.sap.tc.jtools.jlint.jom.interfaces.IQualifiedName;
import com.sap.tc.jtools.jlint.jom.interfaces.IReturnStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.ISimpleName;
import com.sap.tc.jtools.jlint.jom.interfaces.ISimpleType;
import com.sap.tc.jtools.jlint.jom.interfaces.IStringLiteral;
import com.sap.tc.jtools.jlint.jom.interfaces.ISuperConstructorInvocation;
import com.sap.tc.jtools.jlint.jom.interfaces.ISuperExpression;
import com.sap.tc.jtools.jlint.jom.interfaces.ISuperMethodInvocation;
import com.sap.tc.jtools.jlint.jom.interfaces.ISwitchCase;
import com.sap.tc.jtools.jlint.jom.interfaces.ISwitchStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.ISynchronizedStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor;
import com.sap.tc.jtools.jlint.jom.interfaces.IThisExpression;
import com.sap.tc.jtools.jlint.jom.interfaces.IThrowStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.ITryStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.ITypeDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.ITypeDeclarationStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.ITypeLiteral;
import com.sap.tc.jtools.jlint.jom.interfaces.IVariableDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.IVariableDeclarationStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.IWhileStatement;
import com.sap.tc.jtools.jlint.jom.interfaces.Position;
import com.sap.tc.jtools.jlint.jom.interfaces.Traversable;
import com.sap.tc.jtools.jlint.jom.util.NameTool;
import com.sap.tc.jtools.jtci.FilePosition;
import com.sap.tc.jtools.jtci.ParameterTool;
import com.sap.tc.jtools.jtci.TestObjectPosition;
import com.sap.tc.jtools.jtci.interfaces.ITestMessageDescription;
import com.sap.tc.jtools.jtci.interfaces.ParameterInterface;
import com.sap.tc.jtools.jtci.interfaces.ResultInterface;

/**
 * Base class for tests on Java sources.
 * 
 * It provides a default visitor implementation and methods to add errors and manage
 * test parameters.
 * 
 * Each test class must do the following:
 * 
 * <ul>
 * <li>implement getTestName</li>
 * <li>declare a constructor with no parameters</li>
 *</ul>
 * 
 * Additionally,  test classes can do the following:
 * 
 * <ul>
 * <li>overwrite the setParameters method</li>
 * <li>overwrite the processErrors method</li>
 * <li>and, of course, overwrite all the visit and endVisit methods that are necessary to implement the test</li> 
 *</ul>
 * 
 * The visit methods are called before traversing the corresponding node's subtree, the endVisit methods
 * after. For details on the traversing algorithm, see the documentation to the interfaces in package 
 * com.sap.tc.jtools.jlint.jom.interfaces.
 * 
 */

abstract public class JomTestVisitor
  implements ITestVisitor, TestComponentInterface {

  public static final String PARAMETER_JAVA_LIBRARIES = "Java libraries"; //$NON-NLS-1$
  public static final String PARAMETER_SOURCE_FOLDERS = "Source folders"; //$NON-NLS-1$
  public static final String PARAMETER_CLASS_FOLDERS = "Class folders"; //$NON-NLS-1$
  public static final String PARAMETER_PSEUDO_COMMENT_SUFFIX = "PSEUDO_COMMENT_SUFFIX"; //$NON-NLS-1$
  private static final String PARAMETER_PACKAGE = "__PACKAGE";

  public static final String PSEUDO_COMMENT_PREFIX = "$JL-"; //$NON-NLS-1$
  public static final String PSEUDO_COMMENT_SUFFIX = "$"; //$NON-NLS-1$

  private static final String EXCEPTION_TAG = "EXCEPTION";
  private static final String MSG_EXCEPTION_OCCURED = "exception occured";

  protected List errorList = new ArrayList();
  protected String[] dataProviderNames = new String[0];
  protected String[] prereqNames;
  protected ResultWrapper resultWrapper = null;

  public boolean active = true;

  protected Object[] prereqValues = new Object[0];
  protected JavaFileArrayTestObject testObject;
  //  protected String[] filesToBeTested;
  protected ParameterInterface[] parameters;
  protected ProcessorInterface processor;
  protected ClasspathLocation[] searchPaths;
  protected ICompilationUnit currentCompilationUnit = null;

  private ITestMessageDescription[] messages;
  private Map msgKey2prioMap = new HashMap();
  private String pseudoComment;
  protected ParameterInterface[] internalParameters;


  /**
   * Returns the test name (which must coincide with the LABEL-parameter of the test
   * specification in the tool descriptor file)
   * 
   * This method must be implemented by the test class
   * 
   * @return test name
   *   
   */
  abstract public String getTestName();

  /**
   * Sets the input parameters. 
   * 
   * This method is called by the framework before traversal start.
   * It can be overwritten by the test class, if parameter-dependent test initialization 
   * is necessary.
   * 
   * @param parameters array of parameters (names, types, and values)
   * @param testObject the test object being processed
   *   
   */
  public void setParameters(
    ParameterInterface[] parameters,
    com.sap.tc.jtools.jtci.TestObject testObject) {
    if (!testObject.getType().equals(JavaFileArrayTestObject.JAVA_FILE_LIST))
      throw new IllegalArgumentException(
        "wrong testObject type: " + testObject.getType());
    this.testObject = (JavaFileArrayTestObject) testObject;
    //    filesToBeTested = this.testObject.getFileNames();

    Vector classpathsVector = new Vector();

    String[] classFolders = new String[0];
    String[] sourceFolders = new String[0];

    for (int i = 0; i < parameters.length; i++) {
      if (parameters[i].getName().equals(PARAMETER_JAVA_LIBRARIES)) {
        String[] libraries = (String[]) parameters[i].getValue();
        for (int j = 0; j < libraries.length; j++)
          classpathsVector.add(ClasspathLocation.forLibrary(libraries[j]));
      }
      if (parameters[i].getName().equals(PARAMETER_SOURCE_FOLDERS)) {
        sourceFolders = (String[]) parameters[i].getValue();
      }
      if (parameters[i].getName().equals(PARAMETER_CLASS_FOLDERS)) {
        classFolders = (String[]) parameters[i].getValue();
        for (int j = 0; j < classFolders.length; j++)
          classpathsVector.add(
            ClasspathLocation.forBinaryFolder(classFolders[j]));
      }
    }
    for (int j = 0; j < sourceFolders.length; j++)
      classpathsVector.add(
        ClasspathLocation.forSourceFolder(sourceFolders[j], classFolders[0]));

    searchPaths =
      (ClasspathLocation[]) classpathsVector.toArray(
        new ClasspathLocation[classpathsVector.size()]);
    this.parameters = parameters;
  }

  /* Default implemenation of the visitor methods */

  public boolean visit(IAnonymousClassDeclaration node) {
    return true;
  }
  public boolean visit(IArrayAccess node) {
    return true;
  }
  public boolean visit(IArrayCreation node) {
    return true;
  }
  public boolean visit(IArrayInitializer node) {
    return true;
  }
  public boolean visit(IArrayType node) {
    return true;
  }
  public boolean visit(IAssertStatement node) {
    return true;
  }
  public boolean visit(IAssignment node) {
    return true;
  }
  public boolean visit(IBlock node) {
    return true;
  }
  public boolean visit(IBooleanLiteral node) {
    return true;
  }
  public boolean visit(IBreakStatement node) {
    return true;
  }
  public boolean visit(ICastExpression node) {
    return true;
  }
  public boolean visit(ICatchClause node) {
    return true;
  }
  public boolean visit(ICharacterLiteral node) {
    return true;
  }
  public boolean visit(IClassInstanceCreation node) {
    return true;
  }
  public boolean visit(ICompilationUnit node) {
    return true;
  }
  public boolean visit(IConditionalExpression node) {
    return true;
  }
  public boolean visit(IConstructorInvocation node) {
    return true;
  }
  public boolean visit(IContinueStatement node) {
    return true;
  }
  public boolean visit(IDoStatement node) {
    return true;
  }
  public boolean visit(IEmptyStatement node) {
    return true;
  }
  public boolean visit(IExpressionStatement node) {
    return true;
  }
  public boolean visit(IFieldAccess node) {
    return true;
  }
  public boolean visit(IFieldDeclaration node) {
    return true;
  }
  public boolean visit(IForStatement node) {
    return true;
  }
  public boolean visit(IIfStatement node) {
    return true;
  }
  public boolean visit(IImportDeclaration node) {
    return true;
  }
  public boolean visit(IInfixExpression node) {
    return true;
  }
  public boolean visit(IInstanceofExpression node) {
    return true;
  }
  public boolean visit(IInitializer node) {
    return true;
  }
  public boolean visit(ILabeledStatement node) {
    return true;
  }
  public boolean visit(IMethodDeclaration node) {
    return true;
  }
  public boolean visit(IMethodInvocation node) {
    return true;
  }
  public boolean visit(INullLiteral node) {
    return true;
  }
  public boolean visit(INumberLiteral node) {
    return true;
  }
  public boolean visit(IPackageDeclaration node) {
    return true;
  }
  public boolean visit(IPostfixExpression node) {
    return true;
  }
  public boolean visit(IPrefixExpression node) {
    return true;
  }
  public boolean visit(IQualifiedName node) {
    return true;
  }
  public boolean visit(IReturnStatement node) {
    return true;
  }
  public boolean visit(ISimpleName node) {
    return true;
  }
  public boolean visit(ISimpleType node) {
    return true;
  }
  public boolean visit(IStringLiteral node) {
    return true;
  }
  public boolean visit(ISuperConstructorInvocation node) {
    return true;
  }
  public boolean visit(ISuperExpression node) {
    return true;
  }
  public boolean visit(ISuperMethodInvocation node) {
    return true;
  }
  public boolean visit(ISwitchCase node) {
    return true;
  }
  public boolean visit(ISwitchStatement node) {
    return true;
  }
  public boolean visit(ISynchronizedStatement node) {
    return true;
  }
  public boolean visit(IThisExpression node) {
    return true;
  }
  public boolean visit(IThrowStatement node) {
    return true;
  }
  public boolean visit(ITryStatement node) {
    return true;
  }
  public boolean visit(ITypeDeclaration node) {
    return true;
  }
  public boolean visit(ITypeDeclarationStatement node) {
    return true;
  }
  public boolean visit(ITypeLiteral node) {
    return true;
  }
  public boolean visit(IVariableDeclaration node) {
    return true;
  }
  //	public boolean visit(IVariableDeclarationExpression node) {
  //		return true;
  //	}
  public boolean visit(IVariableDeclarationStatement node) {
    return true;
  }
  //	public boolean visit(IVariableDeclarationFragment node) {
  //		return true;
  //	}
  public boolean visit(IWhileStatement node) {
    return true;
  }

  public void endVisit(IAnonymousClassDeclaration node) {
  }
  public void endVisit(IArrayAccess node) {
  }
  public void endVisit(IArrayCreation node) {
  }
  public void endVisit(IArrayInitializer node) {
  }
  public void endVisit(IArrayType node) {
  }
  public void endVisit(IAssertStatement node) {
  }
  public void endVisit(IAssignment node) {
  }
  public void endVisit(IBlock node) {
  }
  public void endVisit(IBooleanLiteral node) {
  }
  public void endVisit(IBreakStatement node) {
  }
  public void endVisit(ICastExpression node) {
  }
  public void endVisit(ICatchClause node) {
  }
  public void endVisit(ICharacterLiteral node) {
  }
  public void endVisit(IClassInstanceCreation node) {
  }
  public void endVisit(ICompilationUnit node) {
  }
  public void endVisit(IConditionalExpression node) {
  }
  public void endVisit(IConstructorInvocation node) {
  }
  public void endVisit(IContinueStatement node) {
  }
  public void endVisit(IDoStatement node) {
  }
  public void endVisit(IEmptyStatement node) {
  }
  public void endVisit(IExpressionStatement node) {
  }
  public void endVisit(IFieldAccess node) {
  }
  public void endVisit(IFieldDeclaration node) {
  }
  public void endVisit(IForStatement node) {
  }
  public void endVisit(IIfStatement node) {
  }
  public void endVisit(IImportDeclaration node) {
  }
  public void endVisit(IInfixExpression node) {
  }
  public void endVisit(IInstanceofExpression node) {
  }
  public void endVisit(IInitializer node) {
  }
  public void endVisit(ILabeledStatement node) {
  }
  public void endVisit(IMethodDeclaration node) {
  }
  public void endVisit(IMethodInvocation node) {
  }
  public void endVisit(INullLiteral node) {
  }
  public void endVisit(INumberLiteral node) {
  }
  public void endVisit(IPackageDeclaration node) {
  }
  public void endVisit(IPostfixExpression node) {
  }
  public void endVisit(IPrefixExpression node) {
  }
  public void endVisit(IQualifiedName node) {
  }
  public void endVisit(IReturnStatement node) {
  }
  public void endVisit(ISimpleName node) {
  }
  public void endVisit(ISimpleType node) {
  }
  public void endVisit(IStringLiteral node) {
  }
  public void endVisit(ISuperConstructorInvocation node) {
  }
  public void endVisit(ISuperExpression node) {
  }
  public void endVisit(ISuperMethodInvocation node) {
  }
  public void endVisit(ISwitchCase node) {
  }
  public void endVisit(ISwitchStatement node) {
  }
  public void endVisit(ISynchronizedStatement node) {
  }
  public void endVisit(IThisExpression node) {
  }
  public void endVisit(IThrowStatement node) {
  }
  public void endVisit(ITryStatement node) {
  }
  public void endVisit(ITypeDeclaration node) {
  }
  public void endVisit(ITypeDeclarationStatement node) {
  }
  public void endVisit(ITypeLiteral node) {
  }
  public void endVisit(IVariableDeclaration node) {
  }
  //	public void endVisit(IVariableDeclarationExpression node) {
  //	}
  public void endVisit(IVariableDeclarationStatement node) {
  }
  //	public void endVisit(IVariableDeclarationFragment node) {
  //	}
  public void endVisit(IWhileStatement node) {
  }

  /**
   * Default implementation.
   * This method can be overwritten to provide a test-specific processing of the 
   * 'raw' error messages, if necessary (e.g. to extract global properties, 
   * average values, etc. out of a set of errors). 
   * 
   * @param rw raw messages
   * @param aggregateOnly whether to retun only aggregates (sum, average,...)
   * @return processed messages. A <code>null</code> value means that no processing occurred
   *   
   */
  public ResultInterface[] processErrors(
    ResultInterface[] rw) {
    return null;
  }

  /* final methods */

  /**
   * Sets the currently processed compilation unit. Automatically invoked as appropriate.
   * 
   * @param currentUnit the current compilation unit
   *   
   */
  public final void setCurrentTree(ICompilationUnit currentUnit) {
    this.currentCompilationUnit = currentUnit;
  }

  /**
   * Returns the array of input parameters.
   * 
   * @param input parameters
   *   
   */
  final public ParameterInterface[] getParameters() {
    return parameters;
  }

  /**
   * Returns the value of the parameter with a given name.
   * 
   * The method returns <code>null</code> if the parameter
   * does not exist.
   * 
   * The return value can be safely cast to the appropriate type (String, String[], etc.)
   * 
   * @return parameter value 
   *   
   */
  public final java.io.Serializable getInputParameter(String parameterName) {
    for (int i = 0; i < parameters.length; i++) {
      if (parameters[i].getName().equals(parameterName))
        return parameters[i].getValue();
    }
    return null;
  }

  /**
   * Returns the value of the parameter with a given name.
   * 
   * The method returns <code>null</code> if the parameter
   * does not exist.
   * 
   * 
   * @return parameter parameter 
   *   
   */
  protected ParameterInterface getParameter(String name) {
    for (int i = 0; i < parameters.length; i++) {
      if (parameters[i].getName().equals(name))
        return parameters[i];
    }
    return null;
  }

  /**
   * Adds an entry to the message list.
   * 
   * @param message message text
   * @param messageKey message key for I18n defined in resource bundle 
   * @param priority severity level (s. com.sap.tc.jtools.jtci.interfaces.ResultInterface)
   *   
   */
  protected void addError(String messageKey, Properties messageParameters) {
    String relativeFileName = new String(currentCompilationUnit.getFileName());
    if (testObject.getBaseDir() != null)
      relativeFileName =
        relativeFileName.substring(testObject.getBaseDir().length() + 1);
    addError(
      new FilePosition(relativeFileName),
      messageKey,
      null,
      messageParameters,
      getPrio(messageKey));

  }

  /**
   * Adds an entry to the error list.
   * 
   * @param messageKey message key as defined in the resource bundle 
   *                   (properties file) for the test
   * @param messageParameters key-value String pairs. Keys can be 
   *                          in the message text surrounded by curly
   *                          brackets. E.g. if you have a parameter
   *                          "FILE" -> "MyFile.txt", the message
   *                          in the resource bundle "error in {FILE}"
   *                          will be expanded to "error in MyFile.txt"
   *                          after localization of the message.
   * @param priority severity level (s. com.sap.tc.jtools.jtci.interfaces.ResultInterface)
   * @param position position (offset with respect to begin of file)
   */
  protected void addError(
    String messageKey,
    Properties messageParameters,
    Position position) {
    addError(
      messageKey,
      null,
      messageParameters,
      getPrio(messageKey),
      position);
  }

  protected void addError(
    String messageKey,
    String link,
    Properties messageParameters,
    Position position) {
    addError(
      messageKey,
      link,
      messageParameters,
      getPrio(messageKey),
      position);
  }

  /**
   * Adds an entry to the error list.
   * 
   * @param messageKey message key as defined in the resource bundle 
   *                   (properties file) for the test
   * @param link general purpose HTTP URL that points to details
   *        for this message (may be null) 
   * @param messageParameters key-value String pairs. Keys can be 
   *                          in the message text surrounded by curly
   *                          brackets. E.g. if you have a parameter
   *                          "FILE" -> "MyFile.txt", the message
   *                          in the resource bundle "error in {FILE}"
   *                          will be expanded to "error in MyFile.txt"
   *                          after localization of the message.
   * @param priority severity level (s. com.sap.tc.jtools.jtci.interfaces.ResultInterface)
   * @param position position (offset with respect to begin of file)
   */
  protected void addError(
    String messageKey,
    String link,
    Properties messageParameters,
    int dynamicPrio,
    Position position) {
    int pos = -1;
    if (position != null) {
      pos = position.getStartPosition();
    }
    Integer defaultPrio = (Integer) msgKey2prioMap.get(messageKey);
    if (defaultPrio != null) {
      // only allow dynamic prio if not overridden in descriptor
      if (defaultPrio.intValue() != ITestMessageDescription.PRIO_DYNAMIC) {
        dynamicPrio = defaultPrio.intValue();
      }
    }
    String relativeFileName =
      (currentCompilationUnit == null)
        ? ""
        : currentCompilationUnit.getFileName();
    if (currentCompilationUnit != null && testObject.getBaseDir() != null)
      relativeFileName =
        relativeFileName.substring(testObject.getBaseDir().length() + 1);
    if (pos == -1 || currentCompilationUnit == null)
      addError(
        new FilePosition(relativeFileName),
        messageKey,
        link,
        messageParameters,
        dynamicPrio);
    else
      addError(
        new FilePosition(
          relativeFileName,
          currentCompilationUnit.lineNumber(pos),
          currentCompilationUnit.columnNumber(pos)),
        messageKey,
        link,
        messageParameters,
        dynamicPrio);
  }

  /**
   * can be used to log an exception that occured
   * in a subclass. Adds an entry to the error list with 
   * severity == Result.SEVERITY_INTERNAL_ERROR and the stacktrace
   * of the exception.
   * pos may be null if not available.
   * 
   * @param pos current Position that was traversed when the
   *         exception occured (may be null)
   * @param e Exception to be logged
   */
  final public void logException(Exception e, Position pos) {
    StringWriter excWriter = new StringWriter();
    e.printStackTrace(new PrintWriter(excWriter));
    Properties msgProps = new Properties();
    msgProps.setProperty(EXCEPTION_TAG, excWriter.toString());
    addError(
      MSG_EXCEPTION_OCCURED,
      "",
      msgProps,
      Result.SEVERITY_INTERNAL_ERROR,
      pos);
  }

  final public void logException(Exception e) {
    logException(e, null);
  }

  /* final, internal methods */

  /* used internally */
  public String getRole() {
    return JomProjectVisitorProcessor.ROLE_JOM_VISITOR;
  }

  /* used internally */
  public final String getName() {
    return getTestName();
  }

  /* used internally */
  final public void setPrerequisites(Object[] values) {
    prereqValues = values;
  }

  /* used internally */
  final public List getErrorList() {
    return errorList;
  }

  /* used internally */
  final public Object getData() {
    if (resultWrapper != null)
      return resultWrapper;
    Result[] array = (Result[]) errorList.toArray(new Result[errorList.size()]);
    return new ResultWrapper(array);
  }

  /* used internally */
  final public void setProcessor(ProcessorInterface processor) {
    this.processor = processor;
  }

  /* used internally */
  public RequirementInterface getProcessorDefinition() {
    ParameterInterface[] par = new ParameterInterface[1];
    par[0] =
      ParameterTool.createParameter(
        ProcessorInterface.PARAMETER_TARGET_FILE_LIST,
        ParameterTool.PAR_TYPE_STRING_ARRAY,
        testObject.getFileNames());
    return new Requirement(JomProjectVisitorProcessor.CLASS_NAME, par) {
      ProcessorInterface processor = null;
      public ProcessorInterface getProcessor() {
        if (processor == null)
          processor = new JomProjectVisitorProcessor(testObject, searchPaths);
        return processor;
      }
    };
  }

  protected void addError(
    TestObjectPosition position,
    String messageKey,
    String link,
    Properties messageParameters) {
    addError(
      position,
      messageKey,
      link,
      messageParameters,
      getPrio(messageKey));
  }

  private void addError(
    TestObjectPosition position,
    String messageKey,
    String link,
    Properties messageParameters,
    int priority) {
    if (messageParameters == null) {
      messageParameters = new Properties();
    }
    if (currentCompilationUnit != null) {
	    IPackageDeclaration pkg = currentCompilationUnit.getPackage();
	    String packageName;
	    if (pkg == null) {
	      packageName = "";
	    } else {
	      packageName = NameTool.toDotNotation(pkg.getName());
	    }
	    messageParameters.setProperty(PARAMETER_PACKAGE, packageName);
    }
    messageParameters.setProperty(ParameterTool.PAR_MSG_KEY, messageKey);
    if (link != null) {
      messageParameters.setProperty(ParameterTool.PAR_LINK, link);
    }
    errorList.add(
      new Result(
        getName(),
        testObject.getType(),
        testObject.getID(),
        position,
        priority,
        "",
        messageParameters));
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#endVisit(com.sap.tc.jtools.jlint.jom.interfaces.IPackageName)
   */
  public void endVisit(IPackageName node) {
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#visit(com.sap.tc.jtools.jlint.jom.interfaces.IPackageName)
   */
  public boolean visit(IPackageName node) {
    return true;
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jlint.TestComponentInterface#setInternalParameters(com.sap.tc.jtools.jtci.interfaces.ParameterInterface[])
   */
  public void setInternalParameters(ParameterInterface[] params) {
    this.internalParameters = params;
    for (int i = 0; i < internalParameters.length; i++) {
      if (PARAMETER_PSEUDO_COMMENT_SUFFIX
        .equals(internalParameters[i].getName())) {
        this.pseudoComment =
          PSEUDO_COMMENT_PREFIX
            + ((String) internalParameters[i].getValue()).trim().toUpperCase()
            + PSEUDO_COMMENT_SUFFIX;
      }
    }
  }

  /**
   * handles disabling by pseudocomment.  
   * 
   * @see com.sap.tc.jtools.jlint.jom.interfaces.ITestVisitor#isDisabledFor(com.sap.tc.jtools.jlint.jom.interfaces.Traversable)
   */
  public final boolean isDisabledFor(Traversable node) {
    if (pseudoComment == null
      || !(node instanceof Commentable)
      || ((Commentable) node).getEndOfLineComment() == null)
      return false;
    return ((Commentable) node).getEndOfLineComment().getText().indexOf(
      pseudoComment)
      != -1;
  }

  /* (non-Javadoc)
   * @see com.sap.tc.jtools.jlint.TestComponentInterface#setMessages(com.sap.tc.jtools.jtci.interfaces.ITestMessageDescription[])
   */
  public void setMessages(ITestMessageDescription[] messages) {
    this.messages = messages;
    for (int i = 0; i < messages.length; i++) {
      msgKey2prioMap.put(
        messages[i].getKey(),
        new Integer(messages[i].getPriority()));
    }
  }

  private int getPrio(String msgKey) {
    Integer prio = (Integer) msgKey2prioMap.get(msgKey);
    if (prio == null)
      throw new IllegalArgumentException("unknown message key: " + msgKey);//$NON-NLS-1$
    return prio.intValue();
  }

}
