/*
 * Copyright (c) 2004 by SAP AG, Walldorf.
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_modules/_jom/_core/src/com/sap/tc/jtools/jlint/javaelements/Method.java#1 $
 */
 
package com.sap.tc.jtools.jlint.javaelements;

import com.sap.tc.jtools.jtci.exceptions.BadTreeException;
import com.sap.tc.jtools.util.persistable.Persistable;
import com.sap.tc.jtools.util.structures.Header;
import com.sap.tc.jtools.util.structures.StructureTree;

/**
 * Implementation of MethodInterface.
 * Instances of this class can be persisted in XML format.
 * 
 * @author d034036
 *
 */
public class Method implements MethodInterface, Persistable {

	private ClassInterface cls;
	private String methodName;
	private boolean isConstructor;
	private int modifiers;
	private String returnType;
	private String[] parameterTypes;
	private String[] exceptionTypes;

	public Method(
		String methodName,
		boolean isConstructor,
		int modifiers,
		String returnType,
		String[] parameterTypes,
		String[] exceptions) {
		this.methodName = methodName;
		this.isConstructor = isConstructor;
		this.modifiers = modifiers;
		this.returnType = returnType;
		this.parameterTypes = parameterTypes;
		this.exceptionTypes = exceptions;
	}

	public Method(StructureTree tree, ClassInterface cls)
		throws BadTreeException {
		if (tree.getHeader().getTag().equals("METHOD")) {
			isConstructor = false;
			returnType = tree.getParameter("RETURN_TYPE");
		} else if (tree.getHeader().getTag().equals("CONSTRUCTOR")) {
			isConstructor = true;
		} else {
			throw new BadTreeException(
				"wrong tag in method object: " + tree.getHeader().getTag());
		}
		this.cls = cls;
		methodName = tree.getParameter("NAME");
		modifiers = Integer.parseInt(tree.getParameter("MODIFIERS"));

		StructureTree[] parameterTrees = tree.getChildren("PARAMETER");
		parameterTypes = new String[parameterTrees.length];
		for (int i = 0; i < parameterTrees.length; i++) {
			parameterTypes[i] = parameterTrees[i].getText();
		}
		StructureTree[] exceptionTrees = tree.getChildren("EXCEPTION");
		exceptionTypes = new String[exceptionTrees.length];
		for (int i = 0; i < exceptionTrees.length; i++) {
			exceptionTypes[i] = exceptionTrees[i].getText();
		}
	}

	public StructureTree toStructureTree() {
		String methodOrConstructor = isConstructor ? "CONSTRUCTOR" : "METHOD";
		Header header = new Header(methodOrConstructor);
		header.setParameter("MODIFIERS", Integer.toString(modifiers));
		if (!isConstructor) {
			header.setParameter("RETURN_TYPE", returnType);
			header.setParameter("NAME", getMethodName());
		}
		StructureTree tree = new StructureTree(header);
		for (int i = 0; i < parameterTypes.length; i++) {
			StructureTree parTree = new StructureTree(new Header("PARAMETER"));
			parTree.setText(parameterTypes[i]);
			tree.addNewChild(parTree);
		}
		for (int i = 0; i < exceptionTypes.length; i++) {
			StructureTree excTree = new StructureTree(new Header("EXCEPTION"));
			excTree.setText(exceptionTypes[i]);
			tree.addNewChild(excTree);
		}
		return tree;
	}

	public String[] getExceptionTypes() {
		return exceptionTypes;
	}

	public String getMethodName() {
		if (isConstructor) {
			return null;
		}
		return methodName;
	}

	public int getModifiers() {
		return modifiers;
	}

	public String[] getParameterTypes() {
		return parameterTypes;
	}

	public String getReturnType() {
		return returnType;
	}

	public boolean isConstructor() {
		return isConstructor;
	}

	public ClassInterface getDefiningClass() {
		return cls;
	}

	public void setDefiningClass(ClassInterface cls) {
		this.cls = cls;
	}

}
