/*
 * Copyright (c) 2004 by SAP AG, Walldorf.
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_modules/_jom/_core/src/com/sap/tc/jtools/jlint/javaelements/Class.java#1 $
 */
 
package com.sap.tc.jtools.jlint.javaelements;

import com.sap.tc.jtools.jtci.exceptions.BadTreeException;
import com.sap.tc.jtools.util.persistable.Persistable;
import com.sap.tc.jtools.util.structures.Header;
import com.sap.tc.jtools.util.structures.StructureTree;

/**
 * Implementation of ClassInterface.
 * Instances of this class can be persisted in XML format.
 * 
 * @author d034036
 *
 */
public class Class implements ClassInterface, Persistable {

	private String pckName;
	private String className;
	private boolean intf;
	private int modifiers;
	private Method[] methods;
	private Field[] fields;
	private Class[] innerClasses;
	private String superclass;
	private String[] superinterfaces;

	public Class(
		String pckName,
		String className,
		String superclass,
		String[] superinterfaces,
		boolean intf,
		int modifiers,
		Method[] methods,
		Field[] fields,
		Class[] innerClasses) {
		this.pckName = pckName;
		this.className = className;
		this.superclass = superclass;
		this.superinterfaces = superinterfaces;
		this.intf = intf;
		this.modifiers = modifiers;
		this.methods = methods;
		this.fields = fields;
		this.innerClasses = innerClasses;

		for (int i = 0; i < methods.length; i++) {
			methods[i].setDefiningClass(this);
		}

		for (int i = 0; i < fields.length; i++) {
			fields[i].setDefiningClass(this);
		}
	}

	public Class(StructureTree tree) throws BadTreeException {
		if (tree.getHeader().getTag().equals("CLASS")) {
			intf = false;
		} else if (tree.getHeader().getTag().equals("INTERFACE")) {
			intf = true;
		} else {
			throw new BadTreeException(
				"wrong tag in class object: " + tree.getHeader().getTag());
		}
		pckName = tree.getParameter("PKG");
		className = tree.getParameter("NAME");

		superclass = tree.getParameter("SUPERCLASS");
		StructureTree[] siTrees = tree.getChildren("SUPERINTERFACE");
		superinterfaces = new String[siTrees.length];
		for (int i = 0; i < siTrees.length; i++) {
			superinterfaces[i] = siTrees[i].getText();
		}
		modifiers = Integer.parseInt(tree.getParameter("MODIFIERS"));

		StructureTree[] methodTrees = tree.getChildren("METHOD");
		StructureTree[] constructorTrees = tree.getChildren("CONSTRUCTOR");
		methods = new Method[methodTrees.length + constructorTrees.length];
		for (int i = 0; i < methodTrees.length; i++) {
			methods[i] = new Method(methodTrees[i], this);
		}
		for (int i = 0; i < constructorTrees.length; i++) {
			methods[methodTrees.length + i] =
				new Method(constructorTrees[i], this);
		}

		StructureTree[] classTrees = tree.getChildren("CLASS");
		StructureTree[] interfaceTrees = tree.getChildren("INTERFACE");
		innerClasses = new Class[classTrees.length + interfaceTrees.length];
		for (int i = 0; i < classTrees.length; i++) {
			innerClasses[i] = new Class(classTrees[i]);

		}
		for (int i = 0; i < interfaceTrees.length; i++) {
			innerClasses[classTrees.length + i] = new Class(interfaceTrees[i]);
		}

		StructureTree[] fieldTrees = tree.getChildren("FIELD");
		fields = new Field[fieldTrees.length];
		for (int i = 0; i < fieldTrees.length; i++) {
			fields[i] = new Field(fieldTrees[i], this);
		}
	}

	public String getClassName() {
		return className;
	}

	public FieldInterface[] getFields() {
		return fields;
	}

	public boolean isInterface() {
		return intf;
	}

	public MethodInterface[] getMethods() {
		return methods;
	}

	public int getModifiers() {
		return modifiers;
	}

	public String getPckName() {
		return pckName;
	}

	public StructureTree toStructureTree() {
		String classOrInterface = intf ? "INTERFACE" : "CLASS";
		Header header = new Header(classOrInterface);
		header.setParameter("PKG", pckName);
		header.setParameter("NAME", className);
		header.setParameter("MODIFIERS", Integer.toString(modifiers));
		if (superclass != null) {
			header.setParameter("SUPERCLASS", superclass);
		}
		StructureTree tree = new StructureTree(header);
		for (int i = 0; i < superinterfaces.length; i++) {
			Header siHeader = new Header("SUPERINTERFACE");
			StructureTree siTree = new StructureTree(siHeader);
			siTree.setText(superinterfaces[i]);
			tree.addNewChild(siTree);
		}

		for (int i = 0; i < fields.length; i++) {
			tree.addNewChild(fields[i].toStructureTree());
		}
		for (int i = 0; i < methods.length; i++) {
			tree.addNewChild(methods[i].toStructureTree());
		}
		for (int i = 0; i < innerClasses.length; i++) {
			tree.addNewChild(innerClasses[i].toStructureTree());
		}
		return tree;
	}

	public String getSuperclass() {
		return superclass;
	}

	public String[] getSuperinterfaces() {
		return superinterfaces;
	}

	public ClassInterface[] getInnerClasses() {
		return innerClasses;
	}

}
