/*
 * Copyright (c) 2004 by SAP AG, Walldorf.
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_modules/_jom/_core/src/com/sap/tc/jtools/jlint/javadiff/wrapper/Wrapper.java#3 $
 */
 
package com.sap.tc.jtools.jlint.javadiff.wrapper;

import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.List;

import com.sap.tc.jtools.jlint.javaelements.Class;
import com.sap.tc.jtools.jlint.javaelements.Field;
import com.sap.tc.jtools.jlint.javaelements.Method;
import com.sap.tc.jtools.jlint.jom.interfaces.IFieldDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.IMethodBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.IMethodDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.IReferenceTypeBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.ITypeBinding;
import com.sap.tc.jtools.jlint.jom.interfaces.ITypeDeclaration;
import com.sap.tc.jtools.jlint.jom.interfaces.IVariableBinding;

/**
 * This class converts Java source elements into the abstract representation used by 
 * the comparator.
 * 
 * 
 * @author d034036
 *
 */
public class Wrapper {

	
	/**
	 * converts a type declaration (i.e. the representation of the source
	 * associated with a class/interface) to its abstract representation
	 * 
	 * @param td type declaration
	 * 
	 * @return abstract representation
	 */
	public static Class wrap(ITypeDeclaration td) {
		IReferenceTypeBinding tmp = td.resolveBinding();
		Class result=wrap(tmp);
		return result;
 
	}
	
	/**
	 * converts a field declaration (i.e. the representation of the source
	 * associated with a field) to its abstract representation
	 * 
	 * @param declaration: field declaration
	 * 
	 * @return abstract representation
	 */
	public static Field wrap(IFieldDeclaration declaration) {
		IVariableBinding tmp = declaration.resolveBinding();
		Field result=wrap(tmp);
		return result;
	}
	
	/**
	 * converts a method declaration (i.e. the representation of the source
	 * associated with a method/constructor) to its abstract representation
	 * 
	 * @param declaration: method declaration
	 * 
	 * @return abstract representation
	 */
	public static Method wrap(IMethodDeclaration declaration) {
		IMethodBinding tmp = declaration.resolveBinding();
		Method result=wrap(tmp);
		return result;
	}
	
	
	/* binding-based methods */
	
	/**
	 * converts a reference type binding (i.e. the binding associated with a class/interface)
	 * to its abstract representation
	 * 
	 * @param tb type binding
	 * 
	 * @return abstract representation
	 */
	public static Class wrap(IReferenceTypeBinding tb) {
		/* pckName: fully qualified package name
		 * className: class name without package (but including all outer classes)
		 */
		String pckName = tb.getPackage().getName();
		String className = pckName == null ? tb.getName() : tb.getName().substring(pckName.length() + 1);

 

		/* class or interface? */
		boolean intf = tb.isInterface();
		
		/* modifiers - remove spurious jdt-internal modifiers */
		int modifiers =
			(tb.getModifiers() % Modifier.STRICT) & ~Modifier.SYNCHRONIZED;
			
		/* methods and constructors*/
		IMethodBinding[] methodsToBeWrapped = tb.getDeclaredMethods();
		List methodList = new ArrayList();

		for (int i = 0; i < methodsToBeWrapped.length; i++) {
			methodList.add(Wrapper.wrap(methodsToBeWrapped[i]));
		}
		Method[] methods =
			(Method[]) methodList.toArray(new Method[methodList.size()]);

		/* variables */
		IVariableBinding[] fieldsToBeWrapped = tb.getDeclaredFields();
		List fieldList = new ArrayList();

		for (int i = 0; i < fieldsToBeWrapped.length; i++) {
			fieldList.add(Wrapper.wrap(fieldsToBeWrapped[i]));
		}

		Field[] fields =
			(Field[]) fieldList.toArray(new Field[fieldList.size()]);

		/* superclass */
		String superclass = null;
		if (tb.getSuperclass() != null
			&& !tb.getSuperclass().getName().equals("java.lang.Object")) {
			superclass = tb.getSuperclass().getName();
		}
		else{
			if (intf){
				// Interfaces do not need a superclass
			}
			else{
				superclass = "java.lang.Object";
			}
		}

		/* superinterfaces */
		IReferenceTypeBinding[] siBindings = tb.getInterfaces();
		String[] superinterfaces = new String[siBindings.length];
		for (int i = 0; i < superinterfaces.length; i++) {
			superinterfaces[i] = siBindings[i].getName();
		}

		/* inner classes(interfaces */
		IReferenceTypeBinding[] innerTypeDeclarations = tb.getDeclaredTypes();
		Class[] innerClasses = new Class[innerTypeDeclarations.length];

		for (int i = 0; i < innerTypeDeclarations.length; i++) {
			innerClasses[i] = wrap(innerTypeDeclarations[i]);
		}
		return new Class(
			pckName,
			className,
			superclass,
			superinterfaces,
			intf,
			modifiers,
			methods,
			fields,
			innerClasses);
	}

	/**
	 * converts a variable binding (i.e. the binding associated with a field)
	 * to its abstract representation
	 * 
	 * @param field variable binding
	 * 
	 * @return abstract representation
	 */
	public static Field wrap(IVariableBinding field) {
		//String value = field.getValue();
		int modifiers = field.getModifiers() % Modifier.STRICT;
		return new Field(
			field.getName(),
			modifiers,
			field.getType().getName(),
			//value
			null);
	}

	/**
	 * converts a method binding (i.e. the binding associated with a method/constructor)
	 * to its abstract representation
	 * 
	 * @param method: method binding
	 * 
	 * @return abstract representation
	 */
	public static Method wrap(IMethodBinding method) {
		String methodName = method.isConstructor() ? null : method.getName();
		boolean isConstructor = method.isConstructor();
		int modifiers = method.getModifiers() % Modifier.STRICT;
		String returnType =
			isConstructor ? null : method.getReturnType().getName();
		ITypeBinding[] paramBindings = method.getParameterTypes();
		String[] parameterTypes = new String[paramBindings.length];
		for (int i = 0; i < parameterTypes.length; i++) {
			parameterTypes[i] = paramBindings[i].getName();
		}
		ITypeBinding[] excepBindings = method.getExceptionTypes();
		String[] excepTypes = new String[excepBindings.length];
		for (int i = 0; i < excepTypes.length; i++) {
			excepTypes[i] = excepBindings[i].getName();
		}

		return new Method(
			methodName,
			isConstructor,
			modifiers,
			returnType,
			parameterTypes,
			excepTypes);
	}


}
