/*
 * Copyright (c) 2004 by SAP AG, Walldorf.
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_modules/_jom/_core/src/com/sap/tc/jtools/jlint/javadiff/comparator/Comparator.java#1 $
 */
 
package com.sap.tc.jtools.jlint.javadiff.comparator;

import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import com.sap.tc.jtools.jlint.javaelements.ClassInterface;
import com.sap.tc.jtools.jlint.javaelements.FieldInterface;
import com.sap.tc.jtools.jlint.javaelements.MethodInterface;

/**
 * Compares classes (recursively), fields, and methods. A filter can be attached to 
 * filter out irrelevant elements (e.g. private members). A listener must be attached to 
 * process the results.
 * 
 * TODO: support for multiple listeners? 
 * 
 * @author d034036
 *
 */
public class Comparator {

	private ListenerInterface listener;

	// filter (default: test everything)
	private FilterInterface filter = new FilterInterface() {
		public boolean ignore(ClassInterface cls) {
			return false;
		}

		public boolean ignore(FieldInterface field) {
			return false;
		}

		public boolean ignore(MethodInterface method) {
			return false;
		}
	};

	/**
	 * compares two classes. It is assumed that the two classes have the same
	 * (qualified) name.
	 * 
	 * @param c1: Class 1
	 * @param c2: Class 2
	 */
	public void compareClasses(ClassInterface c1, ClassInterface c2) {
		// check class properties
		// a) modifiers
		if (c1.getModifiers() != c2.getModifiers()) {
			listener.differentClassModifiers(c1, c2);
		}

		// b) superclass
		String sc1 = c1.getSuperclass();
		String sc2 = c2.getSuperclass();

		if ((sc1 == null && sc2 != null)
			|| (sc1 != null && sc2 == null)
			|| (sc1 != null && sc2 != null && !sc1.equals(sc2))) {
			listener.differentSuperclass(c1, c2);
		}

		// c) superinterfaces

		Set si1 = new HashSet();
		Set si2 = new HashSet();

		si1.addAll(Arrays.asList(c1.getSuperinterfaces()));
		si2.addAll(Arrays.asList(c2.getSuperinterfaces()));

		for (Iterator iter = si1.iterator(); iter.hasNext();) {
			String si = (String) iter.next();

			if (!si2.contains(si)) {
				listener.extraSuperInterfaceIn1(c1, si);
			} else {
				si2.remove(si);
			}
		}
		for (Iterator iter = si2.iterator(); iter.hasNext();) {
			String si = (String) iter.next();
			listener.extraSuperInterfaceIn2(c2, si);
		}

		// check fields
		FieldInterface[] fields1 = c1.getFields();
		Map fieldMap1 = new HashMap();

		for (int i = 0; i < fields1.length; i++) {
			if (!filter.ignore(fields1[i])) {
				fieldMap1.put(fields1[i].getName(), fields1[i]);
			}
		}

		FieldInterface[] fields2 = c2.getFields();

		Map fieldMap2 = new HashMap();
		for (int i = 0; i < fields2.length; i++) {
			if (!filter.ignore(fields2[i])) {
				fieldMap2.put(fields2[i].getName(), fields2[i]);
			}
		}

		for (Iterator iter = fieldMap1.keySet().iterator(); iter.hasNext();) {
			String fieldName = (String) iter.next();
			FieldInterface field1 = (FieldInterface) fieldMap1.get(fieldName);
			FieldInterface field2 = (FieldInterface) fieldMap2.get(fieldName);
			if (field2 == null) {
				listener.extraFieldIn1(field1);
			} else {
				compareFields(field1, field2);
				fieldMap2.remove(fieldName);
			}
		}
		for (Iterator iter = fieldMap2.keySet().iterator(); iter.hasNext();) {
			String fieldName = (String) iter.next();
			FieldInterface field2 = (FieldInterface) fieldMap2.get(fieldName);
			listener.extraFieldIn2(field2);
		}

		// check inner classes
		ClassInterface[] classes1 = c1.getInnerClasses();
		Map classMap1 = new HashMap();

		for (int i = 0; i < classes1.length; i++) {
			if (!filter.ignore(classes1[i])) {
				classMap1.put(classes1[i].getClassName(), classes1[i]);
			}
		}

		ClassInterface[] classes2 = c2.getInnerClasses();
		Map classMap2 = new HashMap();

		for (int i = 0; i < classes2.length; i++) {
			if (!filter.ignore(classes2[i])) {
				classMap2.put(classes2[i].getClassName(), classes2[i]);
			}
		}

		for (Iterator iter = classMap1.keySet().iterator(); iter.hasNext();) {
			String className = (String) iter.next();
			ClassInterface class1 = (ClassInterface) classMap1.get(className);
			ClassInterface class2 = (ClassInterface) classMap2.get(className);
			if (class2 == null) {
				listener.extraClassIn1(c1, class1);
			} else {
				compareClasses(class1, class2);
				classMap2.remove(className);
			}
		}
		for (Iterator iter = classMap2.keySet().iterator(); iter.hasNext();) {
			String className = (String) iter.next();
			ClassInterface class2 = (ClassInterface) classMap2.get(className);
			listener.extraClassIn2(c1, class2);
		}

		// check methods
		MethodInterface[] methods1 = c1.getMethods();
		Map methodMap1 = new HashMap();

		for (int i = 0; i < methods1.length; i++) {
			if (!filter.ignore(methods1[i])) {
				String methodNameAndSignature =
					methods1[i].getMethodName() + "@";
				String[] params = methods1[i].getParameterTypes();
				for (int j = 0; j < params.length; j++) {
					methodNameAndSignature += params[j] + ";";
				}
				methodMap1.put(methodNameAndSignature, methods1[i]);
			}
		}

		MethodInterface[] methods2 = c2.getMethods();
		Map methodMap2 = new HashMap();
		for (int i = 0; i < methods2.length; i++) {
			if (!filter.ignore(methods2[i])) {
				String methodNameAndSignature =
					methods2[i].getMethodName() + "@";
				String[] params = methods2[i].getParameterTypes();
				for (int j = 0; j < params.length; j++) {
					methodNameAndSignature += params[j] + ";";
				}
				methodMap2.put(methodNameAndSignature, methods2[i]);
			}
		}

		Map spareMethods1 = new HashMap();
		for (Iterator iter = methodMap1.keySet().iterator(); iter.hasNext();) {
			String methodName = (String) iter.next();
			MethodInterface method1 =
				(MethodInterface) methodMap1.get(methodName);

			// check exact matching (name and signature) first
			MethodInterface method2 =
				(MethodInterface) methodMap2.get(methodName);
			if (method2 == null) {
				spareMethods1.put(methodName, method1);
			} else {
				compareMethods(method1, method2);
				methodMap2.remove(methodName);
			}
		}

		for (Iterator iter = spareMethods1.keySet().iterator();
			iter.hasNext();
			) {
			String methodName = (String) iter.next();
			MethodInterface method1 =
				(MethodInterface) methodMap1.get(methodName);

			// check different signature
			String method1NameOnly = method1.getMethodName();
			boolean found = false;
			for (Iterator iterator = methodMap2.keySet().iterator();
				iterator.hasNext();
				) {
				String method2Name = (String) iterator.next();
				if (method2Name.startsWith(method1NameOnly + "@")) {
					MethodInterface method2 =
						(MethodInterface) methodMap2.get(method2Name);
					listener.differentMethodSignatures(
						method1,
						method2);
					compareMethods(method1, method2);
					methodMap2.remove(method2Name);
					found = true;
					break;
				}
			}
			if (!found) {
				listener.extraMethodIn1(method1);
			}

		}

		for (Iterator iter = methodMap2.keySet().iterator(); iter.hasNext();) {
			String methodName = (String) iter.next();
			MethodInterface method2 =
				(MethodInterface) methodMap2.get(methodName);
			listener.extraMethodIn2(method2);
		}

	}

	/**
	 * compares two methods. It is assumed that the methods have the same 
	 * name and signature.
	 * 
	 * @param m1: Method 1
	 * @param m2: Method 2
	 */
	public void compareMethods(MethodInterface m1, MethodInterface m2) {

		// compare modifiers
		if (m1.getModifiers() != m2.getModifiers()) {
			listener.differentMethodModifiers(m1, m2);
		}

		// compare return types
		if (!m1.isConstructor()) {
			if (!m1.getReturnType().equals(m2.getReturnType())) {
				listener.differentReturnTypes(m1, m2);
			}
		}

		// signatures have already been checked

		// compare exceptions
		String[] exc1 = m1.getExceptionTypes();
		String[] exc2 = m2.getExceptionTypes();

		if (exc1.length != exc2.length) {
			listener.differentExceptions(m1, m2);
		} else {
			for (int i = 0; i < exc1.length; i++) {
				if (!exc1[i].equals(exc2[i])) {
					listener.differentExceptions(m1, m2);
					break;
				}
			}
		}
	}

	/**
	 * compares two fields. It is assumed that the fields have the same 
	 * name.
	 * 
	 * @param f1: Field 1
	 * @param f2: Field 2
	 */
	public void compareFields(FieldInterface f1, FieldInterface f2) {
		// compare modifiers
		if (f1.getModifiers() != f2.getModifiers()) {
			listener.differentFieldModifiers(f1, f2);
		}

		// compare types
		if (!f1.getType().equals(f2.getType())) {
			listener.differentFieldTypes(f1, f2);
		}

		// compare values
		if ((f1.getValue() != null && !f1.getValue().equals(f2.getValue()))
			|| (f1.getValue() == null
			&& f2.getValue() != null)) {
			listener.differentFieldValues(f1, f2);
		}

	}

	/**
	 * sets listener (the old one is replaced)
	 * 
	 * @param l listener
	 */
	public void attachListener(ListenerInterface l) {
		listener = l;
	}

	/**
	 * sets the filter.
	 * @param f: filter
	 */
	public void attachFilter(FilterInterface f) {
		filter = f;
	}
}
