/*
 * Copyright (c) 2004 by SAP AG, Walldorf.
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_modules/_jom/_core/src/com/sap/tc/jtools/jlint/javadiff/bytecode/BinaryParser.java#1 $
 */
 
package com.sap.tc.jtools.jlint.javadiff.bytecode;

import java.io.UnsupportedEncodingException;
import java.lang.reflect.Modifier;
import java.util.Vector;

import com.sap.tc.jtools.jlint.javaelements.Class;
import com.sap.tc.jtools.jlint.javaelements.Field;
import com.sap.tc.jtools.jlint.javaelements.Method;

/**
 * This class is used to parse Java bytecode and to convert it into the abstract format
 * used by the comparator.
 * 
 * @author d034036
 *
 */
public class BinaryParser {

	private final static int ACC_INTERFACE = 0x200;

	private String className;
	private String packageName;
	private String superTypeName;
	private String[] interfaceNames;
	private int constant_pool_count;
	private ConstantPoolEntry[] constant_pool;
	private int access_flags;
	private int this_class;
	private int super_class;
	private int interfaces_count;
	private int interfaces[];
	private int fields_count;
	private FieldOrMethodInfo field_info[];
	private int methods_count;
	private FieldOrMethodInfo method_info[];
	private int attributes_count;
	private AttributeInfo attribute_info[];

	private Vector methods = new Vector();
	private Vector fields = new Vector();
	private Vector innerClasses = new Vector();

	// text representation for entry types in the constant pool
	final static String[] entryTypes =
		{
			"unused",
			"UTF8",
			"unused",
			"integer",
			"float",
			"long",
			"double",
			"class",
			"string",
			"fieldref",
			"methodref",
			"interfacemethodref",
			"nameandtype" };
	
	/**
	 * converts bytecode into the abstract representation of the corresponding Java class.
	 * 
	 * @param classFile: the class to be parsed
	 * @return the abstract represention of the class
	 */
	public Class parseClass(byte[] classFile) {
		return parseClass(classFile, new Class[0]);
	}
			
	/**
	 * converts bytecode into the abstract representation of the corresponding Java class.
	 * 
	 * @param classFile: the class to be parsed
	 * @return the abstract represention of the class
	 */
	public Class parseClass(byte[] classFile, Class[] innerClasses) {

		int offset = 0;
		parseHeader(classFile);

		offset = parseConstantPool(classFile);
		access_flags = bytesToUShort(classFile, offset);

		offset += 2;
		this_class = bytesToUShort(classFile, offset);

		offset += 2;
		super_class = bytesToUShort(classFile, offset);

		offset += 2;
		interfaces_count = bytesToUShort(classFile, offset);

		offset += 2;
		interfaces = new int[interfaces_count];
		for (int i = 0; i < interfaces_count; i++) {
			interfaces[i] = bytesToUShort(classFile, offset);
			offset += 2;
		}
		fields_count = bytesToUShort(classFile, offset);

		offset += 2;
		field_info = new FieldOrMethodInfo[fields_count];
		for (int i = 0; i < fields_count; i++) {
			field_info[i] = new FieldOrMethodInfo(classFile, offset);
			offset += field_info[i].getSize();
			//				  System.out.println("Field " + i + ": " + field_info[i]);
		}

		methods_count = bytesToInt(classFile, offset, 2);
		//			  System.out.println("Methods Count: " + methods_count);
		offset += 2;
		method_info = new FieldOrMethodInfo[methods_count];
		for (int i = 0; i < methods_count; i++) {
			method_info[i] = new FieldOrMethodInfo(classFile, offset);
			offset += method_info[i].getSize();
			//				  System.out.println("Method " + i + ": " + method_info[i]);
		}

		attributes_count = bytesToInt(classFile, offset, 2);
		//			  System.out.println("Attributes Count: " + attributes_count);
		offset += 2;
		attribute_info = new AttributeInfo[attributes_count];
		for (int i = 0; i < attributes_count; i++) {
			attribute_info[i] = (new AttributeInfo(classFile, offset)).unwrap();
			offset += attribute_info[i].getSize();
			//				  System.out.println("Attribute " + i + ": " + attribute_info[i]);
		}

		resolveClass();

		return new Class(
			packageName,
			className,
			superTypeName,
			interfaceNames,
			(access_flags & ACC_INTERFACE) == ACC_INTERFACE,
			access_flags & ~Modifier.SYNCHRONIZED,
			(Method[]) methods.toArray(new Method[methods.size()]),
			(Field[]) fields.toArray(new Field[fields.size()]),
			innerClasses);

	}

 

	private void parseHeader(byte[] classFile) {
		if (classFile.length < 10) {
			throw new IllegalClassFormat("class file does not contain a complete header");
		}
		byte[] magic = new byte[4];
		magic[0] = classFile[0];
		magic[1] = classFile[1];
		magic[2] = classFile[2];
		magic[3] = classFile[3];
		if (magic[0] != -54
			|| magic[1] != -2
			|| magic[2] != -70
			|| magic[3] != -66)
			throw new IllegalClassFormat("wrong magic number, file is not a java class file.");
	}

	private int parseConstantPool(byte[] classFile) {
		int parse_offset;
		constant_pool_count = bytesToUShort(classFile, 8);
		constant_pool = new ConstantPoolEntry[constant_pool_count];
		//		  System.out.println("constant pool size: " + constant_pool_count);
		parse_offset = 10;
		for (int i = 1; i < constant_pool_count; i++) {
			constant_pool[i] = new ConstantPoolEntry(parse_offset, classFile);
			//			  System.out.println("Index " + i + ": " + constant_pool[i].toString());
			parse_offset += constant_pool[i].getSize();
			if (constant_pool[i].skipNext())
				i++;
		}
		return parse_offset;
	}

	private void resolveClass() {
		className = constant_pool[this_class].getName().replace('/', '.');
		if (className.indexOf('.') != -1) {
			packageName = className.substring(0, className.lastIndexOf('.'));
			className = className.substring(packageName.length() + 1);
		} else {
			packageName = "<null>";
		}
		className = className.replace('$','.');

		superTypeName = constant_pool[super_class].getName().replace('/', '.');
		if (superTypeName.equals("java.lang.Object")) {
			superTypeName = null;
		}

		// these weird checks (class$...) are needed to filter out 
		// synthetic fields added by the compiler in the case
		// of inner classes or .class references
		String name;
		for (int i = 0; i < method_info.length; i++) {
			if (!method_info[i].getName().equals("<clinit>")
				&& !method_info[i].getName().startsWith("class$")) {
				methods.add(parseMethod(method_info[i]));
			}

		}

		for (int i = 0; i < field_info.length; i++) {
			if (!field_info[i].getName().startsWith("class$") 
					&& !field_info[i].getName().startsWith("this$")) {
				fields.add(parseField(field_info[i]));
			}

		}

		interfaceNames = new String[interfaces_count];
		for (int i = 0; i < interfaces_count; i++) {
			interfaceNames[i] =
				constant_pool[interfaces[i]].getName().replace('/', '.');
		}

	}

 
	private Field parseField(FieldOrMethodInfo info) {
		String name = info.getName();
		String type =
			convertType(constant_pool[info.descriptor_index].getName());
		int modifiers = info.access_flags;
		String value = null;
		if (Modifier.isFinal(modifiers)) {
			// get initializer value
			ConstantValueAttribute attr = info.getConstantValue();
			// if there is a ConstantValue-Attribute there is an initializer
			// otherwise we cannot compute the variables value
			if (attr != null) {
				value =
					constant_pool[attr
						.constantvalue_index]
						.getValue()
						.toString();

			}
		}
		return new Field(name, modifiers, type, value);

	}
 
	private Method parseMethod(FieldOrMethodInfo info) {
		String methodName = info.getName();
		boolean isConstructor = methodName.equals("<init>");
		int modifiers = info.access_flags;
		String[] paramTypes;
		String[] exceptionTypes;
		String desc = info.getDescriptor();
		int close = desc.indexOf(')');
		int temp[] = new int[100];
		int count = 0;
		if (desc.charAt(0) != '(' || close == -1) {
			throw new IllegalClassFormat("Invalid Method Descriptor: " + desc);
		}

		String returnType =
			convertType(desc.substring(close + 1, desc.length()));

		if (close > 1) {
			String pars = desc.substring(1, close);
			String type;
			Vector params = new Vector();

			while (!pars.equals("")) {
				temp[count] = getDimension(pars);
				type =
					parseType(
						pars.substring(temp[count], pars.length())).replace(
						'/',
						'.');
				String paramWithDims = convertType(type);
				for (int i = 0; i < temp[count]; i++)
					paramWithDims += "[]";
				params.add(paramWithDims);
				if (pars.charAt(temp[count]) == 'L')
					pars =
						pars.substring(
							temp[count] + type.length() + 2,
							pars.length());
				else
					pars = pars.substring(temp[count] + 1, pars.length());
				count++;
			}
			paramTypes = new String[params.size()];
			paramTypes = (String[]) params.toArray(paramTypes);
		} else {
			paramTypes = new String[0];
		}

		ExceptionsAttribute ex = info.getExceptions();
		if (ex == null) {
			exceptionTypes = new String[0];
		} else {
			exceptionTypes = new String[ex.numberOfExceptions];
			for (int i = 0; i < ex.numberOfExceptions; i++) {
				exceptionTypes[i] =
					constant_pool[ex.exceptionIndexTable[i]].getName().replace(
						'/',
						'.');
			}
		}
		return new Method(
			methodName,
			isConstructor,
			modifiers,
			returnType,
			paramTypes,
			exceptionTypes);
	}

 
	private String convertType(String string) {
		//		System.out.println("converting " + string);
		if (string.equals("V")) {
			return "void";
		}
		if (string.equals("B")) {
			return "byte";
		}
		if (string.equals("Z")) {
			return "boolean";
		}
		if (string.equals("I")) {
			return "int";
		}
		if (string.equals("J")) {
			return "long";
		}
		if (string.equals("C")) {
			return "char";
		}
		if (string.equals("S")) {
			return "short";
		}
		if (string.startsWith("L")) {
			return string.substring(1, string.length() - 1).replace('/', '.');
		}
		if (string.equals("D")) {
			return "double";
		}
		if (string.equals("F")) {
			return "float";
		}
		if (string.startsWith("[")) {
			int dim = 0;
			String brackets = "";
			String bareString = string;
			while (bareString.startsWith("[")) {
				bareString = bareString.substring(1);
				dim++;
				brackets += "[]";
			}
			return convertType(bareString) + brackets;
		}

		return string;

	}

	protected int getDimension(String desc) {
		int pos = 0;
		while (desc.charAt(pos) == '[') {
			pos++;
		}
		return pos;
	}

	protected String parseType(String desc) {
		switch (desc.charAt(0)) {
			case 'B' :
			case 'C' :
			case 'D' :
			case 'F' :
			case 'I' :
			case 'J' :
			case 'S' :
			case 'Z' :
			case 'V' :
				return desc.substring(0, 1);
			case 'L' :
				return desc.substring(1, desc.indexOf(';'));
			default :
				throw new IllegalClassFormat(
					"Invalid Method Descriptor: " + desc);
		}
	}

	static int bytesToInt(byte[] bytes, int startoff, int length) {
		int result = 0;
		for (int i = 0; i < length; i++) {
			result = result << 8;
			result += (bytes[startoff + i] & 0xFF);
		}
		return result;
	}

	static int bytesToUShort(byte[] bytes, int startoff) {
		return (((bytes[startoff] & 0xff) << 8) | (bytes[startoff + 1] & 0xff));
	}

	static long bytesToLong(byte[] bytes, int startoff, int length) {
		long result = 0;
		for (int i = 0; i < length; i++) {
			result = result << 8;
			result += (bytes[startoff + i] & 0xFF);
		}
		return result;
	}

	/**
	 * represents an entry in the constant pool of the class file
	 */
	final class ConstantPoolEntry {
		final static byte CLASS = 7;
		final static byte FIELDREF = 9;
		final static byte METHODREF = 10;
		final static byte INTERFACEMETHODREF = 11;
		final static byte STRING = 8;
		final static byte INTEGER = 3;
		final static byte FLOAT = 4;
		final static byte LONG = 5;
		final static byte DOUBLE = 6;
		final static byte NAMEANDTYPE = 12;
		final static byte UTF8 = 1;

		/**
		 * identifies the type of the class pool entry.
		 */
		byte tag;

		/**
		 * contains the entries data. the type of data depend on the
		 * value of 'tag'
		 */
		ConstantPoolInfo info;

		/**
		 * the length of the datas representation in the class file.
		 * this is required when parsing the class file to advance the cursor.
		 */
		private int length;

		ConstantPoolEntry(final int offset, final byte classFile[]) {
			tag = classFile[offset];
			length = 1;
			switch (tag) {
				case CLASS :
					info =
						new CPIClass(bytesToUShort(classFile, offset + length));
					length += 2;
					break;
				case FIELDREF :
				case METHODREF :
				case INTERFACEMETHODREF :
					info =
						new CPIRef(
							bytesToUShort(classFile, offset + length),
							bytesToUShort(classFile, offset + length + 2));
					length += 4;
					break;
				case STRING :
					info =
						new CPIString(
							bytesToUShort(classFile, offset + length));
					length += 2;
					break;
				case INTEGER :
					info =
						new CPIInt(bytesToInt(classFile, offset + length, 4));
					length += 4;
					break;
				case FLOAT :
					info =
						new CPIFloat(
							Float.intBitsToFloat(
								bytesToInt(classFile, offset + length, 4)));
					length += 4;
					break;
				case LONG :
					info =
						new CPILong(bytesToLong(classFile, offset + length, 8));
					length += 8;
					break;
				case DOUBLE :
					info =
						new CPIDouble(
							Double.longBitsToDouble(
								bytesToLong(classFile, offset + length, 8)));
					length += 8;
					break;
				case NAMEANDTYPE :
					info =
						new CPINameAndType(
							bytesToUShort(classFile, offset + length),
							bytesToUShort(classFile, offset + length + 2));
					length += 4;
					break;
				case UTF8 :
					int len = bytesToUShort(classFile, offset + length);
					info = new CPIUTF8(len, classFile, offset + length + 2);
					length += (2 + len);
					break;
				default :
					throw new IllegalClassFormat(
						"illegal entry in constant pool: tag" + tag);
			}
		}

		Object getValue() {
			switch (tag) {
				case ConstantPoolEntry.STRING :
				case ConstantPoolEntry.DOUBLE :
				case ConstantPoolEntry.FLOAT :
				case ConstantPoolEntry.INTEGER :
				case ConstantPoolEntry.LONG :
					return info.getValue();
				default :
					throw new IllegalClassFormat("constant pool entry has invalid type!");
			}
		}

		/**
		 * returns true iff the entry represents a long or double literal.
		 * These entries take up to 'slots' in the constant pool. (the next index
		 * in the constant pool table is not used)
		 * I'd really like to know who came up with such a brilliant idea...
		 */
		boolean skipNext() {
			if (tag == LONG || tag == DOUBLE) {
				return true;
			} else {
				return false;
			}
		}

		/**
		 * returns the data size of the entry
		 */
		int getSize() {
			return this.length;
		}

		public String toString() {
			return entryTypes[tag] + ";" + info.toString();
		}

		/**
		 * this method is only supported for entries that represent string literals
		 * or classes/interfaces.
		 * it retrieves the contents of the literal or the class/interface name
		 * respectively
		 */
		String getName() {
			switch (tag) {
				case CLASS :
					return constant_pool[((CPIClass) info)
						.name_index]
						.getName();
				case UTF8 :
					CPIUTF8 utf = (CPIUTF8) info;
					return utf.getName();
				default :
					throw new java.lang.UnsupportedOperationException();
			}
		}
	}

	/**
	 * abstract base class for grouping the different constant pool info objects
	 */
	abstract class ConstantPoolInfo {
		abstract Object getValue();
	}

	final class CPIClass extends ConstantPoolInfo {
		int name_index;

		CPIClass(int name_index) {
			this.name_index = name_index;
		}

		public String toString() {
			return "name_index=" + name_index;
		}

		Object getValue() {
			throw new UnsupportedOperationException("getValue() not supported for CPIClass");
		}
	}

	final class CPIRef extends ConstantPoolInfo {
		int class_index;
		int name_and_type_index;

		CPIRef(int class_index, int name_and_type_index) {
			this.class_index = class_index;
			this.name_and_type_index = name_and_type_index;
		}

		public String toString() {
			return "class_index="
				+ class_index
				+ ";name_and_type_index="
				+ name_and_type_index;
		}

		Object getValue() {
			throw new UnsupportedOperationException("getValue() not supported for CPIRef");
		}
	}

	final class CPIString extends ConstantPoolInfo {
		int string_index;

		CPIString(int string_index) {
			this.string_index = string_index;
		}
		public String toString() {
			return "string_index=" + string_index;
		}

		Object getValue() {
			return constant_pool[string_index].getName();
		}
	}

	final class CPIInt extends ConstantPoolInfo {
		int value;

		CPIInt(int value) {
			this.value = value;
		}

		public String toString() {
			return "value=" + value;
		}

		Object getValue() {
			return new Integer(value);
		}
	}

	final class CPIFloat extends ConstantPoolInfo {
		float value;

		CPIFloat(float value) {
			this.value = value;
		}
		public String toString() {
			return "value=" + value;
		}

		Object getValue() {
			return new Float(value);
		}
	}

	final class CPILong extends ConstantPoolInfo {
		long value;

		CPILong(long value) {
			this.value = value;
		}
		public String toString() {
			return "value=" + value;
		}
		Object getValue() {
			return new Long(value);
		}
	}

	final class CPIDouble extends ConstantPoolInfo {
		double value;

		CPIDouble(double value) {
			this.value = value;
		}
		public String toString() {
			return "value=" + value;
		}
		Object getValue() {
			return new Double(value);
		}
	}

	final class CPINameAndType extends ConstantPoolInfo {
		int name_and_type_index;
		int descriptor_index;

		CPINameAndType(int name_and_type_index, int descriptor_index) {
			this.descriptor_index = descriptor_index;
			this.name_and_type_index = name_and_type_index;
		}

		public String toString() {
			return "name_and_type_index="
				+ name_and_type_index
				+ "; descriptor_index="
				+ descriptor_index;
		}
		Object getValue() {
			throw new UnsupportedOperationException("getValue() not supported for CPINameAndType");
		}
	}

	final class CPIUTF8 extends ConstantPoolInfo {
		int length;
		byte values[];
		String val;

		CPIUTF8(int length, byte[] classFile, int offset) {
			this.length = length;
			values = new byte[length];
			System.arraycopy(classFile, offset, values, 0, length);
			try {
				val = new String(values, "UTF-8");
			} catch (UnsupportedEncodingException e) {
				throw new IllegalClassFormat("illegal utf8 string: " + values);
			}
			// free values so that we do not waste space
			values = null;
		}
		public String toString() {
			return "value=" + val;
		}

		public String getName() {
			return val;
		}
		Object getValue() {
			throw new UnsupportedOperationException("getValue() not supported for CPIUTF8");
		}
	}

	final class FieldOrMethodInfo {
		int access_flags;
		int name_index;
		int descriptor_index;
		int attributes_count;
		int exceptions = -1;
		int constantValue = -1;
		AttributeInfo attributes[];

		private int length;

		FieldOrMethodInfo(byte[] byteCode, int offset) {
			length = 8;
			access_flags = bytesToInt(byteCode, offset, 2);
			name_index = bytesToInt(byteCode, offset + 2, 2);
			descriptor_index = bytesToInt(byteCode, offset + 4, 2);
			attributes_count = bytesToInt(byteCode, offset + 6, 2);
			attributes = new AttributeInfo[attributes_count];
			for (int i = 0; i < attributes_count; i++) {
				attributes[i] =
					new AttributeInfo(byteCode, offset + length).unwrap();
				length += attributes[i].getSize();
				if (attributes[i] instanceof ExceptionsAttribute) {
					exceptions = i;
				}
				if (attributes[i] instanceof ConstantValueAttribute) {
					constantValue = i;
				}
			}
		}

		int getSize() {
			return length;
		}

		/*		public String toString() {
					return accessToString(access_flags) +
						   constant_pool[name_index].getName() + " (" + name_index +
						   ") ;descriptor_index=" + descriptor_index +
						   ";attributes_count=" + attributes_count;
				} */

		String getName() {
			return constant_pool[name_index].getName();
		}

		String getDescriptor() {
			return constant_pool[descriptor_index].getName();
		}

		ExceptionsAttribute getExceptions() {
			if (exceptions != -1) {
				return (ExceptionsAttribute) attributes[exceptions];
			} else {
				return null;
			}
		}

		ConstantValueAttribute getConstantValue() {
			if (constantValue != -1) {
				return (ConstantValueAttribute) attributes[constantValue];
			} else {
				return null;
			}
		}
	}

	class AttributeInfo {
		int attribute_name_index;
		long attribute_length;
		byte info[];

		private int length;

		protected AttributeInfo(int name_index, long len) {
			this.attribute_name_index = name_index;
			this.attribute_length = len;
			this.length = 6 + (int) len;
		}

		AttributeInfo(byte byteCode[], int offset) {
			attribute_name_index = bytesToInt(byteCode, offset, 2);
			attribute_length = bytesToLong(byteCode, offset + 2, 4);
			info = new byte[(int) attribute_length];
			for (long i = 0; i < attribute_length; i++) {
				info[(int) i] = byteCode[offset + 6 + (int) i];
			}
			length = 6 + (int) attribute_length;
		}
		public int getSize() {
			return length;
		}

		public String toString() {
			return "name_index="
				+ attribute_name_index
				+ ";atribute_length="
				+ attribute_length
				+ ";info="
				+ info;
		}

		AttributeInfo unwrap() {
			String name = constant_pool[attribute_name_index].getName();
			if (name.equals("SourceFile")) {
				return new SourceFileAttribute(
					attribute_name_index,
					attribute_length,
					info);
			}
			if (name.equals("InnerClasses")) {
				return new InnerClassesAttribute(
					attribute_name_index,
					attribute_length,
					info);
			}
			if (name.equals("ConstantValue")) {
				return new ConstantValueAttribute(
					attribute_name_index,
					attribute_length,
					info);
			}
			if (name.equals("Exceptions")) {
				return new ExceptionsAttribute(
					attribute_name_index,
					attribute_length,
					info);
			}
			return this;
		}
	}

	final class ConstantValueAttribute extends AttributeInfo {
		int attribute_name_index;
		int attribute_length;
		int constantvalue_index;

		ConstantValueAttribute(int name_index, long length, byte[] data) {
			super(name_index, length);
			constantvalue_index = bytesToInt(data, 0, 2);
		}
	}

	final class SourceFileAttribute extends AttributeInfo {
		int sourceFileIndex;

		SourceFileAttribute(int name_index, long length, byte[] data) {
			super(name_index, length);
			sourceFileIndex = bytesToInt(data, 0, 2);
		}

		String getFileName() {
			return constant_pool[sourceFileIndex].getName();
		}
	}

	final class ExceptionsAttribute extends AttributeInfo {
		int numberOfExceptions;
		int exceptionIndexTable[];

		ExceptionsAttribute(int name_index, long length, byte[] data) {
			super(name_index, length);
			numberOfExceptions = bytesToInt(data, 0, 2);
			exceptionIndexTable = new int[numberOfExceptions];
			for (int i = 0; i < numberOfExceptions; i++) {
				exceptionIndexTable[i] = bytesToInt(data, 2 + (i * 2), 2);
			}
		}
	}

	final class InnerClassesInfo {
		int innerClassInfoIndex;
		int outerClassInfoIndex;
		int innerNameIndex;
		int innerClassAccessFlags;

		InnerClassesInfo(byte[] data, int off) {
			innerClassInfoIndex = bytesToInt(data, off, 2);
			outerClassInfoIndex = bytesToInt(data, off + 2, 2);
			innerNameIndex = bytesToInt(data, off + 4, 2);
			innerClassAccessFlags = bytesToInt(data, off + 6, 2);
		}
	}

	final class InnerClassesAttribute extends AttributeInfo {
		int numberOfClasses;
		InnerClassesInfo[] info;

		InnerClassesAttribute(int name_index, long length, byte[] data) {
			super(name_index, length);
			numberOfClasses = bytesToInt(data, 0, 2);
			info = new InnerClassesInfo[numberOfClasses];
			for (int i = 0; i < numberOfClasses; i++) {
				info[i] = new InnerClassesInfo(data, 2 + (i * 8));
			}
		}
	}

}
