/*
 * Copyright (c) 2004 by SAP AG, Walldorf.
 * http://www.sap.com
 * All rights reserved.
 *
 * This software is the confidential and proprietary information
 * of SAP AG, Walldorf. You shall not disclose such Confidential
 * Information and shall use it only in accordance with the terms
 * of the license agreement you entered into with SAP.
 * 
 * $Id: //tc/jtools/630_VAL_REL/src/_jlint/java/_modules/_jom/_core/src/com/sap/tc/jtools/jlint/javadiff/Main.java#1 $
 */
 
package com.sap.tc.jtools.jlint.javadiff;

import java.io.File;
import java.io.FileReader;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import com.sap.tc.jtools.jlint.javadiff.comparator.Comparator;
import com.sap.tc.jtools.jlint.javadiff.comparator.ListenerInterface;
import com.sap.tc.jtools.jlint.javaelements.Class;
import com.sap.tc.jtools.jlint.javaelements.ClassInterface;
import com.sap.tc.jtools.jlint.javaelements.FieldInterface;
import com.sap.tc.jtools.jlint.javaelements.MethodInterface;
import com.sap.tc.jtools.jlint.javadiff.util.APIFilter;
import com.sap.tc.jtools.util.xml.XMLTool;

/**
 * This class ccompares two API file trees. Differences result in messages 
 * to the standard output.
 * 
 * @author d034036
 *
 */
public class Main {

	/**
	 * API file suffix.
	 */
	static public final String API_SUFFIX = "_api.xml"; //$NON-NLS-1$ 
	
	private static String root1;
	private static String root2;
	private static Comparator comparator;

	/**
	 * Compares two API file trees
	 * 
	 * 
	 * @param args: args[0]: first directory
	 *              args[1]: second directory
	 */
	public static void main(String[] args) {
		/* remove trailing separators */
		root1 = args[0];
		if (root1.endsWith("\\") || root1.endsWith("/")) {
			root1 = root1.substring(0, root1.length() - 1);
		}
		root2 = args[1];
		if (root2.endsWith("\\") || root2.endsWith("/")) {
			root2 = root2.substring(0, root2.length() - 1);
		}

		File dir1 = new File(args[0]);
		File dir2 = new File(args[1]);

		/* initialize comparator */
		comparator = new Comparator();
		comparator.attachFilter(new APIFilter());
		comparator.attachListener(new Listener());
		
		/* start recursive descent */
		checkDirectories(dir1, dir2);

	}

	/*
	 * Compares directories (recursively). It is assumed that directories 
	 * contain _api.xml files *only*.
	 *  
	 * @param dir1: first directory
	 * @param dir2: second directory
	 */
	private static void checkDirectories(File dir1, File dir2) {
		/* collect files in the two directories */
		Map children1Map = new HashMap();
		File[] children1 = dir1.listFiles();
		for (int i = 0; i < children1.length; i++) {
			children1Map.put(children1[i].getName(), children1[i]);
		}

		Map children2Map = new HashMap();
		File[] children2 = dir2.listFiles();
		for (int i = 0; i < children2.length; i++) {
			children2Map.put(children2[i].getName(), children2[i]);
		}

		/* check if files or directories in 1 are missing from 2 */
		for (Iterator iter = children1Map.keySet().iterator();
			iter.hasNext();
			) {
			String currentFileName = (String) iter.next();
			File currentFile1 = (File) children1Map.get(currentFileName);
			File currentFile2 = (File) children2Map.get(currentFileName);
			if (currentFile1.isDirectory()) {
				if (currentFile2 == null || !currentFile2.isDirectory()) {
					System.out.println(
						"missing package in 2: "
							+ dir2.getAbsolutePath().substring(
								root2.length() + 1)
							+ File.separator
							+ currentFileName);
					continue;
				}
				checkDirectories(currentFile1, currentFile2);
			} else {
				if (currentFile2 == null || !currentFile2.isFile()) {
					System.out.println(
						"missing class in 2: "
							+ dir2.getAbsolutePath().substring(
								root2.length() + 1)
							+ File.separator
							+ currentFileName.substring(
								0,
								currentFileName.length()
									- API_SUFFIX.length()));
					continue;
				}
				checkFiles(currentFile1, currentFile2);
			}
			children2Map.remove(currentFileName);
		}
		
		/* check if files or directories in 2 are missing from 1 */
		for (Iterator iter = children2Map.keySet().iterator();
			iter.hasNext();
			) {
			String currentFileName = (String) iter.next();
			File currentFile2 = (File) children2Map.get(currentFileName);
			if (currentFile2.isDirectory()) {
				System.out.println(
					"missing package in 1: "
						+ dir1.getAbsolutePath().substring(root1.length() + 1)
						+ File.separator
						+ currentFileName+"\n");
			} else {
				System.out.println(
					"missing class in 1: "
						+ dir2.getAbsolutePath().substring(root2.length() + 1)
						+ File.separator
						+ currentFileName.substring(
							0,
							currentFileName.length() - "_api.xml".length())+"\n");
			}
		}
	}

	/*
	 * Compares files. It is assumed that the files are _api.xml files.
	 *  
	 * @param file1: first file.
	 * @param filer: second file.
	 */
	private static void checkFiles(File file1, File file2) {
		try {
			/* parse XML files */
			Class c1 =
				new Class(XMLTool.parseReader(new FileReader(file1)));
			Class c2 =
				new Class(XMLTool.parseReader(new FileReader(file2)));
			/* compare */
			comparator.compareClasses(c1, c2);
		} catch (Exception e) {
			e.printStackTrace();
		}

	}

	/*
	 * Text-based listener that writes messages to System.out.
	 *
	 */
	private static class Listener implements ListenerInterface {

		public void extraFieldIn1(FieldInterface field1) {
			System.out.println(getPrefix(field1) + ": extra field in 1\n");
		}

		public void extraFieldIn2(FieldInterface field2) {
			System.out.println(getPrefix(field2) + ": extra field in 2\n");
		}

		public void extraMethodIn1(MethodInterface method1) {
			System.out.println(getPrefix(method1) + ": extra method in 1\n");
		}

		public void extraMethodIn2(MethodInterface method2) {
			System.out.println(getPrefix(method2) + ": extra method in 2\n");
		}

		public void differentMethodSignatures(
			MethodInterface method1,
			MethodInterface method2) {
			System.out.println(
				"different method signatures:\n   1:"
					+ getPrefix(method1)
					+ "\n   2:"
					+ getPrefix(method2)
					+ "\n");
		}

		public void differentClassModifiers(
			ClassInterface c1,
			ClassInterface c2) {
			System.out.println(
				"different class modifiers:\n"
					+ "   1:"
					+ getPrefix(c1)
					+ " --> "
					+ c1.getModifiers()
					+ "\n"
					+ "   2:"
					+ getPrefix(c2)
					+ " --> "
					+ c2.getModifiers()
					+ "\n");
		}

		public void differentSuperclass(ClassInterface c1, ClassInterface c2) {
			System.out.println(
				"different superclass:\n"
					+ "   1:"
					+ getPrefix(c1)
					+ " --> "
					+ c1.getSuperclass()
					+ "\n"
					+ "   2:"
					+ getPrefix(c2)
					+ " --> "
					+ c2.getSuperclass()
					+ "\n");
		}

		public void extraSuperInterfaceIn1(ClassInterface c1, String si) {
			System.out.println(
				"extra super interface in 1: "
					+ si
					+ "\n"
					+ "   "
					+ getPrefix(c1)
					+ "\n");
		}

		public void extraSuperInterfaceIn2(ClassInterface c2, String si) {
			System.out.println(
				"extra super interface in 2: "
					+ si
					+ "\n"
					+ "   "
					+ getPrefix(c2)
					+ "\n");

		}

		public void extraClassIn1(ClassInterface c1, ClassInterface cls) {
			System.out.println(
				"extra class in 1:\n"
					+ "   "
					+ getPrefix(c1)
					+ "\n"
					+ "     :"
					+ getPrefix(cls)
					+ "\n");
		}

		public void extraClassIn2(ClassInterface c2, ClassInterface cls) {
			System.out.println(
				"extra class in 2:\n"
					+ "   "
					+ getPrefix(c2)
					+ "\n"
					+ "     :"
					+ getPrefix(cls)
					+ "\n");
		}

		public void differentMethodModifiers(
			MethodInterface m1,
			MethodInterface m2) {
			System.out.println(
				"different method modifiers:\n"
					+ "   1:"
					+ getPrefix(m1)
					+ " --> "
					+ m1.getModifiers()
					+ "\n"
					+ "   2:"
					+ getPrefix(m2)
					+ " --> "
					+ m2.getModifiers()
					+ "\n");
		}

		public void differentReturnTypes(
			MethodInterface m1,
			MethodInterface m2) {
			System.out.println(
				"different return types:\n"
					+ "   1:"
					+ getPrefix(m1)
					+ " --> "
					+ m1.getReturnType()
					+ "\n"
					+ "   2:"
					+ getPrefix(m2)
					+ " --> "
					+ m2.getReturnType()
					+ "\n");
		}

		public void differentExceptions(
			MethodInterface m1,
			MethodInterface m2) {
			System.out.println(
				"different exceptions:\n"
					+ "   1:"
					+ getPrefix(m1)
					+ "\n"
					+ "   2:"
					+ getPrefix(m2)
					+ "\n");
		}

		public void differentFieldModifiers(
			FieldInterface f1,
			FieldInterface f2) {
			System.out.println(
				"different field modifiers:\n"
					+ "   1:"
					+ getPrefix(f1)
					+ " --> "
					+ f1.getModifiers()
					+ "\n"
					+ "   2:"
					+ getPrefix(f2)
					+ " --> "
					+ f2.getModifiers()
					+ "\n");
		}

		public void differentFieldTypes(FieldInterface f1, FieldInterface f2) {
			System.out.println(
				"different field types:\n"
					+ "   1:"
					+ getPrefix(f1)
					+ " --> "
					+ f1.getType()
					+ "\n"
					+ "   2:"
					+ getPrefix(f2)
					+ " --> "
					+ f2.getType()
					+ "\n");
		}

		public void differentFieldValues(
			FieldInterface f1,
			FieldInterface f2) {
			System.out.println(
				"different field values:\n"
					+ "   1:"
					+ getPrefix(f1)
					+ " --> "
					+ f1.getValue()
					+ "\n"
					+ "   2:"
					+ getPrefix(f2)
					+ " --> "
					+ f2.getValue()
					+ "\n");
		}
		
		private String getPrefix(ClassInterface c) {
			String out = "";
			out += "class: " + c.getPckName() + "." + c.getClassName() + "  ";
			return out;
		}

		private String getPrefix(FieldInterface f) {
			ClassInterface c = f.getDefiningClass();
			String out = getPrefix(c);
			out += "field: " + f.getName() + "  ";
			return out;
		}

		private String getPrefix(MethodInterface m) {
			ClassInterface c = m.getDefiningClass();
			String sig = "@";
			String[] sigComponents = m.getParameterTypes();
			for (int i = 0; i < sigComponents.length; i++) {
				sig += sigComponents[i] + ";";
			}
			String out = getPrefix(c);
			String methodName = m.getMethodName();
			if (methodName == null) {
				methodName = "<constructor>";
			}
			out += "method: " + methodName + sig + "  ";
			return out;
		}

	}
}
