package com.sap.tc.jtools.jlint.java;

import com.sap.tc.jtools.util.structures.*;
import com.sap.tc.jtools.jtci.CodeClassificationConstants;
import com.sap.tc.jtools.jtci.FilePosition;
import com.sap.tc.jtools.jtci.TestObject;
import com.sap.tc.jtools.jtci.TestObjectPosition;
import com.sap.tc.jtools.jtci.exceptions.*;
import com.sap.tc.jtools.jtci.interfaces.*;

/**
 * represents a test object consisting of an set of Java source files.
 * 
 * <p>
 * Copyright (c) 2003, SAP AG
 * 
 */

public class JavaFileArrayTestObject extends TestObject {

	/**
	 *  name of the Java-file-list test object type
	 */
	public final static String JAVA_FILE_LIST = "Java file list"; //$NON-NLS-1$ 

	private String[] fileNames;
	private String id;
	private String baseDir;
	private String[] codeClassification = ALL; //default

	/* XML representation */
	protected final static String TAG_ITEM = "ITEM"; //$NON-NLS-1$
	protected final static String PARAMETER_BASEDIR = "BASE_DIR"; //$NON-NLS-1$    
	public static final String CODE_CLASSIFICATION_TAG = "CODE_CLASSIFICATION"; //$NON-NLS-1$

	private static final String[] ALL =
		new String[] { CodeClassificationConstants.ALL_CODE_STR };

	public JavaFileArrayTestObject() {

	}

	/**
	 * Standard constructor.
	 * <p>The object ID is computed internally</p>
	 * 
	 * @param baseDir base directory
	 * @param fileNames array of file names (type: String[])
	 * 
	 */
	public JavaFileArrayTestObject(
		String baseDir,
		String[] fileNames,
		String[] codeClassification)
		throws InvalidTestObjectSpecification {
		this.baseDir = baseDir;
		this.fileNames = fileNames;
		this.id = Integer.toString(fileNames.hashCode());
		this.codeClassification = codeClassification;
	}

	public JavaFileArrayTestObject(String baseDir, String[] fileNames)
		throws InvalidTestObjectSpecification {
		this.baseDir = baseDir;
		this.fileNames = fileNames;
		this.id = Integer.toString(fileNames.hashCode());
	}

	/**
	 * Constructor.
	 * 
	 * @param baseDir base directory
	 * @param fileNames array of file names (type: String[])
	 * @param id object ID
	 * 
	 */
	public JavaFileArrayTestObject(
		String baseDir,
		String[] fileNames,
		String id,
		String[] codeClassification) {
		this.baseDir = baseDir;
		this.fileNames = fileNames;
		this.id = id;
		this.codeClassification = codeClassification;
	}

	/**
	 * Constructor used to restore an object from its StructureTree representation)
	 * 
	 * @param tree test object in StructureTree format.
	 * @throws BadTreeException if the StructureTree does not correspond to a valid object
	 * 
	 */
	public JavaFileArrayTestObject(StructureTree tree)
		throws BadTreeException {
		if (!tree.getTag().equals(TAG_TEST_OBJECT)) {
			throw new BadTreeException();
		}
		StructureTree[] children = tree.getChildren(TAG_ITEM);
		baseDir = tree.getParameter(PARAMETER_BASEDIR);
		String[] files = new String[children.length];
		for (int kk = 0; kk < files.length; kk++) {
			files[kk] = children[kk].getText();
		}
		String testObjectID = tree.getParameter(PARAMETER_ID);
		this.fileNames = files;
		if (testObjectID == null) {
			this.id = Integer.toString(files.hashCode());
		} else {
			this.id = testObjectID;
		}
		StructureTree[] appliesToTrees =
			tree.getChildren(CODE_CLASSIFICATION_TAG);
		if (appliesToTrees.length > 0) {
			codeClassification = new String[appliesToTrees.length];
			for (int i = 0; i < appliesToTrees.length; i++) {
				codeClassification[i] = appliesToTrees[i].getText();
			}
		}
	}

	/**
	 * returns the base directory.
	 * 
	 * @return base directory
	 * 
	 */
	public String getBaseDir() {
		return baseDir;
	}

	/* implementation of methods defined in the superclass (TestObject) */
	public String getType() {
		return JAVA_FILE_LIST;
	}

	public String[] getFileNames() {
		return fileNames;
	}

	/**
	 * @deprecated for backwards compat. of Jlin adapter
	 */
	public Object getValue() {
		return fileNames;
	}

	public String getID() {
		return id;
	}

	/* StructureTree representation */
	public StructureTree toStructureTree() {

		Header sourceHeader = new Header(TEST_OBJECT);
		sourceHeader.setParameter(
			ParameterInterface.PARAMETER_TYPE,
			JAVA_FILE_LIST);
		sourceHeader.setParameter(PARAMETER_ID, getID());
		if (baseDir != null)
			sourceHeader.setParameter(PARAMETER_BASEDIR, baseDir);
		StructureTree sourceTree = new StructureTree(sourceHeader);

		String[] files = getFileNames();
		for (int k = 0; k < files.length; k++) {
			StructureTree fileTree = new StructureTree(new Header(TAG_ITEM));
			fileTree.setText(files[k]);
			sourceTree.addNewChild(fileTree);
		}
		for (int i = 0; i < codeClassification.length; i++) {
			StructureTree newTree =
				new StructureTree(new Header(CODE_CLASSIFICATION_TAG));
			newTree.setText(codeClassification[i]);
			sourceTree.addNewChild(newTree);
		}
		return sourceTree;
	}

	public static TestObjectPosition getPosition(
		String file,
		int line,
		int column) {
		return new FilePosition(file, line, column);
	}

	public static TestObjectPosition getPosition(String file) {
		return new FilePosition(file);
	}

	/**
	 * @see com.sap.tc.jtools.jtci.TestObject#getComponentCount()
	 */
	public int getComponentCount() {
		return fileNames.length;
	}

	/**
	 * @return
	 */
	public String[] getCodeClassification() {
		return codeClassification;
	}

	/**
	 * @param strings
	 */
	public void setCodeClassification(String[] strings) {
		codeClassification = strings;
	}

}