/*
 * SAPMarkets Copyright (c) 2001
 * All rights reserved
 *
 * @version $Id: //sapmarkets/BaseTech/630_VAL_REL/src/_fileloader/java/com/sapmarkets/technology/fileloader/JARFile.java#1 $
 */

package com.sapmarkets.technology.fileloader;

import java.io.*;
import java.util.Enumeration;
import java.util.jar.*;
import java.util.zip.ZipEntry;

/**
 * Class source represented by a JAR file.
 *
 * @created   24. July 2001
 */
public class JARFile implements IFileSource
{
  private File jarFile = null;

  /**
   * Construct JARFile object.
   *
   * @param jarFile                        JAR file
   * @exception ResourceNotFoundException  Exception thrown when class source
   *                                       invalid
   */
  public JARFile(File jarFile) throws ResourceNotFoundException
  {
    if ((jarFile!=null) && (jarFile.exists()) && (jarFile.isFile())) {
      this.jarFile = jarFile;
    }
    else {
      throw new ResourceNotFoundException(
		      "Failed to instantiate class source!");
    }
  }

  /**
   * Get input stream for file name.
   *
   * @param name                        Name of file to be retrieved
   * @return                            Input stream for file to be retrieved
   * @exception ResourceNotFoundException Exception thrown when file to be
   *                                    retrieved not found
   */
  public InputStream getInputStream(String name)
			throws ResourceNotFoundException
  {
    try {
      JarFile jarFile=new JarFile(this.jarFile);
      if (jarFile!=null) {
	JarEntry jarEntry = jarFile.getJarEntry(name);
	// System.out.println("(jar)checking for "+name);
	if (jarEntry!=null) {
	  return jarFile.getInputStream(jarEntry);
	}
      }
    }
    catch( Exception exception ) {
      // Do nothing since an exception will be thrown automatically
    }
    throw new ResourceNotFoundException(name);
  }

  public boolean isDirectory(String name)
			throws ResourceNotFoundException
  { 
    try {
      JarFile jarFile=new JarFile(this.jarFile);
      if (jarFile!=null) {
	JarEntry jarEntry = jarFile.getJarEntry(name);
	// System.out.println("(jar)checking for "+name);
	if (jarEntry!=null) {
	  return jarEntry.isDirectory();
	}
      }
    }
    catch( Exception exception ) {
      // Do nothing since an exception will be thrown automatically
    }
    throw new ResourceNotFoundException(name);
  }

  public long lastModified(String name)
			throws ResourceNotFoundException
  { return jarFile.lastModified();
  }

  public boolean hasFile(String name)
  {
    try {
      JarFile jarFile=new JarFile(this.jarFile);
      if (jarFile!=null) {
	JarEntry jarEntry = jarFile.getJarEntry(name);
	// System.out.println("(jar)checking for "+name);
	return jarEntry!=null;
      }
    }
    catch( Exception exception ) {
      // Do nothing since an exception will be thrown automatically
    }
    return false;
  }

  public Enumeration entries()
  { return new JAREnumeration(jarFile);
  }

  class JAREnumeration implements Enumeration {
    Enumeration entries;

    JAREnumeration(File jarFile)
    {
      try {
	entries=new JarFile(jarFile).entries();
      }
      catch (IOException e) {
      }
    }

    public boolean hasMoreElements()
    { return entries!=null && entries.hasMoreElements();
    }

    public Object nextElement()
    { ZipEntry e=(ZipEntry)entries.nextElement();
      return e.getName();
    }
  }

  /**
   * Get file path this file source references.
   * Each entry including the last must be terminated with a semicolon.
   *
   * @return   File path this file source references
   */
  public String getFilePath()
  {
    return jarFile.getPath() + ";";
  }

  /**
   * Determine whether this and the other object are equal.
   *
   * @param otherObject  The other object
   * @return             Flag indicating whether or not this and the other
   *                     object are equal
   */
  public boolean equals( Object otherObject )
  {
    if (otherObject instanceof JARFile) {
      return ((JARFile)otherObject).getFilePath().equals(getFilePath());
    }
    return false;
  }

}
