/*
 * SAPMarkets Copyright (c) 2001
 * All rights reserved
 *
 * @version $Id: //sapmarkets/BaseTech/630_VAL_REL/src/_fileloader/java/com/sapmarkets/technology/fileloader/FileLoader.java#1 $
 */

package com.sapmarkets.technology.fileloader;


import java.io.*;
import java.security.*;
import java.util.*;
import java.util.jar.*;

/**
 * Custom file loader.
 *
 * @created   24. July 2001
 */
public class FileLoader implements IFileSource
{
  private FileLoader parent;

  // File sources
  protected Vector fileSources;

  // Security access control context
  protected AccessControlContext accessControlContext;

  public FileLoader()
  {
    // Instantiate private members
    fileSources=new Vector();
    accessControlContext=AccessController.getContext();
  }

  public FileLoader(FileLoader parent)
  { this();
    this.parent=parent;
  }

  public FileLoader getParent()
  { return parent;
  }

  public InputStream getInputStream(String name)
			throws ResourceNotFoundException
  {
    try {
      for( Iterator iter=fileSources.iterator(); (iter.hasNext()); ) {
	try {
	  return ((IFileSource)iter.next()).getInputStream(name);
	}
	catch( ResourceNotFoundException exception ) {
	  // Try next one
	}
      }
      if (parent!=null) return parent.getInputStream(name);
    }
    catch (Exception exception) {
      // Do nothing since an exception will be thrown automatically
    }
    throw new ResourceNotFoundException(name);
  }

  public boolean isDirectory(String name)
			throws ResourceNotFoundException
  {
    try {
      for( Iterator iter=fileSources.iterator(); (iter.hasNext()); ) {
	try {
	  return ((IFileSource)iter.next()).isDirectory(name);
	}
	catch( ResourceNotFoundException exception ) {
	  // Try next one
	}
      }
      if (parent!=null) return parent.isDirectory(name);
    }
    catch (Exception exception) {
      // Do nothing since an exception will be thrown automatically
    }
    throw new ResourceNotFoundException(name);
  }

  public long lastModified(String name)
			throws ResourceNotFoundException
  {
    try {
      for( Iterator iter=fileSources.iterator(); (iter.hasNext()); ) {
	try {
	  return ((IFileSource)iter.next()).lastModified(name);
	}
	catch( ResourceNotFoundException exception ) {
	  // Try next one
	}
      }
      if (parent!=null) return parent.lastModified(name);
    }
    catch (Exception exception) {
      // Do nothing since an exception will be thrown automatically
    }
    throw new ResourceNotFoundException(name);
  }

  protected boolean hasLocalFile(String name)
  {
    for( Iterator iter=fileSources.iterator(); (iter.hasNext()); ) {
      if (((IFileSource)iter.next()).hasFile(name)) return true;
    }
    return false;
  }

  public boolean hasFile(String name)
  {
    if (hasLocalFile(name)) return true;
    if (parent!=null) return parent.hasFile(name);
    return false;
  }

  public Enumeration entries()
  { return new SourceEnumeration();
  }

  class SourceEnumeration implements Enumeration {
    Iterator    i;
    Enumeration files;
    boolean     inParent;

    SourceEnumeration()
    { i=fileSources.iterator();
      inParent=false;
      nextSource();
      
    }
    
    void nextSource()
    { 
      while ((files==null || !files.hasMoreElements()) && i.hasNext()) {
	files=((IFileSource)i.next()).entries();
      }
      if (files==null && !inParent) {
	inParent=true;
	if (parent!=null) files=parent.entries();
      }
    }

    public boolean hasMoreElements()
    { 
      return files!=null && files.hasMoreElements();
    }

    public Object nextElement()
    { Object o=files.nextElement();
      nextSource();
      return o;
    }
  }

  /**
   * Get file path this file source references.
   * Each entry including the last must be terminated
   * with a semicolon.
   *
   * @return   File path this file source references
   */
  public String getFilePath()
  { StringBuffer result=new StringBuffer();

    if (getParent()!=null) {
      result.append(getParent().getFilePath());
    }
    for( Iterator iter=fileSources.iterator();iter.hasNext();) {
      result.append(((IFileSource )iter.next()).getFilePath());
    }
    return result.toString();
  }


  /**
   * Append file source to the current file sources (at the end).
   *
   * @param fileSource                     File source to be added
   * @exception ResourceNotFoundException  Exception thrown when class source
   *                                       invalid
   */
  public void appendFileSource(IFileSource fileSource )
      throws ResourceNotFoundException
  {
    if (fileSource!=null) {
      fileSources.add(fileSource);
    }
    else {
      throw new ResourceNotFoundException("Failed to append file source!");
    }
  }


  /**
   * Insert file source into the current file sources (at the beginning).
   *
   * @param fileSource                     File source to be inserted
   * @exception ResourceNotFoundException  Exception thrown when class source
   *                                       invalid
   */
  public void insertFileSource(IFileSource fileSource)
      throws ResourceNotFoundException
  {
    if (fileSource!=null) {
      fileSources.insertElementAt( fileSource, 0 );
    }
    else {
      throw new ResourceNotFoundException("Failed to insert file source!");
    }
  }


  /**
   * Remove file source from the current file sources.
   *
   * @param fileSource                     File source to be removed
   * @exception ResourceNotFoundException  Exception thrown when file source
   *                                       invalid
   */
  public void removeFileSource(IFileSource fileSource)
      throws ResourceNotFoundException
  {
    if (fileSource!=null) {
      fileSources.remove(fileSource);
    }
    else {
      throw new ResourceNotFoundException( "Failed to remove file source!");
    }
  }


  /**
   * Remove all file sources from the current file sources.
   *
   */
  public void removeAllClassSources()
  {
    fileSources.clear();
  }

  /**
   * Suggest file source based on file name.
   *
   * @param fileSourceFile                 file to be used to suggest file
   *                                       source
   * @return                               suggested file source
   * @exception ResourceNotFoundException  Exception raised in failure situation
   */
  public IFileSource suggestFileSource(File fileSourceFile)
      throws ResourceNotFoundException
  {
    if (fileSourceFile.exists()) {

      // Check for directory
      if (fileSourceFile.isDirectory()) {
	return new FileDirectory(fileSourceFile);
      }

      // Check for file
      if (fileSourceFile.isFile() ) {
	try {
	  JarFile jf=new JarFile(fileSourceFile);
	  return new JARFile(fileSourceFile);
        }
	catch (IOException e) {
	  throw new ResourceNotFoundException(fileSourceFile.getPath());
	}
      }

      // Nothing matched
      throw new ResourceNotFoundException( "Failed to suggest class source" );
    }
    else {
      throw new ResourceNotFoundException(
		  "File doesn't exist; can't suggest file source");
    }
  }


  public FileLoader getSubFileLoader(String prefix)
  { return new SubFileLoader(this,prefix);
  }
}
