/*
 * SAPMarkets Copyright (c) 2001
 * All rights reserved
 *
 * @version $Id: //sapmarkets/BaseTech/630_VAL_REL/src/_fileloader/java/com/sapmarkets/technology/fileloader/FileDirectory.java#1 $
 */

package com.sapmarkets.technology.fileloader;

import java.io.*;
import java.util.Enumeration;

/**
 * Class source represented by a class directory.
 *
 * @created   24. July 2001
 */
public class FileDirectory implements IFileSource
{
  private File fileDirectory = null;
  private String filePath    = null;
  private int pathLength     = 0;

  /**
   * Construct FileDirectory object.
   *
   * @param fileDirectory                  File directory
   * @exception ResourceNotFoundException  Exception thrown when file source 
   *                                       invalid
   */
  public FileDirectory(File fileDirectory) throws ResourceNotFoundException
  {
    if ((fileDirectory!=null) && (fileDirectory.isDirectory())) {
      try {
	this.fileDirectory = fileDirectory;
	this.filePath      = fileDirectory.getCanonicalPath();
	this.pathLength    = filePath.length();
      }
      catch (IOException e) {
	throw new ResourceNotFoundException(
				 "Failed to instantiate file source!");
      }
    }
    else {
      throw new ResourceNotFoundException(
			       "Failed to instantiate file source!");
    }
  }

  /**
   * Get input stream for a file name.
   *
   * @param name                        Name of file to be retrieved
   * @return                            Input stream for file to be retrieved
   * @exception ResourceNotFoundException  Exception thrown when file to be 
   *                                    retrieved not found
   */
  public InputStream getInputStream(String name)
			throws ResourceNotFoundException
  {
    name.replace('/',File.separatorChar);
    try {
      File resFile = new File(fileDirectory, name);
      //System.out.println("(dir)checking for "+resFile.getPath());
      if ((resFile!=null) && (resFile.exists()) && (resFile.isFile())) {
	return new FileInputStream(resFile);
      }
    }
    catch (Exception exception) {
	// Do nothing since an exception will be thrown automatically
    }
    throw new ResourceNotFoundException(name);
  }

  public boolean isDirectory(String name)
			throws ResourceNotFoundException
  { 
    name.replace('/',File.separatorChar);
    try {
      return new File(fileDirectory,name).isDirectory();
    }
    catch (Exception e) {
      throw new ResourceNotFoundException(name);
    }
  }

  public long lastModified(String name)
			throws ResourceNotFoundException
  { 
    name.replace('/',File.separatorChar);
    try {
      return new File(fileDirectory,name).lastModified();
    }
    catch (Exception e) {
      throw new ResourceNotFoundException(name);
    }
  }

  public boolean hasFile(String name)
  { 
    name.replace('/',File.separatorChar);
    return new File(fileDirectory,name).exists();
  }

  public Enumeration entries()
  { try {
      return new FileEnumeration(fileDirectory); 
    }
    catch (IOException e) {
      return new FileEnumeration();
    }
  }

  /**
   * Get file path this file source references.
   * Each entry including the last must be terminated
   * with a semicolon.
   *
   * @return   File path this file source references
   */
  public String getFilePath()
  {
    return fileDirectory.getPath() + ";";
  }

  /**
   * Determine whether this and the other object are equal.
   *
   * @param otherObject  The other object
   * @return             Flag indicating whether or not this and the other
   *                     object are equal
   */
  public boolean equals( Object otherObject )
  {
    if (otherObject instanceof FileDirectory) {
      return ((FileDirectory)otherObject).getFilePath().equals(getFilePath());
    }
    return false;
  }

}
