/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.mi.version;

import com.sap.netweaver.bc.rf.common.*;
import com.sap.netweaver.bc.rf.common.exception.*;
import com.sap.netweaver.bc.rf.common.version.*;

import java.util.*;

//TODO mba: insert @link tags where applicable.
//TODO mba: specify ResourceException subtypes on methods that throw specific exceptions

/**
 * Interface describing the write methods of a repository sub manager for basic
 * versioning features. For an explanation of terminological conventions, please
 * see IBasicVersioningMananger. <p>
 *
 * Copyright (c) SAP AG 2003
 *
 * @author manfred.baedke@greenbytes.de
 * @created 20. Mrz 2003
 */
public interface IMutableBasicVersioningManager extends IBasicVersioningManager {

  /**
   * Enables or disables version control for the resource referred by the given
   * resource handle. Depending on the implementation and configuration of the
   * repository, the version history of a vcr may be lost when version control
   * is disabled on the vcr.
   *
   * @param resourceHandle a resource handle
   * @param enabled true if versioning is to be enabled
   * @exception ResourceException if the resource handle is invalid or the
   *      operation failed
   */
  public void setVersionControlEnabled(IResourceHandle resourceHandle, boolean enabled)
    throws ResourceException;


  /**
   * Check out a checked-in vcr to prepare subsequent changes of it's version
   * controlled state. These changes may be persisted in a newly created version
   * by a subsequent call to checkIn(), or they may be rolled back by a
   * subsequent call to undoCheckOut(). <p>
   *
   * The forkOk parameter must be ignored if the server does not support the
   * interface <code>IAdvancedVersioningManager </code> , in which case forks
   * are disallowed anyway. Otherwise, the semantics of this parameter depends
   * on the fork behaviour of the checked-in version of the given vcr (see
   * <code>IAdvancedVersioningManager.getCheckOutForkBehaviour</code> ): If
   * forks are strictly allowed or forbidden (as stated by
   * getCheckOutForkBehaviour()), the parameter must be ignored. Otherwise, if
   * forkOk is false and the server expects a subsequent checkin to create a
   * branched version history, the method must fail.
   *
   * @param vcrHandle the resource handle of a checked-in vcr
   * @param forkOk allow forks in version history upon check in
   * @return ICheckOutInfo object containing the expected resource handle of the
   *      next version
   * @exception ResourceException if the resource handle is invalid or the
   *      operation failed
   */
  public ICheckOutInfo checkOutInPlace(IResourceHandle vcrHandle, boolean forkOk)
    throws ResourceException;


  /**
   * Roll back a previous checkout operation. The version controlled state of
   * the given checked-out vcr is updated from it's checked-out version. <p>
   *
   * The method returns a set of handles of the resources modified by the
   * operation, which is not trivial in the case of a more advanced versioning
   * manager supporting version controlled collections.
   *
   * @param vcrHandle the resource handle of a checked-out vcr
   * @return a Set of IResourceHandles of all resources modified by the
   *      operation
   * @exception ResourceException if the resource handle is invalid or the
   *      operation failed
   */
  public Set undoCheckOut(IResourceHandle vcrHandle)
    throws ResourceException;


  /**
   * Check in a checked-out vcr to create a new version describing the current
   * version controlled state of the vcr. <p>
   *
   * The state of the new version will be initialized from the current state of
   * the given resource, where 'state' is defined to consist of content, dead
   * properties and - in the case of a version controlled collection - the set
   * of version controlled bindings of the collection together with ordering
   * information. <p>
   *
   * If an expected checkin RID is provided, the method may fail if the server
   * cannot assign this RID to the new version. <p>
   *
   * The forkOk parameter must be ignored if the server does not support the
   * interface <code>IAdvancedVersioningManager </code> , in which case forks
   * are disallowed anyway. Otherwise, the semantics of this parameter depends
   * on the fork behaviour of the checked-out version of the given vcr (see
   * <code>IAdvancedVersioningManager.getCheckInForkBehaviour</code> ): If forks
   * are strictly allowed or forbidden (as stated by getCheckInForkBehaviour()),
   * the parameter must be ignored. Otherwise, if forkOk is false and the
   * checkin created a branched version history, the method must fail.
   *
   * @param expectedCheckInRID the expected RID of the new version (may be null)
   * @param forkOk allow a new fork in the version history
   * @param keepCheckedOut if true, the given vcr will be checked out immediatly
   *      after the checkin succeded
   * @param resourceHandle
   * @return an ICheckInInfo object describing the newly created version
   * @exception ResourceException if the resource handle is invalid or the
   *      operation failed
   */
  public ICheckInInfo checkIn(IResourceHandle resourceHandle, IRid expectedCheckInRID,
    boolean forkOk, boolean keepCheckedOut)
    throws ResourceException;


  /**
   * The version controlled state of the checked-in vcr referred by the given
   * vcrHandle parameter will be updated from the state of the version refered
   * by the given versionHandle, and this version will become the new checked-in
   * version of the vcr. <p>
   *
   * The method returns a set of handles of the resources modified by the
   * operation, is be not trivial in the case of a more advanced versioning
   * manager supporting version controlled collections.
   *
   * @param vcrHandle the resource handle of a checked-out vcr
   * @param versionHandle the resource handle of a version of the vcr refered by
   *      vcrHandle
   * @return a Set of IResourceHandles of all resources modified by the
   *      operation
   * @exception ResourceException if the resource handle is invalid or the
   *      operation failed
   */
  public Set updateFromVersion(IResourceHandle vcrHandle, IResourceHandle versionHandle)
    throws ResourceException;


  //-------------------------------------auto child versioning ('enhanced collection') methods-------------------------

  /**
   * Enables or disables auto child version controlling for the collection
   * referred by the given collection handle. Enabling auto child version
   * controlling means that newly created direct non-collection children will
   * automatically put under version control.<p>
   *
   * For performance reasons, the caller should set the parameter returnModified
   * to false, if possible.
   *
   * @param collectionHandle a resource handle referring to a collection
   * @param modifyPlainChildren if true all direct plain children will be put
   *      under version control
   * @param returnModified if false, an empty Set may be returned
   * @param enable true if auto child versioning is to be enabled
   * @return a Set of IResourceHandles of all resources modified by the
   *      operation (if returnModified is true)
   * @exception ResourceException if the resource handle is invalid or the
   *      operation failed
   */
  public Set setAutoChildVersionControlEnabled(IResourceHandle collectionHandle, boolean modifyPlainChildren, boolean returnModified, boolean enable)
    throws ResourceException;

}
