/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.mi.version;
import com.sap.netweaver.bc.rf.common.*;

import com.sap.netweaver.bc.rf.common.exception.*;
import com.sap.netweaver.bc.rf.util.namespace.*;

import java.util.*;

//TODO mba: insert @link tags where applicable.
//TODO mba: specify ResourceException subtypes on methods that throw specific exceptions

/**
 * Read-Only interface describing a repository sub manager for basic versioning
 * features. Essentially, these features offer the possibility to take a
 * non-collection resource under version control with a linear version history
 * (branches are considered an advanced feature). Additional features offered
 * are autoversioning (meaning that a modification of a version controlled
 * resource may automatically generate new versions) and auto version control
 * (formerly referred to as 'enhanced collection feature', meaning that newly
 * created non-collection resources are automatically taken under version
 * control). <p>
 *
 * Throughout the documentation of this class, the following terminological
 * conventions are used: <p>
 *
 * the term 'vcr' is used instead of 'version controlled resource', meaning a
 * resource with an associated set of version resources (the version history).
 * The state of the vcr that is stored in it's version resource is called the
 * 'version controlled state' of the resource. At least, the version controlled
 * state contains the content and the dead properties of the vcr. <p>
 *
 * A vcr is always in exactly one of two possible states, namely 'checked-in'
 * and 'checked-out'. If the resource is checked-in, all operations are
 * forbidden that would change the version controlled state (but be aware of the
 * autoversioning feature, which may force implicit check-outs). <p>
 *
 * When a vcr changes it's state from checked-in to checked-out, a version
 * resource is associated with this state change, describing the current version
 * controlled state of the vcr and called the 'checked-out version' of the
 * checked-out vcr. <p>
 *
 * Similarly, a checked-in vcr is associated with a version resource, describing
 * it's version controlled state and called the 'checked-in version' of the
 * checked-in vcr. <p>
 *
 * The checked-in version (resp. checked-out version) of the vcr may be thought
 * of as the 'current' version of the vcr (but this terminology is avoided for
 * compatability with RFC3253 terminology). <p>
 *
 * Copyright (c) SAP AG 2003
 *
 * @author manfred.baedke@greenbytes.de
 * @created 20. Mrz 2003
 */
public interface IBasicVersioningManager {

  /**
   * Returns true if the given resource handle refers to a vcr.
   *
   * @param resourceHandle a resource handle
   * @return true if the given resource handle refers to a vcr.
   * @exception ResourceException if the resource handle is invalid or the
   *      operation failed
   */
  public boolean isVersionControlEnabled(IResourceHandle resourceHandle)
    throws ResourceException;

  //-------------------------------------version history methods-------------------------------------------------------

  /**
   * If the given resource handle refers to a vcr with a linear version history,
   * this version history is returned as a list of resource handles referring to
   * version resources (note that the last version in the list is the newest).
   *
   * @param resourceHandle a resource handle
   * @return list of all versions of this resource handle
   * @exception ResourceException if the resource handle is invalid or the
   *      operation failed
   */
  public List getVersionHistory(IResourceHandle resourceHandle)
    throws ResourceException;


  /**
   * If the given resource handle refers to a version resource, the method will
   * return the set of resource handles which refer to immediate predecessors of
   * that version in it's version history. The return type is <code>Set</code>
   * to ensure compatability with more advanced versioning managers that may
   * allow non-linear version histories. <p>
   *
   * If the given resource handle refers to a checked-out vcr, the above
   * semantics is applied to the vcr's checked-out version.
   *
   * @param resourceHandle a resource handle
   * @return set of immediate predecessor resource handles (may be empty)
   * @exception ResourceException if the resource handle is invalid or the
   *      operation failed
   */
  public Set getPredecessorVersions(IResourceHandle resourceHandle)
    throws ResourceException;


  /**
   * If the given resource handle refers to a version resource, the method will
   * return the set of resource handles which refer to immediate successors of
   * that version in it's version history. The return type is <code>Set</code>
   * to ensure compatability with more advanced versioning managers that may
   * allow non-linear version histories. <p>
   *
   * If the given resource handle refers to a checked-out vcr, the above
   * semantics is applied to the vcr's checked-out version.
   *
   * @param resourceHandle a resource handle
   * @return set of immediate successor resource handles (may be empty)
   * @exception ResourceException if the resource handle is invalid or the
   *      operation failed
   */
  public Set getSuccessorVersions(IResourceHandle resourceHandle)
    throws ResourceException;


  //-------------------------------------checkin/checkout related methods----------------------------------------------

  /**
   * Returns true if the given resource handle refers to a checked-out vcr.
   *
   * @param vcrHandle a resource handle referring to a vcr
   * @return true if the given resource handle refers to a checked-out vcr
   * @exception ResourceException if the resource handle is invalid or the
   *      operation failed
   */
  public boolean isCheckedOut(IResourceHandle vcrHandle)
    throws ResourceException;


  /**
   * Returns the version resource handle from which the given (checked-out) vcr
   * has been checked out.
   *
   * @param vcrHandle a resource handle referring to a vcr
   * @return the resource handle of the checked-out version of the given vcr
   *      (may be null)
   * @exception ResourceException if the resource handle is invalid or the
   *      operation failed
   */
  public IResourceHandle getCheckedOutVersion(IResourceHandle vcrHandle)
    throws ResourceException;


  /**
   * Returns the version resource the given (checked-in) vcr is based on.
   *
   * @param vcrHandle a resource handle referring to a vcr
   * @return the resource handle of the checked-in version of the given vcr (may
   *      be null)
   * @exception ResourceException if the resource handle is invalid or the
   *      operation failed
   */
  public IResourceHandle getCheckedInVersion(IResourceHandle vcrHandle)
    throws ResourceException;

  //-------------------------------------child auto version control ('enhanced collection') methods--------------------

  /**
   * Returns true if auto version control of children is enabled for the
   * collection resource referred by the given collection handle, which means
   * that newly created direct non-collection children will automatically put
   * under version control.
   *
   * @param collectionHandle a resource handle referring to a collection
   * @return true if auto version control of children is enabled
   * @exception ResourceException if the resource handle is invalid or the
   *      operation failed
   */
  public boolean isAutoChildVersionControlEnabled(IResourceHandle collectionHandle)
    throws ResourceException;

  //-------------------------------------autoversioning methods--------------------------------------------------------

  /**
   * Returns the autoversioning mode of the vcr referred by the given resource
   * handle. The autoversioning mode describes the way the server will handle
   * implicit checkout/checkin of vcrs prior/subsequent to operations which are
   * about to change the version controlled state of the vcr. <p>
   *
   * See <code>IVersionControlledResource</code> for possible modes.
   *
   * @param versionedResourceHandle a resource handle referring to a vcr
   * @return IName of current autoversioning mode (may be null)
   * @exception ResourceException if the resource handle is invalid or the
   *      operation failed
   */
  public IName getAutoVersioningMode(IResourceHandle versionedResourceHandle)
    throws ResourceException;
}
