/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.mi.version;
import com.sap.netweaver.bc.rf.common.*;

import com.sap.netweaver.bc.rf.common.exception.*;

import java.util.*;

//TODO mba: insert @link, @see etc. tags where applicable.
//TODO mba: specify ResourceException subtypes on methods that throw specific exceptions

/**
 * Read-Only interface describing a repository sub manager for the advanced
 * versioning features described below. For an explanation of terminological
 * conventions, please see IBasicVersioningMananger. In addition to the very
 * basic versioning features of <code>IBasicVersionManager</code> , this manager
 * adds support for nonlinear version histories (i.e. forks), the option of
 * sharing version histories between vcrs (i.e. working resources), the option
 * of addressing versions in a version history by a client-defined string (i.e.
 * labels) and the version history resource feature, meaning that the version
 * history is accessible as a resource (in particular, giving it Properties and
 * the delete() method).
 *
 * @author manfred.baedke@greenbytes.de
 * @created 20. Mrz 2003
 */
public interface IAdvancedVersioningManager {

  //-------------------------------------version history methods-------------------------------------------------------

  //JRE: add reference to RFC3253, section 5.2.4
  /**
   * Returns a set of resource handles referring to the collections that are
   * used to store version history resources for the given resource handle. The
   * collections can be at the top of a whole tree in which version histories
   * are stored. Several collections with version histories can be represented
   * by their common ancestor collection.
   *
   * @param vcrHandle a resource handle referring to a vcr
   * @return a set of resource handles of collections containing version
   *      histories
   * @exception ResourceException if the resource handle is invalid or the
   *      operation failed
   */
  public Set getVersionHistoryCollections(IResourceHandle vcrHandle)
    throws ResourceException;


  /**
   * Returns a resource handle referring to the version history resource of the
   * given vcr or version.
   *
   * @param resourceHandle a resource handle referring to a version resource or
   *      a vcr
   * @return a resource handle referring to a version history resource
   * @exception ResourceException if the resource handle is invalid or the
   *      operation failed
   */
  public IResourceHandle getVersionHistoryResource(IResourceHandle resourceHandle)
    throws ResourceException;


  /**
   * Returns a resource handle referring to the version resource that is the
   * root of the given version history (i.e. the first version in the version
   * history).
   *
   * @param versionHistoryHandle a resource handle referring to a version
   *      history resource
   * @return a resource handle referring root version of the given version
   *      history
   * @exception ResourceException if the resource handle is invalid or the
   *      operation failed
   */
  public IResourceHandle getRootVersion(IResourceHandle versionHistoryHandle)
    throws ResourceException;


  /**
   * If the given resource handle refers to a version history resource, a list
   * of resource handles referring to vcrs based on this version history will be
   * returned. If a scope is specified, the result will contain only resource
   * handles which are descendants of the collection referred by the given scope
   * resource handle.
   *
   * @param scope the scope of the request (may be null)
   * @param versionHistoryHandle a resource handle referring to a version
   *      history
   * @return a set of resource handles referring to vcrs based on the given
   *      version history
   * @exception ResourceException if the resource handle is invalid or the
   *      operation failed
   */
  public Set getVCRsBasedOnHistory(IResourceHandle scope, IResourceHandle versionHistoryHandle)
    throws ResourceException;

  //-------------------------------------checkin/checkout related methods----------------------------------------------

  /**
   * If the given resource handle refers to a version resource, the set of
   * resource handles referring to checked-out vcrs based on this version will
   * be returned.
   *
   * @param versionHandle a resource handle referring to a version resource
   * @return a set of resource handles of checked-out vcrs' based on the given
   *      version resource
   * @exception ResourceException if the resource handle is invalid or the
   *      operation failed
   */
  public Set getCheckedOutVCRsByVersion(IResourceHandle versionHandle)
    throws ResourceException;


  /**
   * Returns a set of <code>IName</code> objects describing the fork behavior of
   * the given version resource on checkin. The return type is <code>Set</code>
   * to ensure compatability with future extension of the DeltaV protocol. At
   * the moment, there are three possibilities: forks may be explicitly allowed,
   * they may be strictly forbidden or they may be discouraged, the latter
   * meaning that the value of the forkOk parameter of the method <code>
   * IBasicVersioningManager.checkin()</code> will decide in every case wether
   * forks are allowed or not.
   *
   * @param versionHandle a resource handle referring to a version resource
   * @return a set of <code>IName</code> objects describing the checkin-fork
   *      behaviour of the given version
   * @exception ResourceException if the resource handle is invalid or the
   *      operation failed
   * @see IVersionResource#getCheckInForkBehaviour
   */
  public Set getCheckInForkBehaviour(IResourceHandle versionHandle)
    throws ResourceException;


  /**
   * Returns a set of <code>IName</code> objects describing the fork behavior of
   * the given version resource on checkout. The return type is <code>Set</code>
   * to ensure compatability with future extension of the DeltaV protocol. At
   * the moment, there are three possibilities: forks may be explicitly allowed,
   * they may be strictly forbidden or they may be discouraged, the latter
   * meaning that the value of the forkOk parameter of the method <code>
   * IBasicVersioningManager.checkout()</code> will decide in every case wether
   * forks are allowed or not.
   *
   * @param versionHandle a resource handle referring to a version resource
   * @return a set of <code>IName</code> objects describing the checkout-fork
   *      behaviour of the given version
   * @exception ResourceException if the resource handle is invalid or the
   *      operation failed
   * @see IVersionResource#getCheckOutForkBehaviour
   */
  public Set getCheckOutForkBehaviour(IResourceHandle versionHandle)
    throws ResourceException;


  /**
   * Returns a resource handle referring to the version controlled resource from
   * which the given working resource was created and which will automatically
   * be updated when the working resource is checked in. A working resource is a
   * vcr which is created from a given version resource or vcr by a call to
   * <code> IMutableAdvancedVersioningManager.checkoutWorkingResource()</code> ,
   * thus sharing it's version history with other vcrs. For further explanation,
   * see <code> IMutableAdvancedVersioningManager.checkoutWorkingResource()
   * </code>.
   *
   * @param workingResourceHandle a resource handle referring to a working
   *      resource.
   * @return the resource handle of the vcr which will be updated when the given
   *      working resource is checked in (may be null)
   * @exception ResourceException if the resource handle is invalid or the
   *      operation failed
   */
  public IResourceHandle getAutoUpdateVcr(IResourceHandle workingResourceHandle)
    throws ResourceException;


  //-------------------------------------label methods-----------------------------------------------------------------

  /**
   * Returns a <code>Set</code> of strings representing the labels defined on
   * the given version resource. A label is an client-defined identifier of a
   * version which in unique in it's version history.
   *
   * @param versionHandle a resource handle referring to a version resource
   * @return <code>Set</code> of <code>String</code> objects representing the
   *      labels defined on the given version resource
   * @exception ResourceException if the resource handle is invalid or the
   *      operation failed
   */
  public Set getLabelSet(IResourceHandle versionHandle)
    throws ResourceException;


  /**
   * Returns the resource handle of the version resource which is contained in
   * the given version history and which is tagged by the given label.
   *
   * @param label a version label
   * @param versionHistoryHandle a resource handle referring to a version
   *      history
   * @return a resource handle referring to the version with the given label in
   *      the given version history (may be null)
   * @exception ResourceException if the resource handle is invalid or the
   *      operation failed
   */
  public IResourceHandle getVersionResourceByLabel(IResourceHandle versionHistoryHandle, String label)
    throws ResourceException;
}
