/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.mi.type;
import com.sap.netweaver.bc.rf.common.*;

import com.sap.netweaver.bc.rf.common.exception.*;
import com.sap.netweaver.bc.rf.common.namespace.*;

/**
 * Read-And-Write interface describing a repository sub manager to create,
 * update or delete general objects known to the implementing repository (i.e.
 * it's known how to serialize it in order to store it) without handling the
 * resources directly, i.e. just create, update or delete a resource handling
 * only the "casted" ({@link ITypeManager}) default object.
 *
 * @created 28. Januar 2003
 * @see ITypeManager
 */
public interface IMutableTypeManager extends ITypeManager {
  /**
   * Create resource based on given resource object known to the implementing
   * repository (i.e. it's known how to identify and serialize it in order to
   * create it) and the given creation descriptor for the resource. The resource
   * object will be identified and serialized, a resource for it will be created
   * and the serialized resource object will be stored therein, obeying the
   * creation descriptor for the resource.
   *
   * @param resourceObject resource object
   * @param resourceCreateDescriptor creation descriptor for the resource
   * @return resource handle of the created resource
   * @exception ResourceException when either the resource object is null, the
   *      creation descriptor is either null or invalid or the
   *      identification/serialization/creation failed
   */

  public IResourceHandle createResource(
    Object resourceObject,
    IResourceCreateDescriptor resourceCreateDescriptor)
    throws ResourceException;

  /**
   * Update resource based on given resource object known to the implementing
   * repository (i.e. it's known how to identify and serialize it in order to
   * update it) and the given update descriptor for the resource. The resource
   * object will be identified and serialized, the resource for it will be
   * updated by storing the serialized resource object therein, obeying the
   * update descriptor for the resource.
   *
   * @param resourceObject resource object
   * @param resourceCreateDescriptor update descriptor for the resource
   * @return resource handle of the updated resource
   * @exception ResourceException when either the resource object is null, the
   *      update descriptor is either null or invalid or the
   *      identification/serialization/update failed
   */
  public IResourceHandle updateResource(
    Object resourceObject,
    IResourceCreateDescriptor resourceCreateDescriptor)
    throws ResourceException;

  /**
   * Delete resource based on given resource object known to the implementing
   * repository (i.e. it's known how to identify it in order to delete it). The
   * resource object will be identified and deleted.
   *
   * @param resourceObject resource object
   * @exception ResourceException when either the resource object is null or the
   *      identification/deletion failed
   */
  public void deleteResource(Object resourceObject)
    throws ResourceException;
}
