/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.mi.security.acl;
import com.sap.netweaver.bc.rf.common.*;

import com.sap.netweaver.bc.rf.common.exception.*;
import com.sap.netweaver.bc.rf.common.security.acl.*;
import com.sap.netweaver.bc.rf.mi.security.*;

import com.sap.security.api.*;

import java.util.*;

/**
 * Interface describing an access control list security sub manager, short ACL
 * security sub manager. Each resource may have an ACL with many ACL entries. An
 * entry describes for one principal (may be a user, group or whatever is
 * considered an entity) a permission this principal has on the given resource.
 * The ACL itself is just an envelope for many ACL entries. </p> An ACL entry
 * may define a permission this principal has on the given resource, but it may
 * also define a permission a principal is denied to have (so called negative
 * ACL entry). </p> ACLs are propagated down from above the resource hierarchy
 * on the fly, i.e. they are not really set on the descendants but derived when
 * accessed. All descendants not having an own ACL inherit one from their next
 * legator having one. Legators are ancestors responsible for the security
 * permissions of their descendants. </p> ACLs and ACL entries are factored by
 * this security sub manager and thereby assigned to a resource handle (exactly
 * one ACL per resource handle and null to more ACL entries per ACL). </p> Each
 * resource handle has to be of a certain object type known by the ACL security
 * sub manager and each object type has special supported permissions. Only
 * supported permissions for a given resource handle can be set, others not.
 * This helps identifying, e.g. in GUI, which permissions are allowed to be set
 * and worked on.
 *
 * @author Vedran Lerenc
 * @created 30. Januar 2003
 */
public interface IAclSecurityManager extends ISecurityManager {
  /**
   * Get ACL of a given resource handle. Inherited ACLs are not returned. Please
   * call getInheritedAcl() for that purpose.
   *
   * @param resourceHandle resource handle
   * @return ACL of a given resource handle
   * @exception AclPersistenceException when a problem with the storage where
   *      the ACLs are persisted occurred
   * @exception ResourceException when the resource handle was invalid
   */
  public IAcl getAcl(IResourceHandle resourceHandle)
    throws AclPersistenceException, ResourceException;

  /**
   * Get ACL of a given resource handle or an inherited ACL if none is defined
   * directly for the given resource handle. A call to this method may involve
   * calling all legators until an ACL is found.
   *
   * @param resourceHandle resource handle
   * @return the inherited ACL
   * @exception AclPersistenceException when a problem with the storage where
   *      the ACLs are persisted occurred
   * @exception ResourceException when the resource handle was invalid
   */
  public IAcl getInheritedAcl(IResourceHandle resourceHandle)
    throws AclPersistenceException, ResourceException;

  /**
   * Create a new ACL entry to grant or deny a permission to a principal. The
   * ACL entry must be assigned to an ACL in order to be activated.
   *
   * @param principal principal
   * @param permission permission
   * @param negative true if the entry denies a permission, false if it grants
   *      one
   * @param readonly true if the entry is read only, false otherwise
   * @param sortIndex position of the ACL entry in the ACL (important only with
   *      denials)
   * @return ACL entry
   * @exception AclPersistenceException when a problem with the storage where
   *      the ACLs are persisted occurred
   * @exception UnsupportedOperationException when a negative ACL entry is
   *      requested but not supported
   */
  public IAclEntry createAclEntry(IPrincipal principal, IAclPermission permission, boolean negative, boolean readonly, int sortIndex)
    throws AclPersistenceException, UnsupportedOperationException;

  /**
   * Create a new ACL for a resource handle when the resource handle has no ACL
   * and the current user in the resource context is an owner of the inherited
   * ACL. This user will become the initial owner of the newly created ACL. All
   * ACL entries from the inherited ACL will be copied when applicable to the
   * object type of the given resource handle; all others will be ignored.
   *
   * @param resourceHandle resource handle
   * @return ACL
   * @exception AclPersistenceException when a problem with the storage where
   *      the ACLs are persisted occurred
   * @exception AclExistsException when an ACL already exists for the resource
   *      handle
   * @exception ResourceException when the resource handle was invalid
   * @exception AclNotAuthorizedException Exception raised in failure situation
   */
  public IAcl createAcl(IResourceHandle resourceHandle)
    throws AclPersistenceException, AclNotAuthorizedException, AclExistsException, ResourceException;

  /**
   * Assign an existent ACL, i.e. copy and assign it to a resource handle when
   * the resource handle has no ACL and the current user in the resource context
   * is an owner of the source ACL. All users of the source ACL will become the
   * initial owners of the newly copied and assigned ACL. All ACL entries from
   * the source ACL will be copied when applicable to the object type and ACL
   * security sub manager implementation of the given resource handle; all
   * others will be ignored.
   *
   * @param resourceHandle resource handle
   * @param acl ACL
   * @exception AclPersistenceException when a problem with the storage where
   *      the ACLs are persisted occurred
   * @exception AclExistsException when an ACL already exists for the resource
   *      handle
   * @exception ResourceException when the resource handle was invalid
   * @exception AclNotAuthorizedException Exception raised in failure situation
   */
  public void assignAcl(IResourceHandle resourceHandle, IAcl acl)
    throws AclPersistenceException, AclNotAuthorizedException, AclExistsException, ResourceException;

  /**
   * Remove the ACL from the given resource.
   *
   * @param resourceHandle resource handle
   * @exception AclPersistenceException when a problem with the storage where
   *      the ACLs are persisted occurred
   * @exception ResourceException when the resource handle was invalid
   * @exception AclNotAuthorizedException Exception raised in failure situation
   */
  public void removeAcl(IResourceHandle resourceHandle)
    throws AclPersistenceException, AclNotAuthorizedException, ResourceException;

  /**
   * Remove the given ACL from its resource.
   *
   * @param acl ACL
   * @exception AclPersistenceException when a problem with the storage where
   *      the ACLs are persisted occurred
   * @exception ResourceException when the resource handle assigned to the ACL
   *      was invalid
   * @exception AclNotAuthorizedException Exception raised in failure situation
   */
  public void removeAcl(IAcl acl)
    throws AclPersistenceException, AclNotAuthorizedException, ResourceException;


  /**
   * Identify and get permission with the given name generally known to this ACL
   * security sub manager.
   *
   * @param name permission name
   * @return permission with the given name
   * @exception AclPersistenceException when a problem with the storage where
   *      the ACLs are persisted occurred
   */
  public IAclPermission getPermission(String name)
    throws AclPersistenceException;

  /**
   * Register permission with the given name generally for this ACL security sub
   * manager.
   *
   * @param name permission name
   * @return newly created permission with the given name
   * @exception AclPersistenceException when a problem with the storage where
   *      the ACLs are persisted occurred
   * @exception AclPermissionExistsException Exception raised in failure
   *      situation
   */
  public IAclPermission createPermission(String name)
    throws AclPersistenceException, AclPermissionExistsException;

  /**
   * Unregister permission with the given name generally from this ACL security
   * sub manager. It can only be unregistered if it is not predefined and not
   * used in an ACL.
   *
   * @param permission the permission
   * @exception AclPersistenceException when a problem with the storage where
   *      the ACLs are persisted occurred
   * @exception AclPredefinedPermissionException Exception raised in failure
   *      situation
   * @exception AclPermissionUsedException Exception raised in failure situation
   */
  public void removePermission(IAclPermission permission)
    throws AclPersistenceException, AclPredefinedPermissionException, AclPermissionUsedException;

  /**
   * Get list of object types supported by this ACL security sub manager.
   *
   * @return list of the {@link IObjectType}s
   * @exception AclPersistenceException when a problem with the storage where
   *      the ACLs are persisted occurred
   */
  public List getSupportedObjectTypes()
    throws AclPersistenceException;

  /**
   * Get object type this ACL security sub manager interprets the given resource
   * handle.
   *
   * @param resourceHandle resource handle
   * @return {@link IObjectType}
   * @exception AclPersistenceException when a problem with the storage where
   *      the ACLs are persisted occurred
   * @exception ResourceException when the resource handle was invalid
   */
  public IObjectType getObjectType(IResourceHandle resourceHandle)
    throws AclPersistenceException, ResourceException;

  /**
   * Get list of permissions which are supported by this ACL security sub
   * manager for the given resource handle, i.e. outcome depends on the object
   * type of the given resource handle, e.g. depends whether the resource is a
   * collection or not.
   *
   * @param resourceHandle resource handle
   * @return list of supported permissions of type {@link IAclPermission}
   * @exception ResourceException when the resource handle was invalid
   */
  public List getSupportedPermissions(IResourceHandle resourceHandle)
    throws ResourceException;

  /**
   * Register permission as supported permission for an object type.
   *
   * @param objectType object type
   * @param permission permission
   * @exception AclPersistenceException when a problem with the storage where
   *      the ACLs are persisted occurred
   * @exception AclPredefinedPermissionException Exception raised in failure
   *      situation
   */
  public void addSupportedPermission(IObjectType objectType, IAclPermission permission)
    throws AclPersistenceException, AclPredefinedPermissionException;

  /**
   * Unregister permission as no longer supported permission from an object
   * type. It can only be unregistered if it is not predefined and not used in
   * an ACL.
   *
   * @param objectType object type
   * @param permission permission
   * @exception AclPersistenceException when a problem with the storage where
   *      the ACLs are persisted occurred
   * @exception AclPredefinedPermissionException Exception raised in failure
   *      situation
   * @exception AclPermissionUsedException Exception raised in failure situation
   */
  public void removeSupportedPermission(IObjectType objectType, IAclPermission permission)
    throws AclPersistenceException, AclPredefinedPermissionException, AclPermissionUsedException;

  /**
   * Check whether or not a permission is used in an ACL.
   *
   * @param permission permission
   * @return true if the permission is used in an ACL
   * @exception AclPersistenceException when a problem with the storage where
   *      the ACLs are persisted occurred
   */
  public boolean isPermissionUsedInAcl(IAclPermission permission)
    throws AclPersistenceException;

  /**
   * Check whether or not negative ACL entries are supported by this ACL
   * security sub manager.
   *
   * @return true if negative ACL entries are supported by this ACL security sub
   *      manager
   */
  public boolean supportsNegativeAclEntries();

  /**
   * Check whether or not this ACL security sub manager is read only and no
   * creation or modification operations of ACLs are allowed.
   *
   * @return true if this ACL security sub manager is read only
   * @exception AclPersistenceException when a problem with the storage where
   *      the ACLs are persisted occurred
   */
  public boolean isReadOnly()
    throws AclPersistenceException;
}
