/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.mi.security;
import com.sap.netweaver.bc.rf.common.*;

import com.sap.netweaver.bc.rf.common.exception.*;
import com.sap.netweaver.bc.rf.common.security.*;

import com.sap.security.api.*;

import java.util.*;

/**
 * Read-Only interface describing a repository sub manager for security. This
 * interface allows for simple checking of permission on resources, but doesn't
 * offer any sophisticated operations, because security APIs are usually very
 * much specialized and can't be defined in an general interface for a
 * repository sub manager. Specifically read access more sophisticated than the
 * isAllowed() methods is problematic on respoitories supporting also denies (a
 * grant on a group and a deny on a user must be translated into only grants). A
 * more complex abstraction of security settings including denies and other
 * features would impose too much of a requirement for repositories. Not all
 * repositories might be able to support such a complex API even in parts.
 *
 * @author Vedran Lerenc
 * @created 23. Januar 2003
 */
public interface ISecurityManager {
  /**
   * Get list of permissions which are supported by the security manager for the
   * given resource handle, e.g. outcome depends on whether the resource is a
   * collection or not.
   *
   * @param resourceHandle resource handle
   * @return list of supported permissions of type {@link IPermission}
   * @exception ResourceException when the resource handle is invalid or no
   *      permissions could be looked up
   */
  public List getSupportedPermissions(IResourceHandle resourceHandle)
    throws ResourceException;


  /**
   * Checks whether the given principal has the given permission on the given
   * resource handle.
   *
   * @param resourceHandle resource handle
   * @param principal principal for whom the permissions are checked
   * @param permission single permission to check
   * @return true when the given principal has the given permission on the given
   *      resource handle
   * @exception ResourceException when wrong parameterized
   */
  public boolean isAllowed(
    IResourceHandle resourceHandle,
    IPrincipal principal,
    IPermission permission)
    throws ResourceException;


  /**
   * Checks whether the given principal has all of the given permissions on the
   * given resource handle.
   *
   * @param resourceHandle resource handle
   * @param principal principal for whom the permissions are checked
   * @param permissions list of permissions to check
   * @return true when the given principal has all of the given permissions on
   *      the given resource handle
   * @exception ResourceException when wrong parameterized
   */
  public boolean isAllowed(IResourceHandle resourceHandle, IPrincipal principal, List permissions)
    throws ResourceException;


  /**
   * Checks whether the given principal has the given permission on the given
   * resource identifiers. </p> Note: This method needs not to be handled
   * atomically! Although this operation may fail in parts, the operation on one
   * of the many operands provided through the parameter list is always expected
   * to be atomic!
   *
   * @param ridList list of resource identifiers
   * @param principal principal for whom the permissions are checked
   * @param permission single permission to check
   * @return list of resource identifiers the given principal has the given
   *      permission for
   * @exception ResourceException when wrong parameterized
   * @exception OperationNotCompletedException when the operation failed in
   *      parts
   */
  public Set isAllowed(List ridList,
    IPrincipal principal,
    IPermission permission)
    throws ResourceException, OperationNotCompletedException;


  /**
   * Checks whether the given principal has all of the given permissions on the
   * given resource identifiers. </p> Note: This method needs not to be handled
   * atomically! Although this operation may fail in parts, the operation on one
   * of the many operands provided through the parameter list is always expected
   * to be atomic!
   *
   * @param ridList list of resource identifiers
   * @param principal principal for whom the permissions are checked
   * @param permissions list of permissions to check
   * @return set of resource identifiers the given principal has all of the
   *      given permissions for
   * @exception ResourceException when wrong parameterized
   * @exception OperationNotCompletedException when the operation failed in
   *      parts
   */
  public Set isAllowed(List ridList,
    IPrincipal principal,
    List permissions)
    throws ResourceException, OperationNotCompletedException;
}
