/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.mi.security;
import com.sap.netweaver.bc.rf.common.*;

import com.sap.netweaver.bc.rf.common.exception.*;
import com.sap.netweaver.bc.rf.common.security.*;

import com.sap.security.api.*;

import java.util.*;

/**
 * Read-And-Write interface describing a repository sub manager for security.
 * This interface allows for simple setting and removing of permissions on
 * resources, but doesn't offer any sophisticated operations, denies or
 * orderings, because security APIs are usually very much specialized and can't
 * be defined in an general interface for a repository sub manager. Specifically
 * denies would be way to sophisticated including all the problems of read
 * access and ordering. A more complex abstraction of security settings
 * including denies and other features would impose too much of a requirement
 * for repositories. Not all repositories might be able to support such a
 * complex API even in parts.
 *
 * @author Vedran Lerenc
 * @created 23. Januar 2003
 */
public interface IMutableSecurityManager extends ISecurityManager {
  /**
   * Set permission for the given principal on the given resource handle.
   *
   * @param resourceHandle resource handle
   * @param principal principal for whom the permission should be set
   * @param permission single permission to set
   * @exception ResourceException when wrong parameterized or setting the
   *      permission failed
   */
  public void grant(IResourceHandle resourceHandle,
    IPrincipal principal,
    IPermission permission)
    throws ResourceException;

  /**
   * Set list of permissions for the given principal on the given resource
   * handle.
   *
   * @param resourceHandle resource handle
   * @param principal principal for whom the permissions should be set
   * @param permissions list of permissions to set
   * @exception ResourceException when wrong parameterized or setting the
   *      permission failed
   */
  public void grant(IResourceHandle resourceHandle,
    IPrincipal principal,
    List permissions)
    throws ResourceException;

  /**
   * Set permission for the given principal on the given resource identifiers.
   * </p> Note: This method needs not to be handled atomically! Although this
   * operation may fail in parts, the operation on one of the many operands
   * provided through the parameter list is always expected to be atomic!
   *
   * @param ridList list of resource identifiers
   * @param principal principal for whom the permissions should be set
   * @param permission single permission to set
   * @exception ResourceException when wrong parameterized
   * @exception OperationNotCompletedException when the operation failed in
   *      parts
   */
  public void grant(List ridList,
    IPrincipal principal,
    IPermission permission)
    throws ResourceException, OperationNotCompletedException;

  /**
   * Set list of permissions for the given principal on the given resource
   * identifiers. </p> Note: This method needs not to be handled atomically!
   * Although this operation may fail in parts, the operation on one of the many
   * operands provided through the parameter list is always expected to be
   * atomic!
   *
   * @param ridList list of resource identifiers
   * @param principal principal for whom the permissions should be set
   * @param permissions list of permissions to set
   * @exception ResourceException when wrong parameterized
   * @exception OperationNotCompletedException when the operation failed in
   *      parts
   */
  public void grant(List ridList,
    IPrincipal principal,
    List permissions)
    throws ResourceException, OperationNotCompletedException;

  /**
   * Remove permission for the given principal on the given resource handle.
   *
   * @param resourceHandle resource handle
   * @param principal principal for whom the permission should be removed
   * @param permission single permission to remove
   * @exception ResourceException when wrong parameterized or removing the
   *      permission failed
   */
  public void revoke(IResourceHandle resourceHandle,
    IPrincipal principal,
    IPermission permission)
    throws ResourceException;

  /**
   * Remove list of permissions for the given principal on the given resource
   * handle.
   *
   * @param resourceHandle resource handle
   * @param principal principal for whom the permissions should be removed
   * @param permissions list of permissions to remove
   * @exception ResourceException when wrong parameterized or removing the
   *      permission failed
   */
  public void revoke(IResourceHandle resourceHandle,
    IPrincipal principal,
    List permissions)
    throws ResourceException;

  /**
   * Remove permission for the given principal on the given resource
   * identifiers. </p> Note: This method needs not to be handled atomically!
   * Although this operation may fail in parts, the operation on one of the many
   * operands provided through the parameter list is always expected to be
   * atomic!
   *
   * @param ridList list of resource identifiers
   * @param principal principal for whom the permissions should be removed
   * @param permission single permission to remove
   * @exception ResourceException when wrong parameterized
   * @exception OperationNotCompletedException when the operation failed in
   *      parts
   */
  public void revoke(List ridList,
    IPrincipal principal,
    IPermission permission)
    throws ResourceException, OperationNotCompletedException;

  /**
   * Remove list of permissions for the given principal on the given resource
   * identifiers. </p> Note: This method needs not to be handled atomically!
   * Although this operation may fail in parts, the operation on one of the many
   * operands provided through the parameter list is always expected to be
   * atomic!
   *
   * @param ridList list of resource identifiers
   * @param principal principal for whom the permissions should be removed
   * @param permissions list of permissions to remove
   * @exception ResourceException when wrong parameterized
   * @exception OperationNotCompletedException when the operation failed in
   *      parts
   */
  public void revoke(List ridList,
    IPrincipal principal,
    List permissions)
    throws ResourceException, OperationNotCompletedException;
}
