/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.mi.property;
import com.sap.netweaver.bc.rf.common.*;

import com.sap.netweaver.bc.rf.common.exception.*;
import com.sap.netweaver.bc.rf.common.property.*;

import java.util.*;

/**
 * Read-Only interface describing a repository sub manager for properties.
 * Resources usually have properties describing the resource. All system
 * properties MUST be made available as properties. Resources with content MUST
 * make the content metadata available as properties also! This interface
 * provides means to read the properties of a resource. </p> Most properties are
 * not required, but when missing, certain behaviour is assumed: </p>
 * RESOURCENAME is the last RID part if not existent (and must be the last RID
 * part if set)</br> COLLECTION is assumed to be false, i.e. a simple resource
 * if not existent (and must be true for collections and false for resources)
 * </br> LINKTYPE is assumed to be none, i.e. no link of any kind if not
 * existent</br> DISPLAYNAME is assumed to be the resource name if not existent
 * </br>
 *
 * @created 23. Januar 2003
 */
public interface IPropertyManager {
  /**
   * Get single property of a single resource handle. Returns null if the
   * property is not set.
   *
   * @param resourceHandle resource handle
   * @param propertyName property name
   * @return property
   * @exception ResourceException when either the resource handle or the
   *      property name is invalid or the property retrieval failed
   */
  public IProperty getProperty(
    IResourceHandle resourceHandle,
    IPropertyName propertyName)
    throws ResourceException;

  /**
   * Get list of properties of a single resource handle. Not found properties
   * are ignored and are not stored inside the returned data structure (not even
   * with a null value). Returns an empty map if no property is set.
   *
   * @param resourceHandle resource handle
   * @param propertyNames list of property names
   * @return map of properties indexed by property names
   * @exception ResourceException when either the resource handle or the
   *      property names are invalid or the property retrieval failed
   */
  public Map getListedProperties(IResourceHandle resourceHandle,
    List propertyNames)
    throws ResourceException;

  /**
   * Get map of all properties of a single resource handle. Returns an empty map
   * if no property is available.
   *
   * @param resourceHandle resource handle
   * @return map of properties indexed by property names
   * @exception ResourceException when either the resource handle is invalid or
   *      the property retrieval failed
   */
  public Map getAllProperties(IResourceHandle resourceHandle)
    throws ResourceException;

  /**
   * Get single property of a list of resource handles. Resources without the
   * property are ignored and are not stored inside the returned data structure
   * (not even with a null value). Returns an empty map if the property is not
   * set for all resources. </p> Note: This method needs not to be handled
   * atomically! Although this operation may fail in parts, the operation on one
   * of the many operands provided through the parameter list is always expected
   * to be atomic!
   *
   * @param resourceHandles list of resource handles
   * @param propertyName property name
   * @return map of properties indexed by the resource handles
   * @exception ResourceException when a general problem occured
   * @exception OperationNotCompletedException when the operation failed in
   *      parts (either a resource handle or a property name was invalid or a
   *      property retrieval failed)
   */
  public Map getProperties(
    List resourceHandles,
    IPropertyName propertyName)
    throws ResourceException, OperationNotCompletedException;

  /**
   * Get list of properties of a list of resource handles. Not found properties
   * are ignored and are not stored inside the returned data structure (not even
   * with a null value). If no properties are found for a resource, the whole
   * resource isn't stored inside the returned data structure too. Returns an
   * empty map if no properties are set for all resources. </p> Note: This
   * method needs not to be handled atomically! Although this operation may fail
   * in parts, the operation on one of the many operands provided through the
   * parameter list is always expected to be atomic!
   *
   * @param resourceHandles list of resource handles
   * @param propertyNames list of property names
   * @return map of maps of properties indexed by resource handles; the inner
   *      maps hold properties indexed by property names
   * @exception ResourceException when a general problem occured
   * @exception OperationNotCompletedException when the operation failed in
   *      parts (either a resource handle or a property name was invalid or a
   *      property retrieval failed)
   */
  public Map getListedProperties(
    List resourceHandles,
    List propertyNames)
    throws ResourceException, OperationNotCompletedException;

  /**
   * Get maps of all properties of a list of resource handles. If no properties
   * are available for a resource, the whole resource isn't stored inside the
   * returned data structure. Returns an empty map if no properties are
   * available for all resources. </p> Note: This method needs not to be handled
   * atomically! Although this operation may fail in parts, the operation on one
   * of the many operands provided through the parameter list is always expected
   * to be atomic!
   *
   * @param resourceHandles list of resource handles
   * @return map of maps of properties indexed by resource handles; the inner
   *      maps hold properties indexed by property names
   * @exception ResourceException when a general problem occured
   * @exception OperationNotCompletedException when the operation failed in
   *      parts (either a resource handle or a property name was invalid or a
   *      property retrieval failed)
   */
  public Map getAllProperties(List resourceHandles)
    throws ResourceException, OperationNotCompletedException;
}
