/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.mi.property;
import com.sap.netweaver.bc.rf.common.*;

import com.sap.netweaver.bc.rf.common.exception.*;
import com.sap.netweaver.bc.rf.common.property.*;

import java.util.*;

/**
 * Read-And-Write interface describing a repository sub manager for properties.
 * Resources usually have properties describing the resource. All system
 * properties MUST be made available as properties. Resources with content MUST
 * make the content metadata available as properties also! This interface
 * provides means to write the properties of a resource.
 *
 * @created 23. Januar 2003
 */
public interface IMutablePropertyManager extends IPropertyManager {
  /**
   * Update single property on a single resource handle. Note: Descriptors may
   * not be altered and must be fully released when returning from the call due
   * to performance (reuse) reasons.
   *
   * @param resourceHandle resource handle
   * @param update property update descriptor
   * @exception ResourceException when either the resource handle or property
   *      update descriptor is invalid or the property update failed
   */
  public void updateProperty(
    IResourceHandle resourceHandle,
    IPropertyUpdateDescriptor update)
    throws ResourceException;

  /**
   * Update list of properties on a single resource handle. Note: Descriptors
   * may not be altered and must be fully released when returning from the call
   * due to performance (reuse) reasons.
   *
   * @param resourceHandle resource handle
   * @param updates list of property updates descriptors
   * @exception ResourceException when either the resource handle or property
   *      update descriptors are invalid or the property update failed
   */
  public void updateProperties(IResourceHandle resourceHandle,
    List updates)
    throws ResourceException;

  /**
   * Update single property on a list of resource handles. </p> Note: This
   * method needs not to be handled atomically! Although this operation may fail
   * in parts, the operation on one of the many operands provided through the
   * parameter list is always expected to be atomic! Note: Descriptors may not
   * be altered and must be fully released when returning from the call due to
   * performance (reuse) reasons.
   *
   * @param resourceHandles list of resource handles
   * @param update property update descriptor
   * @exception ResourceException when either the resource handles or property
   *      update descriptor are invalid or the property updates failed generally
   * @exception OperationNotCompletedException when the operation failed in
   *      parts
   */
  public void updateProperty(
    List resourceHandles,
    IPropertyUpdateDescriptor update)
    throws ResourceException, OperationNotCompletedException;

  /**
   * Update list of properties on a list of resource handles. </p> Note: This
   * method needs not to be handled atomically! Although this operation may fail
   * in parts, the operation on one of the many operands provided through the
   * parameter list is always expected to be atomic! Note: Descriptors may not
   * be altered and must be fully released when returning from the call due to
   * performance (reuse) reasons.
   *
   * @param resourceHandles list of resource handles
   * @param updates list of property updates descriptors
   * @exception ResourceException when either the resource handles or property
   *      update descriptors are invalid or the property updates failed
   *      generally
   * @exception OperationNotCompletedException when the operation failed in
   *      parts
   */
  public void updateProperties(List resourceHandles,
    List updates)
    throws ResourceException, OperationNotCompletedException;
}
