/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.mi.namespace;
import com.sap.netweaver.bc.rf.common.*;

import com.sap.netweaver.bc.rf.common.exception.*;
import com.sap.netweaver.bc.rf.common.namespace.*;
import com.sap.netweaver.bc.rf.util.namespace.*;

import java.util.*;

/**
 * Read-Only interface describing a repository sub manager for namespace.
 * Namespace means the realm of names the resources fill in the repository. This
 * manager mainly provides means to traverse resources in the namespace.
 */
public interface INamespaceManager {
  /**
   * Find resources matching certain criteria specified in the find resources
   * descriptor returning a list of the results.
   *
   * @param rootResourceHandle root resource handle
   * @param findResourcesDescriptor find resources descriptor
   * @param offset if a positive number, the first n matching resources will be
   *      skipped; 0 reports all results
   * @param length if a positive number, only the top number of matching
   *      resources will be returned; if less entries are returned no more data
   *      is available, if exact &lt;length&gt; entries are returned there might
   *      be more and a new call with &lt;oldoffset&gt; + &lt;length&gt; as the
   *      new offset is recommended to determine whether the overall result was
   *      complete; a negative number reports all results
   * @param resultStateDescriptor optional parameter identifying a former query
   *      and their results; if the implementation is able to keep a "session
   *      state" for upcoming calls this state can be passed as an extra
   *      argument. In this case the result of this query should be a special
   *      object (implementing List) with a getter for the session result state
   *      descriptor object
   * @return list of the resources matching the find resources descriptor
   *      criteria
   * @exception OperationNotSupportedException Exception raised when the
   *      iterator based counterpart is implemented only
   * @exception ResourceException Exception raised in failure situation
   * @see IFindResourcesDescriptor
   * @see IResultStateList
   */
  public List findResources(IResourceHandle rootResourceHandle,
    IFindResourcesDescriptor findResourcesDescriptor,
    int offset,
    int length,
    Object resultStateDescriptor)
    throws ResourceException, OperationNotSupportedException;

  /**
   * Find resources matching certain criteria specified in the find resources
   * descriptor returning an iterator for the results.
   *
   * @param rootResourceHandle root resource handle
   * @param findResourcesDescriptor find resources descriptor
   * @param startIndex start index from which on the matching resources will be
   *      returned; support is not mandatory; 0 reports all results
   * @param fetchSize hint for the implementation how often next() will usually
   *      be called in a row; support is not mandatory and fetchSize may be
   *      internally overwritten to be of a more optimal value not throwing an
   *      exception; 0 defines no hint
   * @return iterator over the resources matching the find resources descriptor
   *      criteria
   * @exception OperationNotSupportedException Exception raised when the list
   *      based counterpart is implemented only
   * @exception ResourceException Exception raised in failure situation
   * @see IFindResourcesDescriptor
   */
  public Iterator findResources(IResourceHandle rootResourceHandle,
    IFindResourcesDescriptor findResourcesDescriptor,
    int startIndex,
    int fetchSize)
    throws ResourceException, OperationNotSupportedException;

  /**
   * Count resources matching certain criteria specified in the find resources
   * descriptor returning their number.
   *
   * @param rootResourceHandle root resource handle
   * @param findResourcesDescriptor find resources descriptor
   * @return number of resources matching the find resources descriptor criteria
   * @exception ResourceException Exception raised in failure situation
   * @exception OperationNotSupportedException Exception raised in failure
   *      situation
   * @see IFindResourcesDescriptor
   */
  public long countResources(IResourceHandle rootResourceHandle,
    IFindResourcesDescriptor findResourcesDescriptor)
    throws ResourceException, OperationNotSupportedException;

  /**
   * Get a search manager specific query builder, which may be used to build
   * query expressions of type {@link com.sap.netweaver.bc.rf.common.search.IQueryExpression}.
   *
   * @return search manager specific query builder
   * @exception ResourceException Exception raised in failure situation
   * @exception OperationNotSupportedException Exception raised in failure
   *      situation
   */
  public IQueryBuilder getQueryBuilder()
    throws ResourceException, OperationNotSupportedException;

  /**
   * Return whether or not the manager has special knowledge about the given
   * property, i.e. undestands the semantic of this property. This is
   * interesting for a client to know, when queries about special, e. g. live
   * properties like DAV:supported-method-set, are done. In case the search
   * manager does not understand the property, the client might want to test
   * this property himself.
   *
   * @param propertyName property name
   * @return true when the manager has special knowledge about the given
   *      property, i.e. undestands the semantic of this property
   * @exception ResourceException Exception raised in failure situation
   * @exception OperationNotSupportedException Exception raised in failure
   *      situation
   */
  public boolean isQueryPropertyNameUnderstood(IName propertyName)
    throws ResourceException, OperationNotSupportedException;

  /**
   * Check if a resource handle denotes a resource collection.
   *
   * @param handle resource handle
   * @return true if the resource handle identifies a collection
   * @exception ResourceException Exception raised in failure situation
   */
  public boolean isCollection(IResourceHandle handle)
    throws ResourceException;

  /**
   * Check if a resource handle denotes a resource link, i.e. a resource linking
   * to another resource.
   *
   * @param handle resource handle
   * @return true if the resource handle identifies a link
   * @exception ResourceException Exception raised in failure situation
   */
  public boolean isLink(IResourceHandle handle)
    throws ResourceException;

  /**
   * Returns the {@link OrderMechanism} of the collection.
   *
   * @param resourceHandle resource handle of the collection
   * @return the {@link OrderMechanism} of the collection.
   * @exception ResourceException Exception raised in failure situation
   */
  public OrderMechanism getCollectionOrderMechanism(IResourceHandle resourceHandle)
    throws ResourceException;

  /**
   * Returns the {@link ILinkDescriptor} of the link.
   *
   * @param resourceHandle resource handle of the link
   * @return the {@link ILinkDescriptor} of the link.
   * @exception ResourceException Exception raised in failure situation
   */
  public ILinkDescriptor getLinkDescriptor(IResourceHandle resourceHandle)
    throws ResourceException;
}
