/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.mi.namespace;
import com.sap.netweaver.bc.rf.common.*;

import com.sap.netweaver.bc.rf.common.exception.*;
import com.sap.netweaver.bc.rf.common.namespace.*;

import java.util.*;

/**
 * Read-And-Write interface describing a repository sub manager for namespace.
 * Namespace means the realm of names the resources fill in the repository. This
 * manager mainly provides means to create resources in the namespace.
 *
 * @created 23. Januar 2003
 */
public interface IMutableNamespaceManager extends INamespaceManager {
  /**
   * Create a resource based on a descriptor. Default resource create
   * descriptors reside in the same package as the interface does. A repository
   * may support these and/or others. Note: Descriptors may not be altered and
   * must be fully released when returning from the call due to performance
   * (reuse) reasons.
   *
   * @param createDescriptor create descriptor
   * @return resource handle of the newly created resource
   * @exception ResourceException Exception raised in failure situation
   */
  public IResourceHandle createResource(ICreateDescriptor createDescriptor)
    throws ResourceException;

  /**
   * Deletes the resource. If the resource is a collection then all children
   * will be deleted ("deep delete"). This method is not atomic. Although each
   * single delete of a "deep delete" must be atomic. If the resource is
   * versioned it is up to the repositoy implementation to decide whether to
   * delete the complete version history and all revisions or not. Usually the
   * revisions and version history should not be deleted. Note: Descriptors may
   * not be altered and must be fully released when returning from the call due
   * to performance (reuse) reasons.
   *
   * @param resourceHandle resource handle
   * @return list of resource identifiers ({@link
   *      com.sap.netweaver.bc.rf.common.IRid}) of all deleted resources
   * @exception OperationNotCompletedException If the call was successfull but
   *      some of the resources could not be deleted (in case of a "deep
   *      delete"). Possible reasons are: resource is locked, user does not have
   *      delete permission, etc.
   * @exception ResourceException Exception raised in failure situation
   */
  public List deleteResource(IResourceHandle resourceHandle)
    throws ResourceException, OperationNotCompletedException;

  /**
   * Copy resource based on a resource copy descriptor. Default resource copy
   * descriptors reside in the same package as the interface does. A repository
   * may supports these and/or others. The first returned resource handle must
   * be the copied root resource handle, if this operation succeeded at all.
   * </p> Note: This method needs not to be handled atomically! Although this
   * operation may fail in parts, the operation on one of the many operands
   * provided through the parameter list is always expected to be atomic! Note:
   * Descriptors may not be altered and must be fully released when returning
   * from the call due to performance (reuse) reasons.
   *
   * @param resourceHandle resource handle
   * @param copyDescriptor copy descriptor
   * @return list of new resource identifiers ({@link
   *      com.sap.netweaver.bc.rf.common.IRid}) of all copied resources
   * @exception OperationNotCompletedException when the call was successfull but
   *      some of the resources could not be copied (in case of a "deep copy");
   *      possible reasons are, that the target collection is locked, the user
   *      does not have create permission, etc.; child resource identifiers are
   *      not reported
   * @exception ResourceException Exception raised in failure situation
   */
  public List copyResource(
    IResourceHandle resourceHandle,
    ICopyDescriptor copyDescriptor)
    throws ResourceException, OperationNotCompletedException;

  /**
   * Move resource based on a resource move descriptor. Default resource move
   * descriptors reside in the same package as the interface does. A repository
   * may support these and/or others. The first returned resource handle must be
   * the moved root resource handle, if this operation succeeded at all. </p>
   * Note: This method needs not to be handled atomically! Although this
   * operation may fail in parts, the operation on one of the many operands
   * provided through the parameter list is always expected to be atomic! Note:
   * Descriptors may not be altered and must be fully released when returning
   * from the call due to performance (reuse) reasons.
   *
   * @param resourceHandle resource handle
   * @param moveDescriptor move descriptor
   * @return list of all moved resources; resources failed to be moved either of
   *      repository-internal reasons, follow-up reasons or others are stored
   *      with their respective exception inside the list
   * @exception OperationNotCompletedException If the call was successfull but
   *      some of the resources could not be moved (in case of a "deep copy").
   *      Possible reasons are: target collection is locked, user does not have
   *      create permission, etc.
   * @exception ResourceException Exception raised in failure situation
   */
  public List moveResource(
    IResourceHandle resourceHandle,
    IMoveDescriptor moveDescriptor)
    throws ResourceException, OperationNotCompletedException;

  /**
   * Change the {@link OrderMechanism} of the collection.
   *
   * @param resourceHandle resource handle of the collection
   * @param orderMechanism order mechanism of the collection
   * @exception ResourceException Exception raised in failure situation
   */
  public void setCollectionOrderMechanism(
    IResourceHandle resourceHandle,
    OrderMechanism orderMechanism)
    throws ResourceException;

  /**
   * Reorder children of the resource collection. The children of the resource
   * collection will be rearranged according to the given list of {@link
   * OrderPosition}s for the children of the given resource collection.
   *
   * @param resourceHandle resource handle of the collection
   * @param orderPositions list of {@link OrderPosition} s for the children of
   *      the given resource collection
   * @exception ResourceException Exception raised in failure situation
   */
  public void reorderCollection(
    IResourceHandle resourceHandle,
    List orderPositions)
    throws ResourceException;

  /**
   * Changes a link's type and/or target. Note: Descriptors may not be altered
   * and must be fully released when returning from the call due to performance
   * (reuse) reasons.
   *
   * @param resourceHandle resource handle of the link
   * @param linkDescriptor link descriptor of the link
   * @exception ResourceException Exception raised in failure situation
   */
  public void setLinkDescriptor(
    IResourceHandle resourceHandle,
    ILinkDescriptor linkDescriptor)
    throws ResourceException;
}
