/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.mi.lock;

import com.sap.netweaver.bc.rf.common.*;
import com.sap.netweaver.bc.rf.common.lock.*;

import java.util.*;

/**
 * Class implementing lock information.
 *
 * @created 20. Mrz 2003
 */

public class LockInfo implements ILockInfo {
  public final static long LOCK_TIMEOUT_INFINITE = -1;

  private String token = null;
  private IRid rid;
  private LockDepthType depth;
  private LockScopeType scope;
  private LockKindType kind;
  private Date creationDate = null;
  private Date expirationDate = null;
  private long timeout = LOCK_TIMEOUT_INFINITE;
  private String owner = null;
  private String clientSuppliedOwner = null;

  /**
   * Construct object of class LockInfo.
   *
   * @param token lock token identifying the lock
   * @param rid resource id this lock is valid for
   * @param depth {@link LockDepthType} of lock
   * @param scope {@link LockScopeType} of lock
   * @param kind {@link LockKindType} of lock
   * @param timeout lock timeout in milliseconds
   * @param owner owner for the lock
   */
  public LockInfo(
    String token,
    IRid rid,
    LockDepthType depth,
    LockScopeType scope,
    LockKindType kind,
    long timeout,
    String owner) {
    this(token, rid, depth, scope, kind, null, timeout, owner, null);
  }

  /**
   * Construct object of class LockInfo.
   *
   * @param token lock token identifying the lock
   * @param rid resource id this lock is valid for
   * @param depth {@link LockDepthType} of lock
   * @param scope {@link LockScopeType} of lock
   * @param kind {@link LockKindType} of lock
   * @param creationDate creation date of the lock
   * @param timeout lock timeout in milliseconds
   * @param owner owner for the lock
   */
  public LockInfo(
    String token,
    IRid rid,
    LockDepthType depth,
    LockScopeType scope,
    LockKindType kind,
    Date creationDate,
    long timeout,
    String owner) {
    this(token, rid, depth, scope, kind, creationDate, timeout, owner, null);
  }

  /**
   * Construct object of class LockInfo.
   *
   * @param token lock token identifying the lock
   * @param rid resource id this lock is valid for
   * @param depth {@link LockDepthType} of lock
   * @param scope {@link LockScopeType} of lock
   * @param kind {@link LockKindType} of lock
   * @param timeout lock timeout in milliseconds
   * @param owner owner for the lock
   * @param clientSuppliedOwner client supplied owner for the lock
   */
  public LockInfo(
    String token,
    IRid rid,
    LockDepthType depth,
    LockScopeType scope,
    LockKindType kind,
    long timeout,
    String owner,
    String clientSuppliedOwner) {
    this(token, rid, depth, scope, kind, null, timeout, owner, clientSuppliedOwner);
  }

  /**
   * Construct object of class LockInfo.
   *
   * @param token lock token identifying the lock
   * @param rid resource id this lock is valid for
   * @param depth {@link LockDepthType} of lock
   * @param scope {@link LockScopeType} of lock
   * @param kind {@link LockKindType} of lock
   * @param creationDate creation date of the lock
   * @param timeout lock timeout in milliseconds
   * @param owner owner for the lock
   * @param clientSuppliedOwner client supplied owner for the lock
   */
  public LockInfo(
    String token,
    IRid rid,
    LockDepthType depth,
    LockScopeType scope,
    LockKindType kind,
    Date creationDate,
    long timeout,
    String owner,
    String clientSuppliedOwner) {
    if (token == null) {
      this.token = "";
    }
    else {
      this.token = token;
    }
    this.rid = rid;
    this.depth = depth;
    this.scope = scope;
    this.kind = kind;
    if (creationDate == null) {
      this.creationDate = new Date();
    }
    else {
      this.creationDate = creationDate;
    }
    if (timeout < 0) {
      this.timeout = LOCK_TIMEOUT_INFINITE;
      this.expirationDate = null;
    }
    else {
      this.timeout = timeout;
      this.expirationDate = new Date(this.creationDate.getTime() + timeout);
    }
    if (owner == null) {
      this.owner = "";
    }
    else {
      this.owner = owner;
    }
    this.clientSuppliedOwner = clientSuppliedOwner;
  }

  /**
   * Get lock token. The lock token must be either an absolute URI (as defined
   * in <a href="http://www.ietf.org/rfc/rfc2396.txt">RFC2396</a> or a relative
   * URI reference (in which case the <a
   * href="http://greenbytes.de/tech/webdav/rfc2518.html#rfc.section.6.4">
   * opaquelocktoken</a> URI scheme is assumed).
   *
   * @return lock token
   */
  public String getLockToken() {
    return token;
  }

  /**
   * Get resource id this lock info is valid for.
   *
   * @return resource id
   */
  public IRid getRid() {
    return rid;
  }

  /**
   * Get lock depth.
   *
   * @return {@link LockDepthType} of lock
   */
  public LockDepthType getDepth() {
    return depth;
  }

  /**
   * Get lock scope.
   *
   * @return {@link LockScopeType} of lock
   */
  public LockScopeType getScope() {
    return scope;
  }

  /**
   * Get lock kind.
   *
   * @return {@link LockKindType} of lock
   */
  public LockKindType getKind() {
    return kind;
  }

  /**
   * Get creation date.
   *
   * @return creation date
   */
  public Date getCreationDate() {
    return creationDate;
  }

  /**
   * Get lock expiration timestamp.
   *
   * @return lock expiration timestamp
   */
  public Date getExpirationDate() {
    return expirationDate;
  }

  /**
   * Get time remaining until lock expires.
   *
   * @return time remaining until lock expires (or <code>-1</code> for
   *      "infinite").
   */
  public long getTimeRemaining() {
    if (expirationDate != null) {
      long timeRemaining =
        expirationDate.getTime() - new Date().getTime();
      if (timeRemaining < 0) {
        timeRemaining = 0;
      }
      return timeRemaining;
    }
    else {
      return LOCK_TIMEOUT_INFINITE;
    }
  }

  /**
   * Get lock timeout in milliseconds.
   *
   * @return lock timeout in milliseconds
   */
  public long getTimeout() {
    return timeout;
  }

  /**
   * Get user id of lock owner.
   *
   * @return user id of lock owner
   */
  public String getOwner() {
    return owner;
  }

  /**
   * Get client supplied owner for the lock. A client can submit his own owner
   * information that should be returned upon lock discovery. If present, the
   * format must be the XML serialization of a <a
   * href="http://www.greenbytes.de/tech/webdav/rfc2518.html#ELEMENT_owner">
   * DAV:owner</a> element.
   *
   * @return client supplied owner for the lock
   */
  public String getClientSuppliedOwner() {
    return clientSuppliedOwner;
  }

  /**
   * Check this and another object for equality.
   *
   * @param obj object to compare this object with
   * @return true if both objects are considered to be equal, false otherwise
   * @see java.lang.Object#equals(java.lang.Object)
   */
  public boolean equals(Object obj) {
    if (obj instanceof String) {
      return ((String)obj).equals(token);
    }
    if (obj instanceof LockInfo) {
      return ((ILockInfo)obj).getLockToken().equals(token);
    }
    return false;
  }

  /**
   * Get hash code of this object.
   *
   * @return hash code of this object
   * @see java.lang.Object#hashCode()
   */
  public int hashCode() {
    return token.hashCode();
  }

  /**
   * reset the expiration period if timeout info is available
   */
  public void refreshExpirationDate() {
    if (this.timeout > 0) {
      this.expirationDate = new Date((new Date()).getTime() + timeout);
    }
  }
}
