/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.mi.lock;
import com.sap.netweaver.bc.rf.common.*;

import com.sap.netweaver.bc.rf.common.exception.*;
import com.sap.netweaver.bc.rf.common.lock.*;

/**
 * Read-And-Write interface describing a repository sub manager for locking.
 * Resources may be locked shallow or deep (in the resource hierarchy), exclusiv
 * or shared (concerning concurrent access) and for reading or writing. Shallow
 * locks affect only the given resource handle (content, properties, etc. and
 * the namespace, not the resources, below the resource), where deep locks
 * effect also the descendents in the resource hierarchy. Exclusive locks block
 * any other lock from being obtained on the given resource handle and block any
 * unlocked access. Shared locks block any exclusive lock from being obtained on
 * the given resource handle and block any unlocked access too, but allow for
 * further shared locks. Write locks block any other lock from being obtained on
 * the given resource handle and block any unlocked access. Read locks block any
 * write lock from being obtained on the given resource handle and block any
 * unlocked access too, but allow for further read locks.
 *
 * @created 23. Januar 2003
 */
public interface IMutableLockManager extends ILockManager {
  /**
   * Lock a resource (content, properties, etc. and the namespace, not the
   * resources, below the resource).
   *
   * @param resourceHandle resource handle
   * @param lockDescriptor lock descriptor object describing the properties of
   *      the lock
   * @return lock info object describing the state of the lock
   * @exception ResourceException when the lock can't be obtained
   */
  public ILockInfo lock(
    IResourceHandle resourceHandle,
    ILockDescriptor lockDescriptor)
    throws ResourceException;

  /**
   * Unlock a resource (content, properties, etc. and the namespace, not the
   * resources, below the resource).
   *
   * @param resourceHandle resource handle
   * @param lockToken lock token identifying the lock
   * @exception ResourceException when the lock can't be released
   */
  public void unlock(IResourceHandle resourceHandle, String lockToken)
    throws ResourceException;

  /**
   * Refresh, i.e. reset the timer of the given lock for a given resource handle
   * until the lock is released automatically.
   *
   * @param resourceHandle resource handle
   * @param lockToken lock token identifying the lock
   * @return lock info object describing the new state of the lock
   * @exception ResourceException when the lock can't be refreshed
   */
  public ILockInfo refreshLock(IResourceHandle resourceHandle, String lockToken)
    throws ResourceException;
}
