/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.mi.idmapper;

import com.sap.netweaver.bc.rf.common.*;
import com.sap.netweaver.bc.rf.common.exception.*;

import java.util.*;

/**
 * Read-And-Write interface describing a repository sub manager for id mapping.
 * Ids must identify a resource uniquely in the repository, no matter how its
 * position or properties change. When its identity really gets lost or is
 * replaced, the id must never be used again resp. a new id must be assigned to
 * the new resource replacing the old (possibly on the same position). Ids must
 * never be reused in order to undoubtedly identify a resource resp. that the
 * resource no longer exists (e.g. when ids sent through emails are used to
 * identify the referenced resource again). This interface allows to assign
 * unique ids to resources in a repository from outside. They should be globally
 * unique ids in order to fulfill the requirements (uniqueness, no reuse) in a
 * performant way (no check expected here where all resources are checkd for the
 * new id in the repository).
 *
 * @created 23. Januar 2003
 */
public interface IMutableIdMapperManager extends IIdMapperManager {
  /**
   * Assign unique id to resource id.
   *
   * @param resourceId resource id
   * @param uniqueId unique id to be assigned
   * @exception ResourceException when a general problem occures, the resource
   *      id is invalid or the unique id is already in use
   */
  public void assignToRid(IRid resourceId, String uniqueId)
    throws ResourceException;

  /**
   * Assign unique id to resource handle.
   *
   * @param resourceHandle resource handle
   * @param uniqueId unique id to be assigned
   * @exception ResourceException when a general problem occures, the resource
   *      handle is invalid or the unique id is already in use
   */
  public void assignToResource(IResourceHandle resourceHandle, String uniqueId)
    throws ResourceException;

  /**
   * Assign list of unique ids to list of resource ids. </p> Note: This method
   * needs not to be handled atomically! Although this operation may fail in
   * parts, the operation on one of the many operands provided through the
   * parameter list is always expected to be atomic!
   *
   * @param uniqueIdsIndexedByResourceIds map of unique ids to be assigned
   *      indexed by resource ids
   * @exception ResourceException when a general problem occures
   * @exception OperationNotCompletedException when the operation failed in
   *      parts
   */
  public void assignToRids(Map uniqueIdsIndexedByResourceIds)
    throws ResourceException, OperationNotCompletedException;

  /**
   * Assign list of unique ids to list of resource handles. </p> Note: This
   * method needs not to be handled atomically! Although this operation may fail
   * in parts, the operation on one of the many operands provided through the
   * parameter list is always expected to be atomic!
   *
   * @param uniqueIdsIndexedByResourceHandles map of unique ids to be assigned
   *      indexed by resource handles
   * @exception ResourceException when a general problem occures
   * @exception OperationNotCompletedException when the operation failed in
   *      parts
   */
  public void assignToResources(Map uniqueIdsIndexedByResourceHandles)
    throws ResourceException, OperationNotCompletedException;
}
