/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.mi.idmapper;
import com.sap.netweaver.bc.rf.common.*;

import com.sap.netweaver.bc.rf.common.exception.*;

import java.util.*;

/**
 * Read-Only interface describing a repository sub manager for id mapping. Ids
 * must identify a resource uniquely in the repository, no matter how its
 * position or properties change. When its identity really gets lost or is
 * replaced, the id must never be used again resp. a new id must be assigned to
 * the new resource replacing the old (possibly on the same position). Ids must
 * never be reused in order to undoubtedly identify a resource resp. that the
 * resource no longer exists (e.g. when ids sent through emails are used to
 * identify the referenced resource again). This interface allows to look up
 * resources based on ids.
 *
 * @created 23. Januar 2003
 */
public interface IIdMapperManager {
  /**
   * Look up resource in repository by unique id.
   *
   * @param uniqueId unique id
   * @return looked up resource
   * @exception ResourceException when a general problem occures
   */
  public IResourceHandle lookup(String uniqueId)
    throws ResourceException;

  /**
   * Get unique id of resource.
   *
   * @param resourceHandle resource handle
   * @return unique id of resource handle
   * @exception ResourceException when a general problem occures
   */
  public String getUniqueId(IResourceHandle resourceHandle)
    throws ResourceException;

  /**
   * Look up resources in repository by list of unique ids. </p> Note: This
   * method needs not to be handled atomically! Although this operation may fail
   * in parts, the operation on one of the many operands provided through the
   * parameter list is always expected to be atomic!
   *
   * @param uniqueIds list of unique ids
   * @return map of looked up resources indexed by unique id
   * @exception ResourceException when a general problem occures
   * @exception OperationNotCompletedException when the operation failed in
   *      parts
   */
  public Map lookup(List uniqueIds)
    throws ResourceException, OperationNotCompletedException;

  /**
   * Get list of unique ids by list of resources. </p> Note: This method needs
   * not to be handled atomically! Although this operation may fail in parts,
   * the operation on one of the many operands provided through the parameter
   * list is always expected to be atomic!
   *
   * @param resourceHandles list of resource handles
   * @return map of looked up unique ids of resource handles indexed by resource
   *      handles
   * @exception ResourceException when a general problem occures
   * @exception OperationNotCompletedException when the operation failed in
   *      parts
   */
  public Map getUniqueIds(List resourceHandles)
    throws ResourceException, OperationNotCompletedException;
}
