/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.mi;

import com.sap.netweaver.bc.rf.common.*;
import com.sap.netweaver.bc.rf.common.exception.*;
import com.sap.netweaver.bc.rf.common.support.NameInfo;

import java.util.*;

/**
 * Interface describing a repository manager. A repository manager is the main
 * unit, needed to integrate a repository into the repository framework.
 *
 * @created 23. Januar 2003
 */
public interface IManager {
  /**
   * Get the unique id of the repository manager (usually assigned and made
   * unique through the configuration framework).
   *
   * @return unique id of the repository manager
   */
  public String getId();


  /**
   * Get the root token of a resource id, this resource manager claims
   * responsibility for. The framework will call this method to know which
   * repository manager to call when a resource having the matching root token
   * needs to be handled.
   *
   * @return root token of a resource id, this resource manager claims
   *      responsibility for
   */
  public String getRidPrefix();


  /**
   * Get the {@link NameInfo} of a repository manager. The {@link NameInfo}
   * contains information about the restrictions for resource and collection
   * names.
   *
   * @return information about the restrictions for resource and collection
   *      names
   */
  public NameInfo getNameInfo();


  /**
   * Get map of sub managers indexed by sub manager interface, e.g. a repository
   * manager with a content and property sub manager implemented in one class
   * and a separate security sub manager implemented in another class, returns a
   * map with the content and property sub manager added twice via the content
   * sub manager interface and the property sub manager interface and the
   * security sub manager via the security sub manager interface.
   *
   * @return map of sub managers indexed by sub manager interface
   */
  public Map getSubManagers();


  /**
   * Look up a resource in this repository by resource identifer. Must return
   * <code>null</code> if the resource does not exist.
   *
   * @param rid resource identifier to be looked up.
   * @return The resource handle or <code>null</code> if the resource with the
   *      specified RID does not exist. <code>null</code> must be returned only
   *      if the resource definitively does not exist; if the lookup fails for
   *      some other reason (e.g. internal error, invalid RID) an appropriate
   *      exception such as ServiceNotAvailableException must be thrown.
   * @exception ResourceException if the RID is invalid or the lookup failed.
   */
  public IResourceHandle lookup(IRid rid)
    throws ResourceException;


  /**
   * Look up resources in repository by list of resource ids and prefetch
   * content and certain properties when requested.</p> Note: This method needs
   * not to be handled atomically! Although this operation may fail in parts,
   * the operation on one of the many operands provided through the parameter
   * list is always expected to be atomic!
   *
   * @param ridList list of resource ids to be looked up
   * @param contentToBeFetched flag indicating whether or not content should
   *      prefetched; content not prefetched must still be available via
   *      subsequent IContentManager calls
   * @param propertyNamesToBeFetched list of property names to be prefetched;
   *      properties not prefetched must still be available via subsequent
   *      IPropertyManager calls
   * @return looked up resources
   * @exception ResourceException Exception raised in failure situation
   * @exception OperationNotCompletedException when the operation failed in
   *      parts
   */
  public List lookup(List ridList,
    boolean contentToBeFetched,
    List propertyNamesToBeFetched)
    throws ResourceException, OperationNotCompletedException;


  /**
   * Look up resources in repository by list of resource ids and check thereby
   * for the given permission names. This is a performance optimized bulk call:
   * When no content is provided (null or empty content parameter) the real
   * content must be looked up. Otherwise the repository should use the given
   * content for the returned resources and don't spend time on retrieving the
   * real content. The same goes for properties. When no permission names are
   * provided (null or empty permission parameter), no permissions must be
   * checked. </p> Note: This method needs not to be handled atomically!
   * Although this operation may fail in parts, the operation on one of the many
   * operands provided through the parameter list is always expected to be
   * atomic!
   *
   * @param ridList list of resource ids to be looked up
   * @param content map of content to be used for the returned resources, where
   *      the key is the RID and the value is the content
   * @param properties map of properties to be used for the returned resources,
   *      where the key is the RID and the value is a list of properties
   * @param permissionNames array of permission names to be checked for or null
   *      or empty for no check
   * @return looked up resources
   * @exception ResourceException Exception raised in failure situation
   * @exception OperationNotCompletedException when the operation failed in
   *      parts
   */
  public List lookup(List ridList, Map content, Map properties, String[] permissionNames)
    throws ResourceException, OperationNotCompletedException;


  /**
   * Get set of options this repository manager supports. In case of a null
   * resource handle, the repository shall return the set of all supported
   * operations.
   *
   * @param resourceHandle resource handle
   * @return set of options this repository manager supports
   */
  public Set getSupportedOptions(IResourceHandle resourceHandle);
}
