/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.mi;
import com.sap.netweaver.bc.rf.common.support.NameInfo;

import com.sap.netweaver.bc.rf.mi.content.*;
import com.sap.netweaver.bc.rf.mi.lock.*;
import com.sap.netweaver.bc.rf.mi.namespace.*;
import com.sap.netweaver.bc.rf.mi.property.*;
import com.sap.netweaver.bc.rf.mi.security.*;
import com.sap.netweaver.bc.rf.mi.version.*;
import com.sap.netweaver.bc.rf.util.exception.*;
import com.sap.netweaver.bc.rf.util.logging.*;

import com.sap.tc.logging.Location;
import com.sapportals.config.fwk.Configuration;
import com.sapportals.config.fwk.IConfigClientContext;
import com.sapportals.config.fwk.IConfigManager;
import com.sapportals.config.fwk.IConfigPlugin;
import com.sapportals.config.fwk.IConfigurable;
import com.sapportals.wcm.WcmException;

import com.sapportals.wcm.crt.*;
import com.sapportals.wcm.crt.component.*;
import com.sapportals.wcm.crt.configuration.*;

// Begin of section of obsolete imports
import com.sapportals.wcm.repository.IResource;
import com.sapportals.wcm.repository.manager.IResourceEvent;
import com.sapportals.wcm.repository.manager.IResourceEventBroker;
import com.sapportals.wcm.repository.manager.IResourceEventSender;
import com.sapportals.wcm.repository.manager.ResourceEvent;
import com.sapportals.wcm.repository.runtime.ICmRuntimeConst;
import com.sapportals.wcm.util.config.ConfigCrutch;
import com.sapportals.wcm.util.events.EventList;
import com.sapportals.wcm.util.events.IEventList;

import java.util.*;

/**
 * Abstract base class for repository managers.
 *
 * @created 20. Mrz 2003
 */
public abstract class AbstractManager
	implements
		IManager,
		IResourceEventSender,
		IThreadSafe,
		IContainer,
		com.sapportals.wcm.crt.component.IConfigurable,
		IStartable,
		IComponentInfo,
		ILifecycleInfo {
	protected final ComponentStateHandler stateHandler = new ComponentStateHandler(this);

	private final static String CFG_PLUGIN_CM_REPOSITORY_MANAGERS_SECURITY_MANAGERS =
		"/cm/repository_managers/security_managers";

	/**
	 * This manager's configuration
	 */
	protected IConfiguration config;

	protected String prefix;
	private String id;

	private INamespaceManager namespaceManager;
	private IPropertyManager propertyManager;
	private ILockManager lockManager;
	private IBasicVersioningManager basicVersioningManager;
	private IContentManager contentManager;
	private ISecurityManager securityManager;

	/*
	 * Maps sub-manager interface name -> instance
	 */
	private Map subManagers;

	/**
	 * Value of the configuration parameter "sendevents"
	 */
	private boolean eventsEnabled;

	/**
	 * Reference to the event broker
	 */
	private IResourceEventBroker eventBroker;

	/**
	 * The container manager of CRT
	 */
	private IContainerManager containerManager;

	/**
	 * The CRT key of this component
	 */
	private String key;

	/**
	 * Flag: startup was called
	 */
	private boolean started;

	private final static String CFG_SEC_MGR_CLASS = "class";

	private static Location logger = Location.getLocation(com.sap.netweaver.bc.rf.mi.AbstractManager.class);

	private final static String[] NOT_RECONFIGURABLE =
		{
			AbstractManager.CFG.PREFIX,
			AbstractManager.CFG.EVENTS,
			AbstractManager.CFG.SERVICES,
			AbstractManager.CFG.NAMESPACE_MANAGER,
			AbstractManager.CFG.PROPERTY_MANAGER,
			AbstractManager.CFG.LOCK_MANAGER,
			AbstractManager.CFG.VERSIONING_MANAGER,
			AbstractManager.CFG.CONTENT_MANAGER,
			AbstractManager.CFG.SECURITY_MANAGER,
			AbstractManager.CFG.PROPSEARCH_MANAGER };

	private final static String EVENTBROKER_CRTKEY = "eventbroker";
	private final static String EVENTBROKER_CLASSNAME = "com.sapportals.wcm.repository.manager.ResourceEventBroker";
	private final static EventList resourceEvents = new EventList();
	private final static EventList collectionEvents = new EventList();

	/**
	 * A "empty" name info object. It's constructor will take the Framework
	 * defaults
	 */
	private final static NameInfo nameInfo = new NameInfo(new char[] {
	}, new char[] {
	}, -1, -1);


	private static Map securityManagerMap;

	/**
	 * Construct object of class AbstractManager.
	 */
	public AbstractManager() {
	}

	public IConfiguration getConfig() {
		return this.config;
	}

	// ---------------------------------------------------------------------------
	// Getter methods for known sub-managers
	// ---------------------------------------------------------------------------

	public ISecurityManager getSecurityManager() {
		return this.securityManager;
	}

	public INamespaceManager getNamespaceManager() {
		return this.namespaceManager;
	}

	public IPropertyManager getPropertyManager() {
		return this.propertyManager;
	}

	public ILockManager getLockManager() {
		return this.lockManager;
	}

	public IBasicVersioningManager getBasicVersioningManager() {
		return this.basicVersioningManager;
	}

	public IContentManager getContentManager() {
		return this.contentManager;
	}

	// ---------------------------------------------------------------------------
	// interface IRepositoryManager
	// ---------------------------------------------------------------------------

	public String getId() {
		return this.id;
	}

	public String getRidPrefix() {
		return this.prefix;
	}

	public NameInfo getNameInfo() {
		return AbstractManager.nameInfo;
	}

	public Map getSubManagers() {
		return Collections.unmodifiableMap(this.subManagers);
	}

	public IResourceEventBroker getEventBroker() {
		return this.eventBroker;
	}

	// ---------------------------------------------------------------------------
	// interface ILifecycleInfo
	// ---------------------------------------------------------------------------

	public final ComponentState getState() {
		return this.stateHandler.getState();
	}

	public final Date getCreationDate() {
		return this.stateHandler.getCreationDate();
	}

	// ---------------------------------------------------------------------------
	// interface IResourceEventSender
	// ---------------------------------------------------------------------------

	public IEventList getEvents() {
		if (eventsEnabled()) {
			return AbstractManager.collectionEvents;
		}
		else {
			return null;
		}
	}

	public IEventList getEvents(IResource resource) {
		if (!eventsEnabled()) {
			return null;
		}
		if (resource.isCollection()) {
			return AbstractManager.collectionEvents;
		}
		else {
			return AbstractManager.resourceEvents;
		}
	}

	// ---------------------------------------------------------------------------
	// interface IContainer
	// ---------------------------------------------------------------------------

	public IComponentManager getComponentManager() {
		return this.containerManager;
	}

	// ---------------------------------------------------------------------------
	// interface IComponentInfo
	// ---------------------------------------------------------------------------

	public String getName() {
		return this.id;
	}

	public String getDescription(Locale locale) {
		// Default implementation returns description in configuration data
		return "";
	}

	public Properties getProperties() {
		return null;
	}

	public ConfigurationException getLastConfigurationException() {
		return this.stateHandler.getLastConfigurationException();
	}

	public StartupException getStartupException() {
		return this.stateHandler.getStartupException();
	}

	public Date getLastReconfigurationDate() {
		return this.stateHandler.getLastReconfigurationDate();
	}

	public Date getNextAutoRestartDate() {
		return this.stateHandler.getNextAutoRestartDate();
	}

	/**
	 * ResourceImpl will call this method to check if a given resource event type
	 * must be sent at all.
	 * @param type the type of ResourceEvent to send
	 * @return true, if the given ResourceEvent type has to be sent,
	 *         false if no receiver registered for this ResourceEvent type
	 */
	public final boolean mustSendResourceEventType(int type) {
		return this.eventBroker.mustSendResourceEventType(type);
	}

	/**
	 * ResourceImpl will call this method to send the standard resource events
	 *
	 * @param type The event type (ResourceEvent.XXX constant)
	 * @param resource The resource the event belongs to
	 * @param param The event parameter
	 * @param correlationId TBD: Description of the incoming method parameter
	 * @return TBD: Description of the outgoing return value
	 * @todo activate modern ResourceEvent constructor when nemesis replaces
	 *      former framework
	 */
	public final IResourceEvent sendEvent(IResource resource, int type, String correlationId, Object param) {
		if (resource != null) {
			// @todo  activate modern ResourceEvent constructor when nemesis replaces former framework
			IResourceEvent event = new ResourceEvent(type, resource, param);
			// IResourceEvent event = new ResourceEvent( resource, type, correlationId, param );
			try {
				this.eventBroker.send(event, this);
				return event;
			}
			catch (WcmException ex) {
				logger.errorT(
					"sendEvent(445)",
					"Failed to send event "
						+ event.getDescription()
						+ " on correlated event "
						+ correlationId
						+ ": "
						+ ex.getMessage()
						+ ": "
						+ LoggingFormatter.extractCallstack(ex));
			}
		}
		return null;
	}

	/**
	 * Returns true if standard resource events should be sent by the framework.
	 *
	 * @return true if standard resource events should be sent by the framework.
	 */
	public final boolean eventsEnabled() {
		return this.eventsEnabled;
	}

	// ---------------------------------------------------------------------------
	// interface IConfigurable
	// ---------------------------------------------------------------------------

	public final void configure(IConfiguration config) throws ConfigurationException {
		this.stateHandler.preConfigure();
		this.config = config;
		try {
			this.id = config.getAttribute(ICmRuntimeConst.CONFIG_ATTRIBUTE_NAME.ID);
			this.prefix = config.getAttribute(AbstractManager.CFG.PREFIX);
			String value = config.getAttribute(AbstractManager.CFG.EVENTS, "false");
			this.eventsEnabled = new Boolean(value).booleanValue();

			synchronized (AbstractManager.class) {
				if (securityManagerMap == null) {
					securityManagerMap = new HashMap();
					try {
						// register security managers
						IConfigClientContext context = IConfigClientContext.createContext(ConfigCrutch.getConfigServiceUser());
						IConfigManager cfg = Configuration.getInstance().getConfigManager(context);
						IConfigPlugin plugin = cfg.getConfigPlugin(CFG_PLUGIN_CM_REPOSITORY_MANAGERS_SECURITY_MANAGERS);
						if (plugin == null) {
							throw new RuntimeException("missing plugin " + CFG_PLUGIN_CM_REPOSITORY_MANAGERS_SECURITY_MANAGERS);
						}

						IConfigurable[] configurables = plugin.getConfigurables();
						for (int i = 0; i < configurables.length; i++) {
							this.registerSecurityManager(
								configurables[i].getIdValue(),
								configurables[i].getPropertyValue(CFG_SEC_MGR_CLASS));
						}
					}
					catch (Exception e) {
						throw new ConfigurationException(e);
					}
				}
			}
		}
		catch (ConfigurationException x) {
			throw stateHandler.postConfigure(x);
		}
		this.stateHandler.postConfigure();
	}

	// ---------------------------------------------------------------------------
	// interface IStartable
	// ---------------------------------------------------------------------------

	public final void start() throws StartupException {
		this.stateHandler.preStart();

		// Instantiate a event broker as child component and register this repository as sender
		if (!this.started) {
			String crtKey = AbstractManager.EVENTBROKER_CRTKEY;
			try {
				IConfiguration cfg = new DefaultConfiguration(IConfigConst.ELEMENTS.COMPONENT_CONFIG);
				this.containerManager.addChild(
					crtKey,
					AbstractManager.EVENTBROKER_CLASSNAME,
					null,
					cfg,
					new Object[] { this, parseValueList(this.config.getAttribute("services", ""), ",")});
				this.eventBroker = (IResourceEventBroker) this.containerManager.lookupComponent(crtKey);
			}
			catch (Exception ex) {
				logger.errorT("start(528)", "Failed to start manager: " + LoggingFormatter.extractCallstack(ex));
				throw this.stateHandler.postStart(
					new StartupException("Failed to create event broker: " + ex.getMessage(), ex));
			}

			// Create an instance for each sub-manager class and initialize them with a reference
			// to this manager.
			try {
				String cn = this.config.getAttribute(AbstractManager.CFG.NAMESPACE_MANAGER, null);
				if (cn != null) {
					this.assignSubManager(this.createSubManager(cn));
				}
				cn = this.config.getAttribute(AbstractManager.CFG.CONTENT_MANAGER, null);
				if (cn != null) {
					this.assignSubManager(this.createSubManager(cn));
				}
				cn = this.config.getAttribute(AbstractManager.CFG.PROPERTY_MANAGER, null);
				if (cn != null) {
					this.assignSubManager(this.createSubManager(cn));
				}
				cn = this.config.getAttribute(AbstractManager.CFG.LOCK_MANAGER, null);
				if (cn != null) {
					this.assignSubManager(this.createSubManager(cn));
				}
				cn = this.config.getAttribute(AbstractManager.CFG.VERSIONING_MANAGER, null);
				if (cn != null) {
					this.assignSubManager(this.createSubManager(cn));
				}
				cn = this.config.getAttribute(AbstractManager.CFG.SECURITY_MANAGER, null);
				if (cn != null) {
					cn = (String) securityManagerMap.get(cn);
					if (cn == null) {
						throw new StartupException("Unknown security manager");
					}
					this.assignSubManager(this.createSubManager(cn));
				}
				cn = this.config.getAttribute(AbstractManager.CFG.PROPSEARCH_MANAGER, null);
				if (cn != null) {
					this.assignSubManager(this.createSubManager(cn));
				}
			}
			catch (Exception ex) {
				logger.errorT("start(569)", "Failed to start manager: " + LoggingFormatter.extractCallstack(ex));
				throw this.stateHandler.postStart(new StartupException("Failed to init sub-manager: " + ex.getMessage(), ex));
			}
		} // if (!this.started)

		this.started = true;
		try {
			this.startUpImpl();
		}
		catch (ConfigurationException ex) {
			StartupException x = this.stateHandler.postStart(ex);
			this.logStartUpTime();
			throw x;
		}
		catch (StartupException ex) {
			StartupException x = this.stateHandler.postStart(ex);
			this.logStartUpTime();
			throw x;
		}
		catch (Exception ex) {
			StartupException x =
				this.stateHandler.postStart(
					new StartupException(this.getId() + ": runtime exception during start up: " + ex.getMessage(), ex));
			this.logStartUpTime();
			throw x;
		}

		// Now call startUp() methof of sub-managers
		// startup of this manager must fail if one sub-manager fails.
		// (sub-managers are NOT components and are not visible in the CRT).
		try {
			if (this.securityManager != null) {
				((AbstractSubManager) this.securityManager).startUp();
			}
			if (this.namespaceManager != null) {
				((AbstractSubManager) this.namespaceManager).startUp();
			}
			if (this.contentManager != null) {
				((AbstractSubManager) this.contentManager).startUp();
			}
			if (this.propertyManager != null) {
				((AbstractSubManager) this.propertyManager).startUp();
			}
			if (this.lockManager != null) {
				((AbstractSubManager) this.lockManager).startUp();
			}
			if (this.basicVersioningManager != null) {
				((AbstractSubManager) this.basicVersioningManager).startUp();
			}
		}
		catch (BaseException ex) {
			StartupException x =
				this.stateHandler.postStart(
					new StartupException("Exception during start up of sub-manager: " + ex.getMessage(), ex));
			this.logStartUpTime();
			throw x;
		}
		catch (Exception ex) {
			StartupException x =
				this.stateHandler.postStart(
					new StartupException("Runtime exception during startup of sub-manager: " + ex.getMessage(), ex));
			this.logStartUpTime();
			throw x;
		}

		this.stateHandler.postStart();
		this.logStartUpTime();
	}

	public final void stop() {
		this.stateHandler.preStop();
		this.shutDownImpl();
		this.stateHandler.postStop();
	}

	// ---------------------------------------------------------------------------
	// interface IContextualizable
	// ---------------------------------------------------------------------------

	public void contextualize(IContext context) throws ContextException {
		this.stateHandler.preContextualize();
		this.containerManager = context.getContainerManager();
		this.key = context.getKey();
		if (this.containerManager == null) {
			throw new ContextException("No container manager in context");
		}
		this.stateHandler.postContextualize();
	}

	// ---------------------------------------------------------------------------

	/**
	 * The implementation must overwrite this method if it has initial startup
	 * handling to do (acquire resources, open connections to remote systems,
	 * etc). Accoring to the contract of the {@link
	 * com.sapportals.wcm.crt.component.IStartable} interface of the CRT this
	 * method will be called only once. It will be called on systen start up or
	 * when a new instance of this component is added to the persistence
	 * configuration.
	 *
	 * @exception ConfigurationException
	 * @exception StartupException
	 */
	protected void startUpImpl() throws ConfigurationException, StartupException {
		logger.infoT(
			"startUpImpl(691)",
			"The manager implementation does not overwrite the startUpImpl() method: " + this.id);
	}

	/**
	 * The implementation must overwrite this method if it has to release
	 * resources acquired during shutdown. Accoring to the contract of the {@link
	 * com.sapportals.wcm.crt.component.IStartable} interface of the CRT this
	 * method will be called only once. It will be called when the component is
	 * removed from the persistent configuration or when the system shuts down.
	 */
	protected void shutDownImpl() {
	}

	/**
	 * Helper method for reconfigure() implementations. Throws a configuration
	 * exception if any of the specified attributes or any of the not
	 * reconfigurable framework attributes (e.g. prefix) has changed.
	 *
	 * @param notReconfigurable An array of names of attributes that cannot be
	 *      canged wihout restarting the component/server.
	 * @param newConfig The new configuration data received as argument of the
	 *      reconfigure() method.
	 * @exception ConfigurationException
	 */
	protected final void checkNotReconfigurableAttributes(String[] notReconfigurable, IConfiguration newConfig)
		throws ConfigurationException {
		this.checkNotReconfigurableAttributeNames(AbstractManager.NOT_RECONFIGURABLE, newConfig);
		this.checkNotReconfigurableAttributeNames(notReconfigurable, newConfig);
	}

	protected void registerSecurityManager(String name, String manager) {
		securityManagerMap.put(name, manager);
	}

	private void checkNotReconfigurableAttributeNames(String[] names, IConfiguration newConfig)
		throws ConfigurationException {
		for (int i = 0; i < names.length; i++) {
			String name = names[i];
			String newValue = newConfig.getAttribute(name, null);
			String oldValue = this.config.getAttribute(name, null);
			boolean changed =
				(newValue == null && oldValue != null || newValue != null && oldValue == null)
					|| (newValue != null && oldValue != null && !newValue.equals(oldValue));
			if (changed) {
				throw new ConfigurationException("Property cannot be changed at runtime: " + name);
			}
		}
	}

	private Object createSubManager(String className) throws StartupException {
		Object sm = null;
		try {
			java.lang.reflect.Constructor ctor =
				CrtClassLoaderRegistry.forName(className).getConstructor(new Class[] { IManager.class });
			sm = ctor.newInstance(new Object[] { this });
		}
		catch (Exception ex) {
			throw new StartupException(
				"Exception loading and instanciating sub-manager: "
					+ className
					+ " ("
					+ ex.getClass().getName()
					+ ": "
					+ ex.getMessage()
					+ ")",
				ex);
		}
		if (!(sm instanceof AbstractSubManager)) {
			throw new StartupException("Sub-manager class is not an instance of AbstractRepositorySubManager: " + className);
		}
		return sm;
	}

	private void assignSubManager(Object o) {
		if (this.namespaceManager == null && o instanceof INamespaceManager) {
			this.namespaceManager = (INamespaceManager) o;
		}
		if (this.contentManager == null && o instanceof IContentManager) {
			this.contentManager = (IContentManager) o;
		}
		if (this.propertyManager == null && o instanceof IPropertyManager) {
			this.propertyManager = (IPropertyManager) o;
		}
		if (this.lockManager == null && o instanceof ILockManager) {
			this.lockManager = (ILockManager) o;
		}
		if (this.basicVersioningManager == null && o instanceof IBasicVersioningManager) {
			this.basicVersioningManager = (IBasicVersioningManager) o;
		}
		if (this.securityManager == null && o instanceof ISecurityManager) {
			this.securityManager = (ISecurityManager) o;
		}
	}

  /**
   * Must be called by the RM implementation in its reconfigure() method if reconfiguration of security managers at runtime must be supported.
   * @param newConfig The new configuration data (paramater of the reconfigure() method).
   * @return true if the security manager was changed (removed, added or different sub manager), false otherwise.
   * @exception ConfigurationException If the new sub manager can not be started.
   *                                   If the sub manager implements other sub manager interfaces than ISecurityManager.
   *                                   If the sub manager does not implement ISecurityManager.
   *                                   If the sub manager does not extend AbstractRepositorySubManager.
   */
  protected final boolean reconfigureSecurityManager(IConfiguration newConfig) throws ConfigurationException {
    String id = newConfig.getAttribute(AbstractManager.CFG.SECURITY_MANAGER, null);
    String smClassName = (String) AbstractManager.securityManagerMap.get(id);

    if (smClassName == null) {
      // Security manager removed
      try {
        if (this.securityManager != null) {
          ((AbstractSubManager) this.securityManager).shutDown();
        }
        // Note: SM implementation must ensure that clients which still hold a reference
        // to this instance will receive a apropriate exception.
      }
      catch (RuntimeException ex) {
        AbstractManager.logger.errorT(LoggingFormatter.extractCallstack(ex));
      }
      this.securityManager = null;
      // Update sec. checker (will assign a NullChecker instance if no security manager is present)
      /*
      try {
        this.securityChecker = SecurityChecker.getInstance(this);
      }
      catch (ResourceException e) {
        AbstractRepositoryManager.log.errorT(LoggingFormatter.extractCallstack(e));
      }
      */
      AbstractManager.logger.infoT("security manager removed for repository: " + this.getId());
      return true;
    }
    else {
      String oldId = this.config.getAttribute(AbstractManager.CFG.SECURITY_MANAGER, null);
      if (oldId != null && oldId.equals(id)) {
        AbstractManager.logger.debugT("security manager not changed. old and new id are equal: " + id);
        return false;
      }

      Object smInstance = null;
      try {
        smInstance = this.createSubManager(smClassName);
      }
      catch (Exception ex) {
        throw new ConfigurationException(ex.getMessage(), ex);
      }

      if (!(smInstance instanceof ISecurityManager)) {
        throw new ConfigurationException("class does not implement ISecurityManager: " + smClassName);
      }
      /*
      if (smInstance instanceof INamespaceManager
        || smInstance instanceof IContentManager
        || smInstance instanceof IPropertyManager
        || smInstance instanceof ILockManager
        || smInstance instanceof IVersioningManager
        || smInstance instanceof IPropertySearchManager
        || smInstance instanceof IExtendedNamespaceManager
        || smInstance instanceof IExtendedVersioningManager) {
        throw new ConfigurationException(
          "class must not implement other sub manager interfaces other than ISecurityManager: " + smClassName);
      }
      */
      
      try {
        ((AbstractSubManager) smInstance).startUp();
      }
      catch (BaseException ex) {
        throw new ConfigurationException("Failed to start security manager: " + ex.getMessage(), ex);
      }
      catch (RuntimeException ex) {
        AbstractManager.logger.errorT(LoggingFormatter.extractCallstack(ex));
        throw new ConfigurationException("caught runtime exception in startUp() method of security manager: " + id);
      }

      // Note: Until now an existing SM instance remains up and running in case an exception is thrown in the code above
      boolean changed = (this.securityManager != null);
      if (changed) {
        // Security manager changed: Shut down current instance
        try {
          ((AbstractSubManager) this.securityManager).shutDown();
        }
        catch (RuntimeException ex) {
          AbstractManager.logger.errorT(LoggingFormatter.extractCallstack(ex));
        }
      }

      this.securityManager = (ISecurityManager) smInstance;
      /*
      try {
        this.securityChecker = SecurityChecker.getInstance(this);
      }
      catch (ResourceException e) {
        AbstractRepositoryManager.log.errorT(LoggingFormatter.extractCallstack(e));
      }
      */
      
      // @todo This is relevant for security audit logs ??
      AbstractManager.logger.infoT(
        "security manager "
          + (changed ? "changed" : "set")
          + " for repository: "
          + this.getId()
          + ", old ID = "
          + (oldId == null ? "none" : oldId)
          + ", new ID = "
          + (id == null ? "none" : id));
      return true;
    }
  }


	private List parseValueList(String values, String separator) {
		List list = new LinkedList();
		if (values == null) {
			return list;
		}
		StringTokenizer st = new StringTokenizer(values, separator);
		while (st.hasMoreTokens()) {
			list.add(st.nextToken().trim());
		}
		return list;
	}

	private void logStartUpTime() {
		long time = this.stateHandler.getConfigurationTime() + this.stateHandler.getStartTime();
		double sec = (double) time / 1000.0;
		logger.debugT("logStartUpTime(838)", "Startup of repository manager " + this.getId() + " took " + sec + " seconds");
	}

	private interface CFG {
		String PREFIX = "prefix";
		String EVENTS = "sendevents";
		String SERVICES = "services";
		String NAMESPACE_MANAGER = "namespacemgr.class";
		String PROPERTY_MANAGER = "propertymgr.class";
		String LOCK_MANAGER = "lockmgr.class";
		String VERSIONING_MANAGER = "versioningmgr.class";
		String CONTENT_MANAGER = "contentmgr.class";
		String SECURITY_MANAGER = "securitymgr.ref";
		String PROPSEARCH_MANAGER = "propertysearchmgr.class";
	} // contains Strings

	static {
		AbstractManager.resourceEvents.add(ResourceEvent.CHECKIN_TEMPLATE);
		AbstractManager.resourceEvents.add(ResourceEvent.CHECKOUT_TEMPLATE);
		AbstractManager.resourceEvents.add(ResourceEvent.COPY_TEMPLATE);
		AbstractManager.resourceEvents.add(ResourceEvent.DELETE_TEMPLATE);
		AbstractManager.resourceEvents.add(ResourceEvent.GET_TEMPLATE);
		AbstractManager.resourceEvents.add(ResourceEvent.LOCK_TEMPLATE);
		AbstractManager.resourceEvents.add(ResourceEvent.MOVE_TEMPLATE);
		AbstractManager.resourceEvents.add(ResourceEvent.PROPERTY_DELETE_TEMPLATE);
		AbstractManager.resourceEvents.add(ResourceEvent.PROPERTY_SET_TEMPLATE);
		AbstractManager.resourceEvents.add(ResourceEvent.PROPERTY_GET_TEMPLATE);
		AbstractManager.resourceEvents.add(ResourceEvent.RENAME_TEMPLATE);
		AbstractManager.resourceEvents.add(ResourceEvent.SET_TEMPLATE);
		AbstractManager.resourceEvents.add(ResourceEvent.UNDO_CHECKOUT_TEMPLATE);
		AbstractManager.resourceEvents.add(ResourceEvent.UNLOCK_TEMPLATE);
		AbstractManager.collectionEvents.add(ResourceEvent.CHECKIN_TEMPLATE);
		AbstractManager.collectionEvents.add(ResourceEvent.CHECKOUT_TEMPLATE);
		AbstractManager.collectionEvents.add(ResourceEvent.COPY_TEMPLATE);
		AbstractManager.collectionEvents.add(ResourceEvent.DELETE_TEMPLATE);
		AbstractManager.collectionEvents.add(ResourceEvent.GET_TEMPLATE);
		AbstractManager.collectionEvents.add(ResourceEvent.LOCK_TEMPLATE);
		AbstractManager.collectionEvents.add(ResourceEvent.MOVE_TEMPLATE);
		AbstractManager.collectionEvents.add(ResourceEvent.PROPERTY_DELETE_TEMPLATE);
		AbstractManager.collectionEvents.add(ResourceEvent.PROPERTY_SET_TEMPLATE);
		AbstractManager.collectionEvents.add(ResourceEvent.PROPERTY_GET_TEMPLATE);
		AbstractManager.collectionEvents.add(ResourceEvent.RENAME_TEMPLATE);
		AbstractManager.collectionEvents.add(ResourceEvent.SET_TEMPLATE);
		AbstractManager.collectionEvents.add(ResourceEvent.UNDO_CHECKOUT_TEMPLATE);
		AbstractManager.collectionEvents.add(ResourceEvent.UNLOCK_TEMPLATE);
		AbstractManager.collectionEvents.add(ResourceEvent.CREATE_CHILD_TEMPLATE);
		AbstractManager.collectionEvents.add(ResourceEvent.CREATE_COLLECTION_TEMPLATE);
		AbstractManager.collectionEvents.add(ResourceEvent.CREATE_LINK_TEMPLATE);
		AbstractManager.collectionEvents.add(ResourceEvent.GET_CHILDREN_TEMPLATE);
	}

  public String toString() {
    return this.getId();
  }

  public final boolean equals(Object other) {
    if (other == null) {
      return false;
    }       
    if (!(other instanceof AbstractManager)) {
      return false;
    }
    return ((AbstractManager)other).getId().equals(this.getId());
  }
  
  public final int hashCode() {
    return this.getId().hashCode();   
  } 
}
