package com.sapportals.wcm.service.resourceTypeRegistry;

/**
 * Title:        Resource Type Registry
 * Description:  Resource Type Registry
 * Component:    Repository Framework
 * Copyright:    (c) 2003, 2004
 * Company:      SAP AG, Walldorf, Germany
 *               All rights reserved.
 * @author       Heiko Kiessling
 * @version      $Id: //kmgmt/bc.rf.global.service/60NW_SP_COR/src/_rtr/java/api/com/sapportals/wcm/service/resourceTypeRegistry/IResourceTypeRegistryService.java#5 $
 */

import java.util.Locale;

import com.sapportals.wcm.WcmException;

import com.sapportals.wcm.repository.IResource;
import com.sapportals.wcm.util.uri.RID;
import com.sapportals.wcm.repository.IResourceContext;

import com.sapportals.wcm.service.IService;

/**
 * <p>
 *   Provides a service to register properties such as language-dependent short
 *   descriptions or links to icons for resource types.
 * </p>
 */
public interface IResourceTypeRegistryService extends IService {

  /**
   * Returns the resource type with the specified identifier. If the identifier
   * is <code>null</code> or empty, the resource type configured as normal
   * resource type is returned. If the identifier is unknown,
   * {@link IResourceType#UNKNOWN_RESOURCE_TYPE} is returned. As opposed to
   * {@link #get(String, boolean)}, it is not checked whether the specified
   * resource type describes a plain resource or a collection. 
	 * @param  typeId Identifier of the resource type
	 * @return Resource type
	 */
	IResourceType get( String typeId );

  /**
   * Returns the resource type with the specified identifier, but limited to
   * either plain resource or collection types, as opposed to
   * {@link #get(String)}. That is, a call of
   * {@link IResourceType#isCollection()} on the result is guaranteed to yield
   * the value of the parameter <code>isCollection</code>. If the identifier is
   * <code>null</code> or empty, the resource type configured as normal
   * resource type or normal collection type is returned, depending on the
   * value of the parameter <code>isCollection</code>. If the identifier,
   * limited to either plain resource or collection types, is unknown,
   * {@link IResourceType#UNKNOWN_RESOURCE_TYPE} or
   * {@link IResourceType#UNKNOWN_COLLECTION_TYPE} is returned, depending on
   * the value of the parameter <code>isCollection</code>. 
   * @param  typeId       Identifier of the resource type
   * @param  isCollection <code>true</code> iff a collection type is requested
   * @return Resource type
   */
  IResourceType get( String typeId, boolean isCollection );

  /**
   * Returns the resource type of the specified resource. This method is
   * equivalent to calling {@link #get(String, boolean)} with
   * <code>resource.getResourceType()</code> and
   * <code>resource.isCollection()</code> as arguments.
   * @param  resource Resource
   * @return Resource type
   * @deprecated as of NW04. Renamed because of problems with overloading resolution, use
   *             {@link #getViaResource(IResource)} instead. 
   */
  IResourceType get( IResource resource );

  /**
   * Returns the resource type of the specified resource. This method is
   * equivalent to calling {@link #get(String, boolean)} with
   * <code>resource.getResourceType()</code> and
   * <code>resource.isCollection()</code> as arguments.
   * @param  resource Resource
   * @return Resource type
   */
  IResourceType getViaResource( IResource resource );

  /**
   * Returns the resource type of the specified resource. This method is
   * equivalent {@link #get(IResource)} but on demand resolves internal links before determining the resource type. 
   * @param  resource Resource
   * @param  resolveInternalLinks <code>true</code> iff internal links are to
   *                              be resolved before returning the resource
   *                              type
   * @return Resource type
   */
  IResourceType getViaResource( IResource resource, boolean resolveInternalLinks );

  /**
   * Returns all resource types, and an empty array if no resource types have
   * been registered.
   * @return All resource types
   */
  IResourceType[] getResourceTypes();

  /**
   * Returns either all plain resource or all collection types, depending on
   * the value of the parameter <code>isCollection</code>, and an empty array
   * if no such resource types have been registered.
   * @param  isCollection <code>true</code> iff collections types are requested
   * @return All plain resource or collection types, depending on the value of
   *         the parameter <code>isCollection</code>
   */
  IResourceType[] getResourceTypes( boolean isCollection );
  
}
