package com.sapportals.wcm.service.resourceTypeRegistry;

/**
 * Title:        Resource Type Registry
 * Description:  Resource Type Registry
 * Component:    Repository Framework
 * Copyright:    (c) 2003, 2004
 * Company:      SAP AG, Walldorf, Germany
 *               All rights reserved.
 * @author       Heiko Kiessling
 * @version      $Id: //kmgmt/bc.rf.global.service/60NW_SP_COR/src/_rtr/java/api/com/sapportals/wcm/service/resourceTypeRegistry/IResourceType.java#4 $
 */

import java.util.Locale;

import java.util.Map;
import java.util.HashMap;

import com.sapportals.wcm.util.uri.RID;
import com.sapportals.wcm.IWcmConst;

/**
 * <p>
 *   Describes a resource type with language-dependent short descriptions
 *   as well as links to icons.
 * </p>
 */
public interface IResourceType {
  
  final static IResourceType UNKNOWN_RESOURCE_TYPE   = new UnknownResourceType( IWcmConst.RESOURCE_TYPE_NAMESPACE.FRAMEWORK + IWcmConst.NAMESPACE_SEPARATOR + "unknownrt", false ),
                             UNKNOWN_COLLECTION_TYPE = new UnknownResourceType( IWcmConst.RESOURCE_TYPE_NAMESPACE.FRAMEWORK + IWcmConst.NAMESPACE_SEPARATOR + "unknownct", true );

  /**
   * Gets the resource type identifier.
   * @return Resource type identifier
   */
  String getTypeId();

  /**
   * Returns <code>true</code> iff this resource type is reserved for an
   * application, and not for general use.
   * @return <code>true</code> iff reserved for application
   */
  boolean isApplicationProtected();
  
  /**
   * Gets the name of the icon of this resource type.
   * @return Icon RID
   */
  String getIcon();

  /**
   * Gets the short description of this resource type for its default locale.
   * @return Description
   */
  String getDescription();

  /**
   * Gets the short description of this resource type for the specified locale.
   * @param  locale Locale
   * @return Description
   */
  String getDescription( Locale locale );
  
  /**
   * Returns <code>true</code> iff this resource type is a collection.
   * @return <code>true</code> iff collection
   */
  boolean isCollection();
  
  /**
   * Gets the custom properties of this resource type as a single string.
   * @return Properties
   */
  String getProperties();

  /**
   * Gets the custom properties of this resource type as a map, using
   * <code>','</code> as a separator. This method is equivalent to calling
   * {@link #getPropertiesAsMap(char)} with <code>','</code> as argument. 
   * @return Properties
   */
  Map getPropertiesAsMap();

  /**
   * Gets the custom properties of this resource type, using a specified
   * character as separator. The thus parsed list of properties is mapped in
   * accordance with the <code>java.util.Properties</code> format. The
   * separator can be escaped with a preceding <code>'\'</code>, and must not
   * be <code>'='</code>, <code>':'</code> or whitespace.  
   * @param  separator Character separating different properties
   * @return Properties
   * @throws IllegalArgumentException <code>'='</code>, <code>':'</code>, or
   *                                  whitespace used as separator
   */
  Map getPropertiesAsMap(char separator) throws IllegalArgumentException;
  
}

class UnknownResourceType implements IResourceType {
  
  private String  typeId;
  private boolean isCollection;
  
  final String EMPTY_PROPERTIES = "";
  final Map    EMPTY_MAP        = new HashMap();
  
  /**
   * Constructs an anonymous resource type from properties. This constructor
   * should be used for constructing internal resource types used to represent
   * unsuccessful queries only.
   * @param typeId       Resource type identifier
   * @param isCollection <code>true</code> iff collection type
   */
  UnknownResourceType( String typeId, boolean isCollection ) {
    this.typeId       = typeId.trim();
    this.isCollection = isCollection;
  }

  public String getTypeId() {
    return typeId;
  }
  
  public boolean isApplicationProtected() {
    return false;
  }
  
  public String getIcon() {
    return null;
  }

  public String getDescription() {
    return getDescription( Locale.getDefault() );
  }

  public String getDescription( Locale locale ) {
    return getTypeId();
  }
  
  public boolean isCollection() {
    return isCollection;
  }
  
  public String getProperties() {
    return EMPTY_PROPERTIES;
  }
  
  public Map getPropertiesAsMap() {
    return getPropertiesAsMap( ',' );
  }

  public Map getPropertiesAsMap( char separator ) throws IllegalArgumentException {
    return EMPTY_MAP;
  }

  public String toString() {
    return getTypeId();
  }
}
