/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id: //kmgmt/bc.sf/60NW_SP_COR/src/_framework/java/api/com/sapportals/wcm/service/ServiceFactory.java#2 $
 */

package com.sapportals.wcm.service;

import com.sap.tc.logging.Location;

import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.crt.CrtClassLoaderRegistry;
import com.sapportals.wcm.util.logging.LoggingFormatter;
import java.lang.reflect.*;

import java.util.*;

/**
 * <p>
 *
 * <!--COPYRIGHT TAG:-->Copyright 2004 SAP AG<!-- -->
 *
 * @author m.breitenfelder@sapportals.com
 * @version $Id: //kmgmt/bc.sf/dev/src/_framework/java/api/com/sapportals/wcm/service/ServiceFactory.java#5
 *      $
 */
public class ServiceFactory implements IServiceFactory {

  private final static String CM_SYSTEM_CLASS = "com.sapportals.wcm.repository.runtime.CmAdapter";

  private static Location LOC = Location.getLocation(com.sapportals.wcm.service.ServiceFactory.class);

  private static ServiceFactory instance;

  private static Class cmAdapterClass;
  private static Object cmAdapterInstance;
  private static Method methodGetService;
  private static Method methodGetAllServices;
  private static Method methodReleaseComponent;

  private ServiceFactory() { }

  /**
   * Returns the ServiceFactory instance
   *
   * @return instance
   * @exception WcmException Exception raised in failure situation
   */
  public static synchronized ServiceFactory getInstance()
    throws WcmException {
    if (ServiceFactory.instance == null) {
      ServiceFactory.instance = new ServiceFactory();

      // Call CmSystem methods via reflection because service API cannot import
      // com.sapportals.wcm.repository.runtime - cyclic dependency in Ant !
      try {
        cmAdapterClass = CrtClassLoaderRegistry.forName(ServiceFactory.CM_SYSTEM_CLASS);
        methodGetService = cmAdapterClass.getMethod("getService", new Class[]{String.class});
        methodGetAllServices = cmAdapterClass.getMethod("getAllServices", null);
        methodReleaseComponent = cmAdapterClass.getMethod("releaseService", new Class[]{IService.class});
        cmAdapterInstance = cmAdapterClass.getMethod("getInstance", null).invoke(null, null);
      }
      catch (Throwable x) {
        x.printStackTrace(System.err);
        LOC.fatalT("getInstance(76)", "Failed to initialize the ServiceFactory: " + LoggingFormatter.extractCallstack(x));
        throw new WcmException("Exception accessing CmSystem: " + x.getMessage(), x);
      }
    }

    return ServiceFactory.instance;
  }

  public IService getService(String type)
    throws WcmException {
    try {
      return (IService)ServiceFactory.methodGetService.invoke(cmAdapterInstance, new Object[]{type});
    }
    catch (Exception x) {
      LOC.warningT("getService(90)", "Failed to get service of type " + type + " : " + LoggingFormatter.extractCallstack(x));
      throw new WcmException(x.getMessage(), x);
    }
  }

  public boolean isAvailable(String type)
    throws WcmException {
    return (this.getService(type) != null);
  }

  public Collection getAllServices()
    throws WcmException {
    try {
      return (Collection)ServiceFactory.methodGetAllServices.invoke(cmAdapterInstance, null);
    }
    catch (Exception x) {
      LOC.warningT("getAllServices(106)", "Failed to get all services : " + LoggingFormatter.extractCallstack(x));
      throw new WcmException(x.getMessage(), x);
    }
  }

  public void releaseService(IService service)
    throws WcmException {
    try {
      ServiceFactory.methodReleaseComponent.invoke(cmAdapterInstance, new Object[]{service});
    }
    catch (Exception x) {
      LOC.warningT("releaseService(117)", "Failed to release service of type " + service.getID() + " : " + LoggingFormatter.extractCallstack(x));
      throw new WcmException(x.getMessage(), x);
    }
  }
}
