/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id: //kmgmt/bc.sf/60NW_SP_COR/src/_framework/java/api/com/sapportals/wcm/service/AbstractService.java#2 $
 */

package com.sapportals.wcm.service;
import com.sap.tc.logging.Location;

import com.sapportals.wcm.IWcmConst;
import com.sapportals.wcm.WcmException;

import com.sapportals.wcm.crt.component.*;
import com.sapportals.wcm.crt.configuration.*;
import com.sapportals.wcm.util.logging.LoggingFormatter;

import java.util.*;

/**
 * The abstract base class for global Services. <p>
 *
 * Copyright (c) SAP AG 2001 - 2002
 *
 * @author Markus Breitenfelder
 * @version $Id: //kmgmt/bc.sf/dev/src/_framework/java/api/com/sapportals/wcm/service/AbstractService.java#4
 *      $
 */
public abstract class AbstractService implements IService, IComponent,
  IConfigurable, IStartable, IComponentInfo, ILifecycleInfo {

  private static Location log = Location.getLocation(com.sapportals.wcm.service.AbstractService.class);

  // Note: Service API cannot import com.sapportals.wcm.repository.runtime.ICmRuntimeConst
  // Cyclic dependency in make !
  /**
   * TBD: Description of the class.
   */
  public interface CONFIG_ATTRIBUTE_NAME {
    String ID = "name";
    String DESCRIPTION = "description";
    String MANAGER_LIST = "service_manager_list";
  }

  private String id;
  private String description;

  /**
   * This service's configuration
   */
  protected IConfiguration config;

  /**
   * This service's configuration converted to a Properties object. This will be
   * null unless the manager implementation calls the conversion method in
   * startUp !!
   */
  protected Properties properties;

  protected final ComponentStateHandler stateHandler = new ComponentStateHandler(this);

  /**
   * The default constructor
   */
  public AbstractService() { }


  /**
   * @param properties TBD: Description of the incoming method parameter
   * @exception WcmException Exception raised in failure situation
   * @deprecated As of EP 5.0 SP5, replaced by default (public, no arguments)
   *      constructor.
   */
  public AbstractService(Properties properties)
    throws WcmException {
    if (properties != null) {
      this.properties = (Properties)(properties.clone());
    }
    this.id = (String)this.properties.get(CONFIG_ATTRIBUTE_NAME.ID);
  }


  // ---------------------------------------------------------------------------
  // interface IService
  // ---------------------------------------------------------------------------

  public final String getID() {
    return this.id;
  }

  public String getDescription() {
    return this.description;
  }

  /**
   * @return type
   * @deprecated As of EP 5.0 SP5, The "service type" is the name of the
   *      configurable class.
   */
  public String getType() {
    return this.id;
  }

  // ---------------------------------------------------------------------------
  // interface IConfigurable
  // ---------------------------------------------------------------------------

  public final void configure(IConfiguration config)
    throws ConfigurationException {
    this.stateHandler.preConfigure();

    try {
      this.config = config;
      this.id = config.getAttribute(AbstractService.CONFIG_ATTRIBUTE_NAME.ID);
      this.description = config.getAttribute(AbstractService.CONFIG_ATTRIBUTE_NAME.DESCRIPTION, null);
    }
    catch (ConfigurationException x) {
      throw this.stateHandler.postConfigure(x);
    }
    this.stateHandler.postConfigure();
  }

  // ---------------------------------------------------------------------------
  // interface IStartable
  // ---------------------------------------------------------------------------


  public final void start()
    throws StartupException {
    this.stateHandler.preStart();

    try {
      this.startUpImpl();
    }
    catch (ConfigurationException ex) {
      StartupException x = this.stateHandler.postStart(ex);
      this.logStartUpTime();
      throw x;
    }
    catch (StartupException ex) {
      StartupException x = this.stateHandler.postStart(ex);
      this.logStartUpTime();
      throw x;
    }
    catch (Exception ex) {
      StartupException x = this.stateHandler.postStart(new StartupException(this.getID() + ": runtime exception during start up: " + ex.getMessage(), ex));
      this.logStartUpTime();
      throw x;
    }

    this.stateHandler.postStart();
    this.logStartUpTime();
  }


  public final void stop() {
    this.stateHandler.preStop();
    this.shutDownImpl();
    this.stateHandler.postStop();
  }


  // ---------------------------------------------------------------------------
  // interface IComponentInfo
  // ---------------------------------------------------------------------------

  public String getName() {
    return this.id;
  }

  public String getDescription(Locale locale) {
    // Default implementation returns description of config
    return this.getDescription();
  }

  public Properties getProperties() {
    Properties props = convertConfigToProperties(this.config, "config-");
    props.setProperty("type", "Service");
    return props;
  }

  // ---------------------------------------------------------------------------
  // interface ILifecycleInfo
  // ---------------------------------------------------------------------------

  public ComponentState getState() {
    return this.stateHandler.getState();
  }

  public ConfigurationException getLastConfigurationException() {
    return this.stateHandler.getLastConfigurationException();
  }

  public StartupException getStartupException() {
    return this.stateHandler.getStartupException();
  }

  public final Date getCreationDate() {
    return this.stateHandler.getCreationDate();
  }

  public Date getLastReconfigurationDate() {
    return this.stateHandler.getLastReconfigurationDate();
  }

  public Date getNextAutoRestartDate() {
    return this.stateHandler.getNextAutoRestartDate();
  }

  // ---------------------------------------------------------------------------

  protected void preService()
    throws WcmException {
    if (!this.stateHandler.preService()) {
      throw new WcmException("Service not available");
    }
  }

  protected void postService() {
    this.stateHandler.postService();
  }

  // ---------------------------------------------------------------------------

  /**
   * The implementation must overwrite this method if it has initial startup
   * handling to do (acquire resources, open connections to remote systems,
   * etc). Accoring to the contract of the {@link
   * com.sapportals.wcm.crt.component.IStartable} interface of the CRT this
   * method will be called only once. It will be called on systen start up or
   * when a new instance of this component is added to the persistence
   * configuration.
   *
   * @exception ConfigurationException Exception raised in failure situation
   * @exception StartupException Exception raised in failure situation
   */
  protected void startUpImpl()
    throws ConfigurationException, StartupException {
    try {
      this.startUp();
    }
    catch (WcmException x) {
      throw new StartupException(x);
    }
  }

  /**
   * The implementation must overwrite this method if it has to release
   * resources acquired during shutdown. Accoring to the contract of the {@link
   * com.sapportals.wcm.crt.component.IStartable} interface of the CRT this
   * method will be called only once. It will be called when the component is
   * removed from the persistent configuration or when the system shuts down.
   */
  protected void shutDownImpl() {
    try {
      this.shutDown();
    }
    catch (WcmException x) {
      AbstractService.log.errorT("shutDownImpl(265)", "shutDown failed: " + x.getMessage() + ": " + LoggingFormatter.extractCallstack(x));
    }
  }


  /**
   * The implementation must overwrite this method if it has initial startup
   * handling to do (acquire resources, open connections to remote systems,
   * etc).
   *
   * @exception WcmException Exception raised in failure situation
   * @deprecated As of EP 5.0 SP5, replaced by {@link #shutDownImpl()}.
   */
  protected void startUp()
    throws WcmException {
    AbstractService.log.infoT("startUp(280)", "The service implementation does not overwrite the startUpImpl() method: " + this.id);
  }


  /**
   * The implementation must overwrite this method if it has to release
   * resources acquired during shutdown.
   *
   * @exception WcmException Exception raised in failure situation
   * @deprecated As of EP 5.0 SP5, replaced by {@link #startUpImpl()}.
   */
  protected void shutDown()
    throws WcmException {
    AbstractService.log.infoT("shutDown(293)", "The service implementation does not overwrite the shutDownImpl() method: " + this.id);
  }


  /**
   * Helper method for conversion of configuration data into properties.
   *
   * @param config which attributes should be converted to properties
   * @return converted config as properties
   */
  public final static Properties convertConfigToProperties(IConfiguration config) {
    return convertConfigToProperties(config, null);
  }

  /**
   * Helper method for conversion of configuration data into properties.
   *
   * @param config which attributes should be converted to properties
   * @param prefix to prepend to config attribute names
   * @return converted config as properties
   */
  protected final static Properties convertConfigToProperties(IConfiguration config, String prefix) {
    Properties props = new Properties();
    String[] attributeNames = config.getAttributeNames();
    for (int i = 0; i < attributeNames.length; i++) {
      String name = attributeNames[i];
      String value = config.getAttribute(name, null);
      if (value != null) {
        if (prefix != null) {
          name = prefix + name;
        }
        props.setProperty(name, value);
      }
    }
    return props;
  }


  private void logStartUpTime() {
    long time = this.stateHandler.getConfigurationTime() + this.stateHandler.getStartTime();
    double sec = (double)time / 1000.0;
    AbstractService.log.debugT("logStartUpTime(320)", "Startup of service " + this.getID()
       + " took " + sec + " seconds");
  }
}

