/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.events;
import com.sap.tc.logging.Location;
import com.sapportals.wcm.IFrameworkTransaction;
import com.sapportals.wcm.util.logging.LoggingFormatter;

/**
 * ReceiverEntry contains a receiver, the template, its mode and collected
 * events. Every asynchronous receiver gets its own event queue and sender
 * thread.
 */
public class ReceiverEntry {

  private static Location log = Location.getLocation(ReceiverEntry.class);

	private Receiver receiver;
  private IEvent template;
  private IEventMapper mapping;
  private IEventList events;
  private SenderMode state;
  private boolean collect;
  private boolean async;
  private int priority;

  ReceiverEntry(Receiver receiver, IEvent template, IEventMapper mapping, int priority, boolean async) {
    this.receiver = receiver;
    this.template = template;
    this.mapping = mapping;
    this.events = new EventList();
    this.state = SenderMode.SENDING;
    this.collect = false;
    this.async = async;
    this.priority = priority;
  }

  public boolean isAsync() {
    return this.async;
  }

  /**
   * Get priority
   *
   * @return priority
   */
  int getPriority() {
    return this.priority;
  }

  /**
   * get the receiver
   *
   * @return receiver
   */
  IEventReceiver getReceiver() {
    return this.receiver.getReceiver();
  }

  /**
   * get the event template for filtering
   *
   * @return template
   */
  IEvent getTemplate() {
    return this.template;
  }

  /**
   * Overwrite toString() for debugging outputs
   *
   * @return TBD: Description of the outgoing return value
   */
  public String toString() {
    if (this.template == null) {
      return this.receiver.toString() + ":null";
    }
    else {
      return this.receiver.toString() + ":" + this.template.getDescription();
    }
  }

  /**
   * overwrite equal for list-search and contains()
   *
   * @param other TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  public boolean equals(Object other) {
    if (other instanceof ReceiverEntry) {
      return (
        (this.receiver == ((ReceiverEntry)other).receiver)
          && ((this.template == null) || (this.template.isLike(((ReceiverEntry)other).template))));
    }
    else {
      return false;
    }
  }
  
  /**
   * call a receivers received() function or enqueue event
   *
   * @param event TBD: Description of the incoming method parameter
   */
  public boolean doSend(IFrameworkTransaction ft, IEvent event) {
    IEventReceiver rec = this.receiver.getReceiver();
    if (! (rec instanceof ITransactionalEventReceiver)) return true; 
        
    boolean ok = true;
    // Event mapping stuff...
    IEvent sendEvent = null;
    if (this.mapping == null) {
      sendEvent = event;
    }
    else {
      try {
        sendEvent = this.mapping.map(event);
      }
      catch (Throwable t) {
        ReceiverEntry.log.errorT(
          "doSend(146)",
          "Exception in event mapper: "
            + t.getMessage()
            + " - "
            + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(t));
      }
    }
    if (sendEvent != null) {
      try{
        ok = ((ITransactionalEventReceiver)rec).received(ft, sendEvent);
      }
      catch (Throwable t) {
        ok = false;
        ReceiverEntry.log.errorT(
          "doSend(158)",
          "Exception in event handler: "
            + t.getMessage()
            + " - "
            + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(t));
      } 
    }
    return ok;
  }  


  /**
   * call a receivers received() function or enqueue event
   *
   * @param event TBD: Description of the incoming method parameter
   */
  void doSend(IEvent event) {

    // Event mapping stuff...
    IEvent sendEvent = null;
    if (this.mapping == null) {
      sendEvent = event;
    }
    else {
      try {
        sendEvent = this.mapping.map(event);
      }
      catch (Throwable t) {
        ReceiverEntry.log.errorT(
          "doSend(146)",
          "Exception in event mapper: "
            + t.getMessage()
            + " - "
            + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(t));
      }
    }
    if (sendEvent == null) {
      return;
    }

    if (!this.async) {
      try {
        this.receiver.getReceiver().received(sendEvent);
      }
      catch (Throwable t) {
        ReceiverEntry.log.errorT(
          "doSend(158)",
          "Exception in event handler: "
            + t.getMessage()
            + " - "
            + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(t));
      }
      if (log.beDebug()) {
        ReceiverEntry.log.debugT("doSend(161)", "received() method returned: " + receiver + ", event=" + sendEvent);
      }
    }
    else {
      // Asynchronous events: put into queue
      if (!this.receiver.getQueue().enqueue(sendEvent)) {
        ReceiverEntry.log.errorT(
          "doSend(167)",
          "Failed to enqueue event: "
            + sendEvent.getDescription()
            + "for receiver "
            + this.receiver
            + " - "
            + LoggingFormatter.extractCallstack(new Exception()));
      }
    }
  }

  /**
   * send the events in the hold list
   */
  void sendHoldEvents() {
    IEventListIterator iterator = this.events.listIterator();
    IEvent event;
    while (iterator.hasNext()) {
      doSend(iterator.next());
    }
    this.events = new EventList();
    this.collect = false;
  }

  void setSendHoldEvents(boolean collect) {
    this.collect = collect;
  }

  /**
   * add an event to the list of hold events
   *
   * @param event TBD: Description of the incoming method parameter
   */
  void holdEvent(IEvent event) {

    IEvent sendEvent;
    if (this.mapping == null) {
      sendEvent = event;
    }
    else {
      sendEvent = this.mapping.map(event);
    }
    if (sendEvent == null) {
      return;
    }

    this.events.addLast(sendEvent);

  }

  /**
   * get the list of hold events
   *
   * @return holdEvents
   */
  IEventList getHoldEvents() {
    return this.events;
  }

  /**
   * get the list of hold events and clear it
   *
   * @return TBD: Description of the outgoing return value
   */
  IEventList clearHoldEvents() {
    IEventList events = this.events;
    this.events = new EventList();
    return events;
  }

  void setState(SenderMode state) {
    boolean sendhold = ((this.state.equals(SenderMode.HOLD)) && (state.equals(SenderMode.SENDING)) && (this.collect));
    /*
     * SENDING   -> SENDING    => SENDING
     * SUSPEND   -> SENDING    => SENDING
     * HOLD      -> SENDING    => SENDING
     * SENDING   -> SUSPEND    => SUSPEND
     * SUSPEND   -> SUSPEND    => SUSPEND
     * HOLD      -> SUSPEND    => SUSPEND
     * SENDING   -> HOLD       => HOLD
     * SUSPEND   -> HOLD       => HOLD
     * HOLD      -> HOLD       => HOLD
     */
    this.state = state;
    if (sendhold) {
      sendHoldEvents();
    }
  }

  SenderMode getState() {
    return this.state;
  }

}
