/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.events;

import java.util.LinkedList;

/**
 * A blocking FIFO queue of events. Event consumers (EventSenderThread) are
 * blocked on an empty queue until new events are avialable.
 */
public class EventQueue {

  private LinkedList list = null;
  private int size = 0;

  EventQueue(int maxsize) {
    if (maxsize <= 0) {
      throw new RuntimeException("size parameter is invalid: " + size);
    }
    this.size = maxsize;
    this.list = new LinkedList();
  }

  EventQueue() {
    this.list = new LinkedList();
  }

  /**
   * Enqueues an events (adds it to the ebd of the list).
   *
   * @param event TBD: Description of the incoming method parameter
   * @return True, if the event was enqueued. False, if the maximum size of the
   *      queue has been reached.
   */
  public synchronized boolean enqueue(IEvent event) {
    // Queue has reached max. size
    if (this.size > 0 && this.list.size() == this.size) {
      return false;
    }

    this.list.addLast(event);
    // Wake up waiting worker thread(s)
    notifyAll();
    return true;
  }

  /**
   * Dequeues an event (removes the first from the list). If the queue is empty
   * the calling thread is blocked until new events are available.
   *
   * @return The event from the queue
   */
  public synchronized IEvent dequeue() {
    try {
      while (this.list.size() == 0) {
        // Wait until queue is not empty
        wait();
      }
    }
    catch (InterruptedException e) {
      return null;
    }
    return (IEvent)this.list.removeFirst();
  }

  /**
   * Dequeues the specified number of events. If the current size of the queue
   * is smaller than the specified number the size of the returned array will be
   * equal to the current size of the queue. If the queue is empty the calling
   * thread is blocked until new events are available.
   *
   * @param num The number of events to dequeue.
   * @return An array of events. The size can be smaller than the specified
   *      number.
   */
  public synchronized IEvent[] dequeue(int num) {
    if (this.size > 0 && (num < 0 || num > this.size)) {
      throw new RuntimeException("Parameter num is out of bounds");
    }
    try {
      while (this.list.size() == 0) {
        // Wait until queue is not empty
        wait();
      }
    }
    catch (InterruptedException e) {
      return new IEvent[0];
    }
    int batchSize = java.lang.Math.min(num, this.list.size());
    IEvent[] batch = new IEvent[batchSize];
    for (int i = 0; i < batchSize; i++) {
      batch[i] = (IEvent)this.list.removeFirst();
    }
    return batch;
  }
}
