/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.events;

import com.sap.tc.logging.Location;
import com.sapportals.wcm.WcmException;

import java.util.*;

/**
 * An event broker implementation <p>
 *      
 * Copyright (c) SAP AG 2001-2002
 *
 * @author Markus Breitenfelder
 * @see IEventBroker
 *
 * @version $Id: //javabas/com.sapportals.wcm/50_COR/src/java/util/api/com/sapportals/wcm/util/events/EventBroker.java#14
 *      $
 */
public class EventBroker implements IEventBroker {

  private static Location log = Location.getLocation(EventBroker.class);

	/**
	 * static Map: IEventReceiver -> Receiver
	 */
	protected static Map receiverMap = new HashMap();

  /**
   * All registered senders
   */
  protected final HashMap senderMap;

  /**
   * The list of receiver entries, in priority order
   */
  protected List receiverEntries;


  public EventBroker() {
    this.senderMap = new HashMap();
    this.receiverEntries = new ArrayList();
  }


  // ----------------------------------- Interface -----------------------------

  public void send(IEvent event, IEventSender sender)
    throws WcmException {
    if (event != null) {
      /*
      if (!this.senderMap.containsKey(((Object)sender).toString())) {
        throw new WcmException("Event sender is not registered: " + ((Object)sender).toString());
      }
      */
      List rl = getReceiverEntries();
      for (int i = 0, n = rl.size(); i < n; ++i) {
        ReceiverEntry entry = (ReceiverEntry)rl.get(i);
        if( event.isLike(entry.getTemplate()) ) {
          if (entry.getState().equals(SenderMode.SENDING)) {
            entry.doSend(event);
          }
          else if (entry.getState().equals(SenderMode.HOLD)) {
            entry.holdEvent(event);
          }
        }
      }
    }
  }

  public void register(IEventSender sender)
    throws WcmException {
    synchronized (this.senderMap) {
      if (!this.senderMap.containsKey(((Object)sender).toString())) {
        this.senderMap.put(((Object)sender).toString(), sender);
      }
      else {
        throw new WcmException("Sender is already registered: " + ((Object)sender).toString());
      }
    }
  }

  public void register(IEventReceiver receiver, IEvent template)
    throws WcmException {
    this.register(receiver, template, null, IEventBroker.PRIO_MIN, false);
  }

  public void register(IEventReceiver receiver, IEvent template, IEventMapper mapping)
    throws WcmException {
    this.register(receiver, template, mapping, IEventBroker.PRIO_MIN, false);
  }

  public void register(IEventReceiver receiver, IEvent template, int priority, boolean async)
    throws WcmException {
    this.register(receiver, template, null, priority, async);
  }

  public void register(IEventReceiver receiver, IEvent template, IEventMapper mapping, int priority, boolean async)
    throws WcmException {    	
    boolean newReceiver = false;
		Receiver rec = (Receiver)EventBroker.receiverMap.get(receiver);
		if (rec == null) {
			rec = new Receiver(receiver);
			newReceiver = true;
		}
		
    ReceiverEntry entry = new ReceiverEntry(rec, template, mapping, priority, async);
    if (this.addReceiverEntry(entry)) {
      log.debugT("register(106)", "Receiver registered: " + ((Object)receiver).toString() +
        ", event=" + template + ", mapping=" + mapping + ", prio=" + priority + ", async=" + async);

      if (async) {
        rec.setAsync();
      }      
			if (newReceiver) {
				EventBroker.receiverMap.put(receiver, rec);
			}    
    }
    else {
      log.debugT("register(110)", "Receiver is already registered for event(s): " + ((Object)receiver).toString() +
        ", event=" + template + ", mapping=" + mapping + ", prio=" + priority + ", async=" + async);
    }
  }

  public void unregister(IEventSender sender)
    throws WcmException {
    synchronized (this.senderMap) {
      this.senderMap.remove(((Object)sender).toString());
    }
  }

  public void unregister(IEventReceiver receiver, IEvent template)
    throws WcmException {
    this.removeReceiverEntry(receiver);
  }

  public void suspend(IEventReceiver receiver) {
    ReceiverEntry entry = searchEntry(receiver);
    if (entry != null) {
      entry.setState(SenderMode.SUSPEND);
      return;
    }
  }

  public void hold(IEventReceiver receiver, boolean collect) {
    ReceiverEntry entry = searchEntry(receiver);
    if (entry != null) {
      entry.setSendHoldEvents(collect);
      entry.setState(SenderMode.HOLD);
      return;
    }
  }

  public void resume(IEventReceiver receiver) {
    ReceiverEntry entry = searchEntry(receiver);
    if (entry != null) {
      entry.setState(SenderMode.SENDING);
      return;
    }
  }

  public SenderMode getMode(IEventReceiver receiver) {
    ReceiverEntry entry = searchEntry(receiver);
    if (entry != null) {
      return entry.getState();
    }
    else {
      return SenderMode.OFF;
    }
  }

  public IEventList getHoldEvents(IEventReceiver receiver) {
    ReceiverEntry entry = searchEntry(receiver);
    if (entry != null) {
      return entry.getHoldEvents();
    }
    else {
      return null;
    }
  }

  public IEventList clearHoldEvents(IEventReceiver receiver) {
    ReceiverEntry entry = searchEntry(receiver);
    if (entry != null) {
      return entry.clearHoldEvents();
    }
    else {
      return null;
    }
  }

  public IEventList getEvents() {
    IEventList list = new EventList();
    synchronized (this.senderMap) {
      Iterator it = this.senderMap.values().iterator();
      while (it.hasNext()) {
        IEventSender sender = (IEventSender)it.next();
        IEventList l = sender.getEvents();
        if (l != null) {
          list.addAll(l);
        }
      }
    }
    return list;
  }


  public void stop() {
		Iterator it = EventBroker.receiverMap.entrySet().iterator();
		while (it.hasNext()) {
			((Receiver)((Map.Entry)it.next()).getValue()).terminateThread();
		}
		EventBroker.receiverMap.clear();
		this.receiverEntries.clear();		
  }

  // ------------------------------- private -----------------------------------

  protected synchronized List getReceiverEntries() {
    return this.receiverEntries;
  }

  private synchronized boolean addReceiverEntry(ReceiverEntry entry) {
    if (!this.receiverEntries.contains(entry)) {
      List nrec = new ArrayList(this.receiverEntries.size() + 1);
      nrec.addAll(this.receiverEntries);
      nrec.add(entry);
      Collections.sort(nrec, new PrioComparator());
      this.receiverEntries = nrec;
      return true;
    }
    else {
      return false;
    }
  }

  private synchronized void removeReceiverEntry(IEventReceiver receiver) {
    List rl = new ArrayList(this.receiverEntries);
    for (int i = rl.size() - 1; i >= 0; i--) {
      ReceiverEntry entry = (ReceiverEntry)rl.get(i);
      if (entry.getReceiver() == receiver) {
        rl.remove(i);
      }
    }
    Collections.sort(rl, new PrioComparator());
    this.receiverEntries = rl;   
  }

  private ReceiverEntry searchEntry(IEventReceiver receiver) {
    List rl = getReceiverEntries();
    for (int i = 0, n = rl.size(); i < n; i++) {
      ReceiverEntry entry = (ReceiverEntry)rl.get(i);
      if (entry.getReceiver() == receiver) {
        return entry;
      }
    }
    return null;
  }

  /**
   * TBD: Description of the class.
   */
  private class PrioComparator implements Comparator {
    public int compare(Object o1, Object o2) {
      ReceiverEntry e1 = (ReceiverEntry)o1;
      ReceiverEntry e2 = (ReceiverEntry)o2;
      if ((e1.isAsync() && e2.isAsync()) || (!e1.isAsync() && !e2.isAsync())) {
        // if both rec. are sync. or async. compare the priority
        return e1.getPriority() - e2.getPriority();
      }
      else {
        if (e1.isAsync()) {
          // async. receivers must always receive events after sync. receivers
          return 1;
        }
        else {
          return -1;
        }
      }
    }
  }
}
