/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository.wrapping.property;

import com.sapportals.wcm.repository.*;
import com.sapportals.wcm.repository.enum.*;

import java.util.*;

/**
 * Transforms properties and property maps.
 */
public class PropertyTransformer {
  /**
   * Transforms a property.
   *
   * @param orgProperty
   * @return
   * @exception com.sap.netweaver.bc.rf.common.exception.ResourceException
   * @exception ResourceException
   */
  public static com.sap.netweaver.bc.rf.common.property.IProperty transform(com.sapportals.wcm.repository.IProperty orgProperty)
    throws com.sap.netweaver.bc.rf.common.exception.ResourceException, ResourceException {
    if (orgProperty == null) {
      return null;
    }
    com.sap.netweaver.bc.rf.common.property.IPropertyName name = PropertyNameTransformer.transform(orgProperty.getPropertyName());
    if (orgProperty.getType() == com.sapportals.wcm.repository.enum.PropertyType.STRING) {
      if (orgProperty.isMultivalued()) {
        List values = orgProperty.getValues();
        String[] strings = new String[values.size()];
        int i = 0;
        for (Iterator iter = values.iterator(); iter.hasNext(); ) {
          strings[i++] = (String)iter.next();
        }
        return new com.sap.netweaver.bc.rf.common.property.stringtype.StringMultiValueProperty(name, strings, orgProperty.getAttributes(), transform(orgProperty.getPropertyDef()));
      }
      else {
        return new com.sap.netweaver.bc.rf.common.property.stringtype.StringSingleValueProperty(name, orgProperty.getStringValue(), orgProperty.getAttributes(), transform(orgProperty.getPropertyDef()));
      }
    }
    else if (orgProperty.getType() == com.sapportals.wcm.repository.enum.PropertyType.INTEGER) {
      if (orgProperty.isMultivalued()) {
        List values = orgProperty.getValues();
        int[] ints = new int[values.size()];
        int i = 0;
        for (Iterator iter = values.iterator(); iter.hasNext(); ) {
          ints[i++] = ((Integer)iter.next()).intValue();
        }
        return new com.sap.netweaver.bc.rf.common.property.inttype.IntMultiValueProperty(name, ints, orgProperty.getAttributes(), transform(orgProperty.getPropertyDef()));
      }
      else {
        return new com.sap.netweaver.bc.rf.common.property.inttype.IntSingleValueProperty(name, orgProperty.getIntValue(), orgProperty.getAttributes(), transform(orgProperty.getPropertyDef()));
      }
    }
    else if (orgProperty.getType() == com.sapportals.wcm.repository.enum.PropertyType.LONG) {
      if (orgProperty.isMultivalued()) {
        List values = orgProperty.getValues();
        long[] longs = new long[values.size()];
        int i = 0;
        for (Iterator iter = values.iterator(); iter.hasNext(); ) {
          longs[i++] = ((Long)iter.next()).longValue();
        }
        return new com.sap.netweaver.bc.rf.common.property.longtype.LongMultiValueProperty(name, longs, orgProperty.getAttributes(), transform(orgProperty.getPropertyDef()));
      }
      else {
        return new com.sap.netweaver.bc.rf.common.property.longtype.LongSingleValueProperty(name, orgProperty.getLongIntValue(), orgProperty.getAttributes(), transform(orgProperty.getPropertyDef()));
      }
    }
    else if (orgProperty.getType() == com.sapportals.wcm.repository.enum.PropertyType.DATE) {
      if (orgProperty.isMultivalued()) {
        List values = orgProperty.getValues();
        Date[] dates = new Date[values.size()];
        int i = 0;
        for (Iterator iter = values.iterator(); iter.hasNext(); ) {
          dates[i++] = (Date)iter.next();
        }
        return new com.sap.netweaver.bc.rf.common.property.datetype.DateMultiValueProperty(com.sap.netweaver.bc.rf.common.property.PropertyType.DATETIME, name, dates, orgProperty.getAttributes(), transform(orgProperty.getPropertyDef()));
      }
      else {
        return new com.sap.netweaver.bc.rf.common.property.datetype.DateSingleValueProperty(com.sap.netweaver.bc.rf.common.property.PropertyType.DATETIME, name, orgProperty.getDateValue(), orgProperty.getAttributes(), transform(orgProperty.getPropertyDef()));
      }
    }
    else if (orgProperty.getType() == com.sapportals.wcm.repository.enum.PropertyType.BOOLEAN) {
      if (orgProperty.isMultivalued()) {
        List values = orgProperty.getValues();
        boolean[] booleans = new boolean[values.size()];
        int i = 0;
        for (Iterator iter = values.iterator(); iter.hasNext(); ) {
          booleans[i++] = ((Boolean)iter.next()).booleanValue();
        }
        return new com.sap.netweaver.bc.rf.common.property.booleantype.MutableBooleanMultiValueProperty(name, booleans, orgProperty.getAttributes(), transform(orgProperty.getPropertyDef()));
      }
      else {
        return new com.sap.netweaver.bc.rf.common.property.booleantype.MutableBooleanSingleValueProperty(name, orgProperty.getBooleanValue(), orgProperty.getAttributes(), transform(orgProperty.getPropertyDef()));
      }
    }
    else if (orgProperty.getType() == com.sapportals.wcm.repository.enum.PropertyType.XML) {
      if (orgProperty.isMultivalued()) {
        List values = orgProperty.getValues();
        String[] strings = new String[values.size()];
        int i = 0;
        for (Iterator iter = values.iterator(); iter.hasNext(); ) {
          strings[i++] = ((XMLMarkup)iter.next()).toString();
        }
        return new com.sap.netweaver.bc.rf.common.property.xmltype.XmlMultiValueProperty(name, strings, orgProperty.getAttributes(), transform(orgProperty.getPropertyDef()));
      }
      else {
        return new com.sap.netweaver.bc.rf.common.property.xmltype.XmlSingleValueProperty(name, orgProperty.getStringValue(), orgProperty.getAttributes(), transform(orgProperty.getPropertyDef()));
      }
    }
    else {
      throw new IllegalArgumentException("Type is not understood!");
    }
  }


  /**
   * Transforms a property.
   *
   * @param orgProperty
   * @return
   * @exception com.sap.netweaver.bc.rf.common.exception.ResourceException
   * @exception ResourceException
   */
  public static com.sapportals.wcm.repository.IProperty transform(com.sap.netweaver.bc.rf.common.property.IProperty orgProperty)
    throws com.sap.netweaver.bc.rf.common.exception.ResourceException, ResourceException {
    if (orgProperty == null) {
      return null;
    }
    com.sapportals.wcm.repository.IPropertyName name = PropertyNameTransformer.transform(orgProperty.getPropertyName());
    if (orgProperty instanceof com.sap.netweaver.bc.rf.common.property.stringtype.StringMultiValueProperty) {
      List values = new ArrayList();
      String[] strings = ((com.sap.netweaver.bc.rf.common.property.stringtype.StringMultiValueProperty)orgProperty).getValues();
      for (int i = 0; i < strings.length; i++) {
        values.add(strings[i]);
      }
      return new com.sapportals.wcm.repository.Property(name, transform(com.sapportals.wcm.repository.enum.PropertyType.STRING, true, orgProperty.getFlags()), values, orgProperty.getAttributes());
    }
    else if (orgProperty instanceof com.sap.netweaver.bc.rf.common.property.stringtype.StringSingleValueProperty) {
      return new com.sapportals.wcm.repository.Property(name, transform(com.sapportals.wcm.repository.enum.PropertyType.STRING, false, orgProperty.getFlags()), ((com.sap.netweaver.bc.rf.common.property.stringtype.StringSingleValueProperty)orgProperty).getValue(), orgProperty.getAttributes());
    }
    else if (orgProperty instanceof com.sap.netweaver.bc.rf.common.property.inttype.IntMultiValueProperty) {
      List values = new ArrayList();
      int[] ints = ((com.sap.netweaver.bc.rf.common.property.inttype.IntMultiValueProperty)orgProperty).getValues();
      for (int i = 0; i < ints.length; i++) {
        values.add(new Integer(ints[i]));
      }
      return new com.sapportals.wcm.repository.Property(name, transform(com.sapportals.wcm.repository.enum.PropertyType.INTEGER, true, orgProperty.getFlags()), values, orgProperty.getAttributes());
    }
    else if (orgProperty instanceof com.sap.netweaver.bc.rf.common.property.inttype.IntSingleValueProperty) {
      return new com.sapportals.wcm.repository.Property(name, transform(com.sapportals.wcm.repository.enum.PropertyType.INTEGER, false, orgProperty.getFlags()), new Integer(((com.sap.netweaver.bc.rf.common.property.inttype.IntSingleValueProperty)orgProperty).getValue()), orgProperty.getAttributes());
    }
    else if (orgProperty instanceof com.sap.netweaver.bc.rf.common.property.longtype.LongMultiValueProperty) {
      List values = new ArrayList();
      long[] longs = ((com.sap.netweaver.bc.rf.common.property.longtype.LongMultiValueProperty)orgProperty).getValues();
      for (int i = 0; i < longs.length; i++) {
        values.add(new Long(longs[i]));
      }
      return new com.sapportals.wcm.repository.Property(name, transform(com.sapportals.wcm.repository.enum.PropertyType.LONG, true, orgProperty.getFlags()), values, orgProperty.getAttributes());
    }
    else if (orgProperty instanceof com.sap.netweaver.bc.rf.common.property.longtype.LongSingleValueProperty) {
      return new com.sapportals.wcm.repository.Property(name, transform(com.sapportals.wcm.repository.enum.PropertyType.LONG, false, orgProperty.getFlags()), new Long(((com.sap.netweaver.bc.rf.common.property.longtype.LongSingleValueProperty)orgProperty).getValue()), orgProperty.getAttributes());
    }
    else if (orgProperty instanceof com.sap.netweaver.bc.rf.common.property.datetype.DateMultiValueProperty) {
      List values = new ArrayList();
      Date[] dates = ((com.sap.netweaver.bc.rf.common.property.datetype.DateMultiValueProperty)orgProperty).getValues();
      for (int i = 0; i < dates.length; i++) {
        values.add(dates[i]);
      }
      return new com.sapportals.wcm.repository.Property(name, transform(com.sapportals.wcm.repository.enum.PropertyType.DATE, true, orgProperty.getFlags()), values, orgProperty.getAttributes());
    }
    else if (orgProperty instanceof com.sap.netweaver.bc.rf.common.property.datetype.DateSingleValueProperty) {
      return new com.sapportals.wcm.repository.Property(name, transform(com.sapportals.wcm.repository.enum.PropertyType.DATE, false, orgProperty.getFlags()), ((com.sap.netweaver.bc.rf.common.property.datetype.DateSingleValueProperty)orgProperty).getValue(), orgProperty.getAttributes());
    }
    else if (orgProperty instanceof com.sap.netweaver.bc.rf.common.property.booleantype.BooleanMultiValueProperty) {
      List values = new ArrayList();
      boolean[] booleans = ((com.sap.netweaver.bc.rf.common.property.booleantype.BooleanMultiValueProperty)orgProperty).getValues();
      for (int i = 0; i < booleans.length; i++) {
        values.add(new Boolean(booleans[i]));
      }
      return new com.sapportals.wcm.repository.Property(name, transform(com.sapportals.wcm.repository.enum.PropertyType.BOOLEAN, true, orgProperty.getFlags()), values, orgProperty.getAttributes());
    }
    else if (orgProperty instanceof com.sap.netweaver.bc.rf.common.property.booleantype.BooleanSingleValueProperty) {
      return new com.sapportals.wcm.repository.Property(name, transform(com.sapportals.wcm.repository.enum.PropertyType.BOOLEAN, false, orgProperty.getFlags()), new Boolean(((com.sap.netweaver.bc.rf.common.property.booleantype.BooleanSingleValueProperty)orgProperty).getValue()), orgProperty.getAttributes());
    }
    else if (orgProperty instanceof com.sap.netweaver.bc.rf.common.property.xmltype.XmlMultiValueProperty) {
      List values = new ArrayList();
      String[] strings = ((com.sap.netweaver.bc.rf.common.property.stringtype.StringMultiValueProperty)orgProperty).getValues();
      for (int i = 0; i < strings.length; i++) {
        values.add(new XMLMarkup(strings[i]));
      }
      return new com.sapportals.wcm.repository.Property(name, transform(com.sapportals.wcm.repository.enum.PropertyType.XML, true, orgProperty.getFlags()), values, orgProperty.getAttributes());
    }
    else if (orgProperty instanceof com.sap.netweaver.bc.rf.common.property.xmltype.XmlSingleValueProperty) {
      return new com.sapportals.wcm.repository.Property(name, transform(com.sapportals.wcm.repository.enum.PropertyType.XML, false, orgProperty.getFlags()), new XMLMarkup(((com.sap.netweaver.bc.rf.common.property.xmltype.XmlSingleValueProperty)orgProperty).getValue()), orgProperty.getAttributes());
    }
    else {
      throw new IllegalArgumentException("Type is not understood!");
    }
  }

  /**
   * Transforms a property map.
   *
   * @param map TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception com.sap.netweaver.bc.rf.common.exception.ResourceException
   *      Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   */
  public static IPropertyMap transform(Map map)
    throws com.sap.netweaver.bc.rf.common.exception.ResourceException, ResourceException {
    if (map == null) {
      return null;
    }
    IMutablePropertyMap result = new MutablePropertyMap(map.size());
    Iterator it = map.entrySet().iterator();
    while (it.hasNext()) {
      Object p = ((Map.Entry)it.next()).getValue();
      result.put(PropertyTransformer.transform((com.sap.netweaver.bc.rf.common.property.IProperty)p));
    }
    return result;
  }

  /**
   * Transforms a property map.
   *
   * @param map TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception com.sap.netweaver.bc.rf.common.exception.ResourceException
   *      Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   */
  public static Map transform(IPropertyMap map)
    throws com.sap.netweaver.bc.rf.common.exception.ResourceException, ResourceException {
    if (map == null) {
      return null;
    }
    Map result = new HashMap(map.size());
    IPropertyIterator it = map.iterator();
    while (it.hasNext()) {
      com.sap.netweaver.bc.rf.common.property.IProperty p = PropertyTransformer.transform(it.next());
      result.put(p.getPropertyName(), p);
    }
    return result;
  }

  /**
   * Transforms to a property list.
   *
   * @param map TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception com.sap.netweaver.bc.rf.common.exception.ResourceException
   *      Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   */
  public static List transformToList(IPropertyMap map)
    throws com.sap.netweaver.bc.rf.common.exception.ResourceException, ResourceException {
    if (map == null) {
      return null;
    }
    List resultList = new java.util.ArrayList(map.size());
    IPropertyIterator it = map.iterator();
    while (it.hasNext()) {
      IProperty property = it.next();
      resultList.add(PropertyTransformer.transform(property));
    }
    return resultList;
  }

  /**
   * Transforms property definition into the property flags.
   *
   * @param properytDef TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception com.sap.netweaver.bc.rf.common.exception.ResourceException
   *      Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   */
  private static int transform(IPropertyDef properytDef)
    throws com.sap.netweaver.bc.rf.common.exception.ResourceException, ResourceException {
    return com.sap.netweaver.bc.rf.common.property.PropertyFlags.constructFlags(properytDef.isRequired(), false, properytDef.isReadonly(), properytDef.isHidden());
  }

  /**
   * Transforms property flags into the property definition.
   *
   * @param propertyType TBD: Description of the incoming method parameter
   * @param isMultivalued TBD: Description of the incoming method parameter
   * @param propertyFlags TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception com.sap.netweaver.bc.rf.common.exception.ResourceException
   *      Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   */
  private static IPropertyDef transform(PropertyType propertyType, boolean isMultivalued, int propertyFlags)
    throws com.sap.netweaver.bc.rf.common.exception.ResourceException, ResourceException {
    return new PropertyDef(propertyType,
      isMultivalued,
      com.sap.netweaver.bc.rf.common.property.PropertyFlags.isRequired(propertyFlags),
      com.sap.netweaver.bc.rf.common.property.PropertyFlags.isReadOnly(propertyFlags),
      false,
      com.sap.netweaver.bc.rf.common.property.PropertyFlags.isHidden(propertyFlags));
  }
}
