/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository.wrapping.content;
import com.sap.netweaver.bc.rf.common.content.*;
import com.sap.netweaver.bc.rf.common.exception.*;

import com.sapportals.wcm.repository.IEntityMetadata;
import com.sapportals.wcm.repository.IMutableEntityMetadata;
import com.sapportals.wcm.repository.wrapping.RidTransformer;
import java.io.InputStream;
import java.util.Date;

/**
 * Wrappes a com.sapportals.wcm.repository.Content object
 *
 * @created 14. April 2003
 */
public class ContentWrapperIn implements IMutableContent, IContentMetadata {
  private final com.sapportals.wcm.util.content.IContent content;
  private boolean contentMetadataUpdatedState = false;

  /**
   * Construct object of class ContentWrapperIn.
   *
   * @param content TBD: Description of the incoming method parameter
   */
  ContentWrapperIn(com.sapportals.wcm.util.content.IContent content) {
    if (content == null) {
      throw new NullPointerException("parameter content is null");
    }
    this.content = content;
  }


  /**
   * Set the metadata attribute of the ContentWrapperIn object.
   *
   * @param contentMetadata contentMetadata to be set
   * @exception ResourceException Exception raised in failure situation
   */
  public void setContentMetadata(IContentMetadata contentMetadata)
    throws ResourceException {
    if (contentMetadata == null) {
      throw new NullPointerException("parameter metadata is null");
    }
    if (this.content instanceof IMutableEntityMetadata) {
      try {
        ((IMutableEntityMetadata)this.content).setEntityTag(contentMetadata.getETag());
        ((IMutableEntityMetadata)this.content).setLastModified(contentMetadata.getLastModified());
        contentMetadataUpdatedState = true;
      }
      catch (com.sapportals.wcm.repository.ResourceException ex) {
        throw new ResourceException(RidTransformer.transform(ex.getRID()), ex.getMessage(), ex, true);
      }
    }
    else {
      // This should not happen if the client uses com.sapportals.wcm.repository.Content !
      // TODO: logging
    }
  }

  /**
   * Check whether or not the metadata was updated since the creation of this
   * mutable content or the last reset of the metadata updated state.
   *
   * @return flag whether or not the metadata was updated
   */
  public boolean getContentMetadataUpdatedState() {
    return this.contentMetadataUpdatedState;
  }

  /**
   * Reset the metadata updated state to false.
   */
  public void resetContentMetadataUpdatedState() {
    this.contentMetadataUpdatedState = false;
  }

  // -- IMutableContent --

  /**
   * Get the inputStream attribute of the ContentWrapperIn object.
   *
   * @return The inputStream value
   * @exception ResourceException Exception raised in failure situation
   */
  public InputStream getInputStream()
    throws ResourceException {
    try {
      return this.content.getInputStream();
    }
    catch (com.sapportals.wcm.util.content.ContentException ex) {
      throw new ResourceException(null, ex.getMessage(), ex, true);
    }
  }

  /**
   * Get the metadata attribute of the ContentWrapperIn object.
   *
   * @return The metadata value
   * @exception ResourceException Exception raised in failure situation
   */
  public IContentMetadata getContentMetadata()
    throws ResourceException {
    return this;
  }

  // -- IMetadata --

  /**
   * Get the contentLength attribute of the ContentWrapperIn object.
   *
   * @return The contentLength value
   */
  public long getContentLength() {
    try {
      return this.content.getContentLength();
    }
    catch (com.sapportals.wcm.util.content.ContentException ex) {
      throw new java.lang.RuntimeException("Content Exception: " + ex.getMessage());
    }
  }

  /**
   * Get the contentType attribute of the ContentWrapperIn object.
   *
   * @return The contentType value
   */
  public String getContentType() {
    try {
      return this.content.getContentType();
    }
    catch (com.sapportals.wcm.util.content.ContentException ex) {
      throw new java.lang.RuntimeException("Content Exception: " + ex.getMessage());
    }
  }

  /**
   * Get the encoding attribute of the ContentWrapperIn object.
   *
   * @return The encoding value
   */
  public String getContentEncoding() {
    try {
      return this.content.getEncoding();
    }
    catch (com.sapportals.wcm.util.content.ContentException ex) {
      throw new java.lang.RuntimeException("Content Exception: " + ex.getMessage());
    }
  }


  /**
   * Get the language attribute of the ContentWrapperIn object.
   *
   * @return The language value
   */
  public String getContentLanguage() {
    if (this.content instanceof IEntityMetadata) {
      try {
        return ((IEntityMetadata)this.content).getLanguage();
      }
      catch (com.sapportals.wcm.repository.ResourceException ex) {
        throw new java.lang.RuntimeException("Resource Exception: " + ex.getMessage());
      }
    }
    else {
      return null;
    }
  }


  /**
   * Get the entityTag attribute of the ContentWrapperIn object.
   *
   * @return The entityTag value
   */
  public String getETag() {
    try {
      return this.content.getEncoding();
    }
    catch (com.sapportals.wcm.util.content.ContentException ex) {
      throw new java.lang.RuntimeException("Content Exception: " + ex.getMessage());
    }
  }


  /**
   * Get the expiryDate attribute of the ContentWrapperIn object.
   *
   * @return The expiryDate value
   */
  public Date getExpires() {
    if (this.content instanceof IEntityMetadata) {
      try {
        return ((IEntityMetadata)this.content).getExpiryDate();
      }
      catch (com.sapportals.wcm.util.content.ContentException ex) {
        throw new java.lang.RuntimeException("Content Exception: " + ex.getMessage());
      }
    }
    else {
      return null;
    }
  }


  /**
   * Get the lastModified attribute of the ContentWrapperIn object.
   *
   * @return The lastModified value
   */
  public Date getLastModified() {
    if (this.content instanceof IEntityMetadata) {
      try {
        return ((IEntityMetadata)this.content).getLastModified();
      }
      catch (com.sapportals.wcm.repository.ResourceException ex) {
        throw new java.lang.RuntimeException("Resource Exception: " + ex.getMessage());
      }
    }
    else {
      return null;
    }
  }

  // -------------------


  /**
   * Get the wrappedContent attribute of the ContentWrapperIn object.
   *
   * @return The wrappedContent value
   */
  public com.sapportals.wcm.util.content.IContent getWrappedContent() {
    return this.content;
  }
}
