/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.repository.service;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Properties;
import java.util.StringTokenizer;

import com.sap.tc.logging.Location;
import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.crt.CrtSystem;
import com.sapportals.wcm.crt.component.ComponentEvent;
import com.sapportals.wcm.crt.component.ComponentException;
import com.sapportals.wcm.crt.component.ComponentState;
import com.sapportals.wcm.crt.component.ComponentStateHandler;
import com.sapportals.wcm.crt.component.IAutoStartable;
import com.sapportals.wcm.crt.component.IComponent;
import com.sapportals.wcm.crt.component.IComponentInfo;
import com.sapportals.wcm.crt.component.IComponentListener;
import com.sapportals.wcm.crt.component.IConfigurable;
import com.sapportals.wcm.crt.component.ILifecycleInfo;
import com.sapportals.wcm.crt.component.IThreadSafe;
import com.sapportals.wcm.crt.component.StartupException;
import com.sapportals.wcm.crt.configuration.ConfigurationException;
import com.sapportals.wcm.crt.configuration.IConfiguration;
import com.sapportals.wcm.repository.IResourceContext;
import com.sapportals.wcm.repository.ResourceException;
import com.sapportals.wcm.repository.manager.IRepositoryManager;
import com.sapportals.wcm.repository.runtime.CmAdapter;
import com.sapportals.wcm.repository.runtime.CmRegistry;
import com.sapportals.wcm.repository.runtime.ICmRuntimeConst;
import com.sapportals.wcm.util.logging.LoggingFormatter;

/**
 * This is the abstract base class for all repository service implementations.
 * <p>
 *
 * Copyright (c) SAP AG 2002
 *
 * @author Markus Breitenfelder
 * @version $Id:$
 */
public abstract class AbstractRepositoryService
  implements
    IRepositoryService,
    IThreadSafe,
    IComponent,
    IConfigurable,
    IAutoStartable,
    IComponentInfo,
    ILifecycleInfo,
    IComponentListener {

  private static Location log =
    Location.getLocation(com.sapportals.wcm.repository.service.AbstractRepositoryService.class);

  // Note: Service API cannot import com.sapportals.wcm.repository.runtime.ICmRuntimeConst
  // Cyclic dependency in make !
  /**
   * TBD: Description of the class.
   */
  public interface CONFIG_ATTRIBUTE_NAME {
    String ID = "name";
    String DESCRIPTION = "description";
    String MANAGER_LIST = "service_manager_list";
  }

  //constants to create and read out calls of the ServiceServlet
  public final static String PARAMETER_SERVICE = "_Service";
  public final static String PARAMETER_ACTION = "_Action";
  public final static String PARAMETER_RESOURCE = "_Resource";
  public final static String PARAMETER_REPOSITORY = "_Repository";
  public final static String PARAMETER_IS_GLOBAL = "_isGlobal";
  public final static String PARAMETER_LOCATION = "_Location";
  public final static String PARAMETER_POPUP = "_Popup";
  public final static String PARAMETER_PREFIX = "_";
  public final static String PARAMETER_TRUE = "true";
  public final static String PARAMETER_FALSE = "false";

  protected String id = null;
  protected String description = null;

  protected Properties properties = null;
  protected IConfiguration config = null;

  protected final ComponentStateHandler stateHandler = new ComponentStateHandler(this);

  /**
   * The default constructor.
   */
  public AbstractRepositoryService() {
  }

  /**
   * @param properties TBD: Description of the incoming method parameter
   * @exception ResourceException Exception raised in failure situation
   * @deprecated As of EP 5.0 SP5, replaced by the default (public, no
   *      arguments) constructor.
   */
  public AbstractRepositoryService(Properties properties) throws ResourceException {
    if (properties != null) {
      this.properties = (Properties) (properties.clone());
    }
    this.id = (String)this.properties.get(CONFIG_ATTRIBUTE_NAME.ID);
  }

  /**
   * A service implementation must overwrite this method to specify it's
   * instance type
   *
   * @return The service's instance type
   * @see com.sapportals.wcm.IWcmConst
   * @deprectated This method is not required
   */
  public static int getInstanceType() {
    throw new java.lang.RuntimeException("Must overwrite static method getInstanceType()");
  }

  // ---------------------------------------------------------------------------
  // IRepositoryService
  // ---------------------------------------------------------------------------

  public final String getID() {
    return this.id;
  }

  public String getDescription() {
    return this.description;
  }

  public String acceptServletCall(Properties properties, IResourceContext context) throws ResourceException {
    return "No Action performed. The called service doesn't support your call.";
  }

  // ---------------------------------------------------------------------------
  // interface IConfigurable
  // ---------------------------------------------------------------------------

  public final void configure(IConfiguration config) throws ConfigurationException {
    this.stateHandler.preConfigure();

    this.config = config;
    this.id = config.getAttribute(ICmRuntimeConst.CONFIG_ATTRIBUTE_NAME.ID);
    this.description = config.getAttribute(ICmRuntimeConst.CONFIG_ATTRIBUTE_NAME.DESCRIPTION, null);

    this.stateHandler.postConfigure();
  }

  // ---------------------------------------------------------------------------
  // interface IStartable
  // ---------------------------------------------------------------------------

  public final void start() throws StartupException {
    this.stateHandler.preStart();

    try {
      // Extract list of initial repository managers from config
      CmAdapter system = CmAdapter.getInstance();
      Collection rms = new ArrayList();
      String list = config.getAttribute(ICmRuntimeConst.CONFIG_ATTRIBUTE_NAME.MANAGER_LIST);
      List managerIDs = this.parseValueList(list, ICmRuntimeConst.LIST_SEPARATOR);
      for (int i = 0; i < managerIDs.size(); i++) {
        IRepositoryManager rm = system.getRepositoryManager((String)managerIDs.get(i));
        if (rm != null) {
          rms.add(rm);
        }
      }

      this.startUpImpl(rms);
    }
    catch (ConfigurationException ex) {
      StartupException x = this.stateHandler.postStart(ex);
      this.logStartUpTime();
      throw x;
    }
    catch (StartupException ex) {
      StartupException x = this.stateHandler.postStart(ex);
      this.logStartUpTime();
      throw x;
    }
    catch (ServiceNotAvailableException ex) {
      // The service is not available for some repositories: Remove registry entries
      List ids = ex.getRepositoryManagerIDs();
      if (ids != null && ids.size() > 0) {
        AbstractRepositoryService.log.errorT(
          "start(199)",
          "Service "
            + this.getID()
            + " will be not available at these repositories: "
            + ids.toString()
            + ", see previous errors for details");
        Iterator it = ids.iterator();
        while (it.hasNext()) {
          try {
            CmRegistry.registry.updateRepository((String)it.next(), this.getID(), false);
          }
          catch (ConfigurationException cx) {
            //
            // TODO: it is not a good idea to abort the while loop here
            //
            StartupException x = this.stateHandler.postStart(cx);
            this.logStartUpTime();
            throw x;
          }
        }
      }
      else {
        AbstractRepositoryService.log.errorT(
          "start(217)",
          "List of repository manager IDs in ServiceNotAvailableException is null or empty"
            + ": "
            + LoggingFormatter.extractCallstack(ex));
      }

      // Overall startup successfull: Do NOT throw ServiceNotAvailableException and continue with postStart() !
    }
    catch (Exception ex) {
      StartupException x =
        this.stateHandler.postStart(
          new StartupException(this.getID() + ": runtime exception during start up: " + ex.getMessage(), ex));
      this.logStartUpTime();
      throw x;
    }

    this.stateHandler.postStart();
    this.logStartUpTime();
  }

  public final void stop() {
    this.stateHandler.preStop();
    this.shutDownImpl();
    this.stateHandler.postStop();
  }

  // ---------------------------------------------------------------------------
  // interface IComponentInfo
  // ---------------------------------------------------------------------------

  public String getName() {
    return this.id;
  }

  public String getDescription(Locale locale) {
    // Default implementation returns description in configuration data
    return this.getDescription();
  }

  public Properties getProperties() {
    Properties props = convertConfigToProperties(this.config, "config-");
    props.setProperty("type", "RepositoryService");
    return props;
  }

  // ---------------------------------------------------------------------------
  // interface ILifecycleInfo
  // ---------------------------------------------------------------------------

  public final ComponentState getState() {
    return this.stateHandler.getState();
  }

  public ConfigurationException getLastConfigurationException() {
    return this.stateHandler.getLastConfigurationException();
  }

  public StartupException getStartupException() {
    return this.stateHandler.getStartupException();
  }

  public final Date getCreationDate() {
    return this.stateHandler.getCreationDate();
  }

  public Date getLastReconfigurationDate() {
    return this.stateHandler.getLastReconfigurationDate();
  }

  public Date getNextAutoRestartDate() {
    return this.stateHandler.getNextAutoRestartDate();
  }

  // ---------------------------------------------------------------------------
  // interface IComponentListener
  // ---------------------------------------------------------------------------

  public final void notify(ComponentEvent event) {
    // Removing repositories or changing service list is not supported yet
    if (!event.getType().equals(ComponentEvent.Type.COMPONENT_ADDED)) {
      return;
    }

    if (IRepositoryManager.class.isAssignableFrom(event.getComponentClass())) {
      IRepositoryManager mgr = null;
      try {
        mgr = (IRepositoryManager)CrtSystem.getInstance().lookupComponentByUri(event.getComponentUri());
      }
      catch (ComponentException ex) {
        AbstractRepositoryService.log.errorT("notify(299)", LoggingFormatter.extractCallstack(ex));
        return;
      }
      if (mgr == null) {
        AbstractRepositoryService.log.errorT(
          "notify(303)",
          "Repository not found: " + event.getComponentUri().toString());
        return;
      }

      if ((mgr instanceof ILifecycleInfo) && !ComponentState.RUNNING.equals(((ILifecycleInfo)mgr).getState())) {
        return;
      }

      if (null != CmRegistry.registry.getRepositoryServiceID(mgr.getID(), this.getServiceType())) {
        try {
          this.addRepositoryAssignment(mgr);
        }
        catch (ServiceNotAvailableException ex) {
          // The service is not available for this repository: Remove registry entry
          try {
            AbstractRepositoryService.log.errorT(
              "notify(320)",
              "Service "
                + this.getID()
                + " will be not available at these repositories: "
                + id
                + ", see previous errors for details");
            CmRegistry.registry.updateRepository(mgr.getID(), this.getID(), false);
          }
          catch (ConfigurationException cx) {
            AbstractRepositoryService.log.errorT("notify(324)", LoggingFormatter.extractCallstack(cx));
          }
        }
      }

    }
  }

  // ---------------------------------------------------------------------------

  /**
   * The implementation must overwrite this method if it has initial startup
   * handling to do (acquire resources, open connections to remote systems,
   * etc). Accoring to the contract of the {@link
   * com.sapportals.wcm.crt.component.IStartable} interface of the CRT this
   * method will be called only once. It will be called on systen start up or
   * when a new instance of this component is added to the configuration.
   *
   * @param repositoryManagers A collection of repository manager instances that
   *      this services is attached to via configuration. Usually a service
   *      implementation will register for certain resource events at the
   *      repository's event broker.
   * @exception ConfigurationException Exception raised in failure situation
   * @exception StartupException Exception raised in failure situation
   * @exception ServiceNotAvailableException Exception raised in failure
   *      situation
   */
  protected void startUpImpl(Collection repositoryManagers)
    throws ConfigurationException, StartupException, ServiceNotAvailableException {
    try {
      this.startUp(repositoryManagers);
    }
    catch (WcmException x) {
      throw new StartupException(x);
    }
  }

  /**
   * The implementation must overwrite this method if it has initial startup
   * handling to do (acquire resources, open connections to remote systems,
   * etc).
   *
   * @param repositoryManagers TBD: Description of the incoming method parameter
   * @exception WcmException Exception raised in failure situation
   * @deprecated As of EP 5.0 SP5, replaced by {@link #shutDownImpl()}.
   */
  protected void startUp(Collection repositoryManagers) throws WcmException {
    AbstractRepositoryService.log.infoT(
      "startUp(376)",
      "The repository service implementation does not overwrite the startUpImpl() method: " + this.id);
  }

  /**
   * The implementation must overwrite this method if it has to release
   * resources acquired during startup. Accoring to the contract of the {@link
   * com.sapportals.wcm.crt.component.IStartable} interface of the CRT this
   * method will be called only once. It will be called when the component is
   * removed from the persistent configuration or when the system shuts down.
   */
  protected void shutDownImpl() {
    try {
      this.shutDown();
    }
    catch (WcmException x) {
      AbstractRepositoryService.log.errorT(
        "shutDownImpl(391)",
        "shutDown failed: " + x.getMessage() + ": " + LoggingFormatter.extractCallstack(x));
    }
  }

  /**
   * The implementation must overwrite this method if it has to release
   * resources acquired during shutdown.
   *
   * @exception WcmException Exception raised in failure situation
   * @deprecated As of EP 5.0 SP5, replaced by {@link #startUpImpl(Collection)}.
   */
  protected void shutDown() throws WcmException {
    AbstractRepositoryService.log.infoT(
      "shutDown(405)",
      "The repository service implementation does not overwrite the shutDownImpl() method: " + this.id);
  }

  /**
   * Called by the framework if a new repository with services is added or the
   * service was added to the list of an existing repository.
   *
   * @param mgr The repository manager instance that this service was assigned
   *      to in configuration.
   * @exception ServiceNotAvailableException If the service cannot be assigned
   *      to the repository.
   */
  protected void addRepositoryAssignment(IRepositoryManager mgr) throws ServiceNotAvailableException {
    AbstractRepositoryService.log.errorT(
      "addRepositoryAssignment(419)",
      "The repository service implementation does not overwrite the addRepositoryAssignment() method: " + this.id);
  }

  /**
   * Called by the framework if a repository is removed from the configuration
   * or the service is removed from the list.
   *
   * @param mgr TBD: Description of the incoming method parameter
   * @exception WcmException Exception raised in failure situation
   */
  protected void removeRepositoryAssignment(IRepositoryManager mgr) throws WcmException {
  }

  /**
   * Helper method for conversion of configuration data into properties.
   *
   * @param config to be converted to properties
   * @return attributes of config as properties
   */
  public final static Properties convertConfigToProperties(IConfiguration config) {
    return convertConfigToProperties(config, null);
  }
  
  /**
   * Helper method for conversion of configuration data into properties.
   *
   * @param config to be converted to properties
   * @return attributes of config as properties
   */
  protected final static Properties convertConfigToProperties(IConfiguration config, String prefix) {
    Properties props = new Properties();
    String[] attributeNames = config.getAttributeNames();
    for (int i = 0; i < attributeNames.length; i++) {
      String name = attributeNames[i];
      String value = config.getAttribute(name, null);
      if (value != null) {
        if (prefix != null) {
          name = prefix + name;
        }
        props.setProperty(name, value);
      }
    }
    return props;
  }

  /**
   * Helper method: Tokenize a string with a separator char into a List
   *
   * @param values TBD: Description of the incoming method parameter
   * @param separator TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  private List parseValueList(String values, String separator) {
    List list = new LinkedList();
    if (values == null) {
      return list;
    }
    StringTokenizer st = new StringTokenizer(values, separator);
    while (st.hasMoreTokens()) {
      list.add(st.nextToken().trim());
    }
    return list;
  }

  private void logStartUpTime() {
    long time = this.stateHandler.getConfigurationTime() + this.stateHandler.getStartTime();
    double sec = (double)time / 1000.0;
    AbstractRepositoryService.log.debugT(
      "logStartUpTime(473)",
      "Startup of repository service " + this.getID() + " took " + sec + " seconds");
  }
}
