/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id: //kmgmt/bc.rf/60NW_SP_COR/src/_runtime/java/api/com/sapportals/wcm/repository/security/SecurityAudit.java#1 $
 */
package com.sapportals.wcm.repository.security;

import com.sap.tc.logging.*;
import com.sapportals.portal.security.usermanagement.*;
import com.sapportals.wcm.repository.manager.*;
import com.sapportals.wcm.util.acl.*;
import com.sapportals.wcm.util.uri.*;
import com.sapportals.wcm.util.systemconfig.*;

import java.util.*;

/**
 * Logging for all security relevant events.
 *
 * @author Markus Liepold
 */
public class SecurityAudit {

  // -----------
  // Inner Class --------------------------------------------------------------
  // -----------

  private class Log {
    // dummy class just for logging
  }
  
  
  // ---------
  // Constants ----------------------------------------------------------------
  // ---------

  private final static String SEPARATOR = "\t| ";

  private final static String ACL_CREATE = "ACL.CREATE";
  private final static String ACL_MODIFY = "ACL.MODIFY";
  private final static String ACL_DELETE = "ACL.DELETE";

  private final static String ACCESS_GRANT = "ACCESS.OK";
  private final static String ACCESS_DENY = "ACCESS.ERROR";

  private final static String PERMISSION_CREATE = "ACLPERM.CREATE";
  private final static String PERMISSION_DELETE = "ACLPERM.DELETE";

  private final static String SUPPORTEDPERMISSION_ADD = "ACLPERM.ADD";
  private final static String SUPPORTEDPERMISSION_REMOVE = "ACLPERM.REMOVE";

  private static com.sap.tc.logging.Category category = com.sap.tc.logging.Category.getCategory("/System/Security/Audit");
  private static com.sap.tc.logging.Location location = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.repository.security.SecurityAudit.Log.class);
  private static com.sap.tc.logging.Location trace = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.repository.security.SecurityAudit.class);


  // -----------------------
  // Static Member Variables -------------------------------------------------
  // -----------------------

  private static SecurityAudit instance = null;

  // -----------------------
  // Static Member Variables -------------------------------------------------
  // -----------------------

  private HashSet systemIds = null;

  // ---------------
  // Private Methods ----------------------------------------------------------
  // ---------------

  // ------------------------------------------------------------------------
  private String getUserID(com.sapportals.portal.security.usermanagement.IUser user) {
    if( user == null ) {
      return "<null>";
    }
    return user.getName();
  }


  // ------------------------------------------------------------------------
  private String getPrincipalIDs(IUMPrincipalList principals) {
    if( principals == null ) {
      return "<null>";
    }
    StringBuffer ids = new StringBuffer();
    IUMPrincipalListIterator iterator = principals.iterator();
    while( iterator.hasNext() ) {
      ids.append(iterator.next().getDisplayName());
      if( iterator.hasNext() ) {
        ids.append(',');
      }
    }
    return ids.toString();
  }


  // ------------------------------------------------------------------------
  private String getAclPermissionID(IAclPermission permission) {
    if( permission == null ) {
      return "<null>";
    }
    try {
      return permission.getName();
    }
    catch( AclPersistenceException e ) {
      return "<unknown>";
    }
  }


  // ------------------------------------------------------------------------
  private String getPermissionID(IPermission permission) {
    if( permission == null ) {
      return "<null>";
    }
    return permission.getName();
  }


  // ------------------------------------------------------------------------
  private String getPermissionIDs(IPermissionList permissions) {
    if( permissions == null ) {
      return "<null>";
    }
    StringBuffer ids = new StringBuffer();
    IPermissionListIterator iterator = permissions.iterator();
    while( iterator.hasNext() ) {
      ids.append(iterator.next().getName());
      if( iterator.hasNext() ) {
        ids.append(',');
      }
    }
    return ids.toString();
  }


  // ------------------------------------------------------------------------
  private boolean isUserToBeLogged(com.sapportals.portal.security.usermanagement.IUser user) {
    if( user == null ) {
      return false;
    }
    if( this.systemIds.contains(user.getId()) ) {
      // service users won't be logged
      return false;
    }
    return true;
  }
  

  // ------------
  // Constructors -------------------------------------------------------------
  // ------------

  // ------------------------------------------------------------------------
  /**
   * Construct a SecurityAudit.
   */
  public SecurityAudit() { }


  // --------------
  // Public Methods -----------------------------------------------------------
  // --------------

  // ------------------------------------------------------------------------
  /**
   * Get an SecurityAudit instance.
   *
   * @return instance
   */
  public static SecurityAudit getInstance() {
    synchronized (SecurityAudit.class) {
      if( SecurityAudit.instance == null ) {
        SecurityAudit.instance = new SecurityAudit();
        SecurityAudit.instance.systemIds = new HashSet();
        try {
          ISystemPrincipalList systemUsers = SystemConfig.getInstance().getSystemUsers();
          ISystemPrincipalListIterator systemUserIterator = systemUsers.iterator();
          while( systemUserIterator.hasNext() ) {
            SecurityAudit.instance.systemIds.add(systemUserIterator.next().getPrincipal().getId());
          }
        }
        catch( Exception e ) {
          trace.fatalT("SecurityAudit.getInstance()", "unable to get system users: they will be logged! " + e.getMessage());
        }
      }
    }
    return SecurityAudit.instance;
  }


  // ------------------------------------------------------------------------
  /**
   * Access denied by ACL on a resource for a given permission, because the user
   * is not authenticated.
   *
   * @param user TBD: Description of the incoming method parameter
   * @param permission TBD: Description of the incoming method parameter
   * @param rid TBD: Description of the incoming method parameter
   */
  public void aclDeny_UserNotAuthenticated(com.sapportals.portal.security.usermanagement.IUser user,
                                           IAclPermission permission,
                                           RID rid) {
    if(   SecurityAudit.category.beFatal()
       && isUserToBeLogged(user)
      ) {
      SecurityAudit.category.fatalT(
                                   SecurityAudit.location,
                                   getUserID(user) + SEPARATOR
                                   + ACCESS_DENY + SEPARATOR
                                   + rid.getPath() + SEPARATOR
                                   + getAclPermissionID(permission)
                                   + ": not authenticated"
                                   );
    }
  }

  // ------------------------------------------------------------------------
  /**
   * Access denied by ACL on a resource for a given permission, because the user
   * is not authenticated.
   *
   * @param user TBD: Description of the incoming method parameter
   * @param permission TBD: Description of the incoming method parameter
   * @param rid TBD: Description of the incoming method parameter
   */
  public void aclDeny_UserNotAuthenticated(com.sapportals.portal.security.usermanagement.IUser user,
                                           IPermission permission,
                                           RID rid) {
    if(   SecurityAudit.category.beFatal()
       && isUserToBeLogged(user)
      ) {
      SecurityAudit.category.fatalT(
                                   SecurityAudit.location,
                                   getUserID(user) + SEPARATOR
                                   + ACCESS_DENY + SEPARATOR
                                   + rid.getPath() + SEPARATOR
                                   + getPermissionID(permission)
                                   + ": not authenticated"
                                   );
    }
  }

  // ------------------------------------------------------------------------
  /**
   * Access denied by ACL on a resource for a given set of permissions, because
   * the user is not authenticated.
   *
   * @param user TBD: Description of the incoming method parameter
   * @param permissions TBD: Description of the incoming method parameter
   * @param rid TBD: Description of the incoming method parameter
   */
  public void aclDeny_UserNotAuthenticated(com.sapportals.portal.security.usermanagement.IUser user,
                                           IPermissionList permissions,
                                           RID rid) {
    if(   SecurityAudit.category.beFatal()
       && isUserToBeLogged(user)
      ) {
      SecurityAudit.category.fatalT(
                                   SecurityAudit.location,
                                   getUserID(user) + SEPARATOR
                                   + ACCESS_DENY + SEPARATOR
                                   + rid.getPath() + SEPARATOR
                                   + getPermissionIDs(permissions)
                                   + ": not authenticated"
                                   );
    }
  }

  // ------------------------------------------------------------------------
  /**
   * Access denied by ACL on several resources for a given set of permissions,
   * because the user is not authenticated.
   *
   * @param user TBD: Description of the incoming method parameter
   * @param permissions TBD: Description of the incoming method parameter
   * @param ridlist TBD: Description of the incoming method parameter
   */
  public void aclDeny_UserNotAuthenticated(com.sapportals.portal.security.usermanagement.IUser user,
                                           IPermissionList permissions,
                                           IRidList ridlist) {
    if(   SecurityAudit.category.beFatal()
       && isUserToBeLogged(user)
      ) {
      String prefix = getUserID(user) + SEPARATOR + ACCESS_DENY + SEPARATOR;
      String postfix = SEPARATOR + getPermissionIDs(permissions) + ": not authenticated";
      IRidIterator iterator = ridlist.listIterator();
      while( iterator.hasNext() ) {
        SecurityAudit.category.fatalT(
                                     SecurityAudit.location,
                                     prefix + iterator.next().getPath() + postfix
                                     );
      }
    }
  }

  // ------------------------------------------------------------------------
  /**
   * Access denied by ACL on a resource for a given permission, because an ACL
   * exists, where the given user is not listed or a negative ACL exists for
   * this user.
   *
   * @param user TBD: Description of the incoming method parameter
   * @param permission TBD: Description of the incoming method parameter
   * @param rid TBD: Description of the incoming method parameter
   */
  public void aclDeny_ByAcl(com.sapportals.portal.security.usermanagement.IUser user,
                            IAclPermission permission,
                            RID rid) {
    if(   SecurityAudit.category.beFatal()
       && isUserToBeLogged(user)
      ) {
      SecurityAudit.category.fatalT(
                                   SecurityAudit.location,
                                   getUserID(user) + SEPARATOR
                                   + ACCESS_DENY + SEPARATOR
                                   + rid.getPath() + SEPARATOR
                                   + getAclPermissionID(permission)
                                   );
    }
  }

  // ------------------------------------------------------------------------
  /**
   * Access denied by ACL on a resource for a given permission, because an ACL
   * exists, where the given user is not listed or a negative ACL exists for
   * this user.
   *
   * @param user TBD: Description of the incoming method parameter
   * @param permission TBD: Description of the incoming method parameter
   * @param rid TBD: Description of the incoming method parameter
   */
  public void aclDeny_ByAcl(com.sapportals.portal.security.usermanagement.IUser user,
                            IPermission permission,
                            RID rid) {
    if(   SecurityAudit.category.beFatal()
       && isUserToBeLogged(user)
      ) {
      SecurityAudit.category.fatalT(
                                   SecurityAudit.location,
                                   getUserID(user) + SEPARATOR
                                   + ACCESS_DENY + SEPARATOR
                                   + rid.getPath() + SEPARATOR
                                   + getPermissionID(permission)
                                   );
    }
  }

  // ------------------------------------------------------------------------
  /**
   * Access denied by ACL on a resource for a given set of permissions, because
   * an ACL exists, where the given user is not listed or a negative ACL exists
   * for this user.
   *
   * @param user TBD: Description of the incoming method parameter
   * @param permissions TBD: Description of the incoming method parameter
   * @param rid TBD: Description of the incoming method parameter
   */
  public void aclDeny_ByAcl(com.sapportals.portal.security.usermanagement.IUser user,
                            IPermissionList permissions,
                            RID rid) {
    if(   SecurityAudit.category.beFatal() 
       && isUserToBeLogged(user)
      ) {
      SecurityAudit.category.fatalT(
                                   SecurityAudit.location,
                                   getUserID(user) + SEPARATOR
                                   + ACCESS_DENY + SEPARATOR
                                   + rid.getPath() + SEPARATOR
                                   + getPermissionIDs(permissions)
                                   );
    }
  }

  // ------------------------------------------------------------------------
  /**
   * Access denied by ACL on a resource for a given permission, because the
   * permission can not be mapped to the ACLs permissions.
   *
   * @param user TBD: Description of the incoming method parameter
   * @param permission TBD: Description of the incoming method parameter
   * @param rid TBD: Description of the incoming method parameter
   */
  public void aclDeny_PermissionNotMapped(com.sapportals.portal.security.usermanagement.IUser user,
                                          IPermission permission,
                                          RID rid) {
    if(   SecurityAudit.category.beFatal()
       && isUserToBeLogged(user)
      ) {
      SecurityAudit.category.fatalT(
                                   SecurityAudit.location,
                                   getUserID(user) + SEPARATOR
                                   + ACCESS_DENY + SEPARATOR
                                   + rid.getPath() + SEPARATOR
                                   + getPermissionID(permission)
                                   + ": unmapped"
                                   );
    }
  }

  // ------------------------------------------------------------------------
  /**
   * Access denied by ACL on a resource for a given set of permissions, because
   * the permission can not be mapped to the ACLs permissions.
   *
   * @param user TBD: Description of the incoming method parameter
   * @param permissions TBD: Description of the incoming method parameter
   * @param rid TBD: Description of the incoming method parameter
   */
  public void aclDeny_PermissionNotMapped(com.sapportals.portal.security.usermanagement.IUser user,
                                          IPermissionList permissions,
                                          RID rid) {
    if(   SecurityAudit.category.beFatal()
       && isUserToBeLogged(user)
      ) {
      SecurityAudit.category.fatalT(
                                   SecurityAudit.location,
                                   getUserID(user) + SEPARATOR
                                   + ACCESS_DENY + SEPARATOR
                                   + rid.getPath() + SEPARATOR
                                   + getPermissionIDs(permissions)
                                   + ": unmapped"
                                   );
    }
  }

  // ------------------------------------------------------------------------
  /**
   * Access granted by ACL on a resource for a given permission, because the
   * user has the permission by configuration.
   *
   * @param user TBD: Description of the incoming method parameter
   * @param permission TBD: Description of the incoming method parameter
   * @param rid TBD: Description of the incoming method parameter
   */
  public void aclGrant_BySystemPrincipal(com.sapportals.portal.security.usermanagement.IUser user,
                                         IPermission permission,
                                         RID rid) {
    if(   SecurityAudit.category.beInfo() 
       && isUserToBeLogged(user)
      ) {
      SecurityAudit.category.infoT(
                                  SecurityAudit.location,
                                  getUserID(user) + SEPARATOR
                                  + ACCESS_GRANT + SEPARATOR
                                  + rid.getPath() + SEPARATOR
                                  + getPermissionID(permission)
                                  + ": configured"
                                  );
    }
  }

  // ------------------------------------------------------------------------
  /**
   * Access granted by ACL on a resource for a given set of permissions, because
   * the user has the permissions by configuration.
   *
   * @param user TBD: Description of the incoming method parameter
   * @param permissions TBD: Description of the incoming method parameter
   * @param rid TBD: Description of the incoming method parameter
   */
  public void aclGrant_BySystemPrincipal(com.sapportals.portal.security.usermanagement.IUser user,
                                        IPermissionList permissions,
                                        RID rid) {
    if(   SecurityAudit.category.beInfo()
       && isUserToBeLogged(user)
      ) {
      SecurityAudit.category.infoT(
                                  SecurityAudit.location,
                                  getUserID(user) + SEPARATOR
                                  + ACCESS_GRANT + SEPARATOR
                                  + rid.getPath() + SEPARATOR
                                  + getPermissionIDs(permissions)
                                  + ": configured"
                                  );
    }
  }

  // ------------------------------------------------------------------------
  /**
   * Access granted by ACL on several resources for a given set of permissions,
   * because the user has the permissions by configuration.
   *
   * @param user TBD: Description of the incoming method parameter
   * @param permissions TBD: Description of the incoming method parameter
   * @param ridlist TBD: Description of the incoming method parameter
   */
  public void aclGrant_BySystemPrincipal(com.sapportals.portal.security.usermanagement.IUser user,
                                         IPermissionList permissions,
                                         IRidList ridlist) {
    if(   SecurityAudit.category.beInfo()
       && isUserToBeLogged(user)
      ) {
      String prefix = getUserID(user) + SEPARATOR + ACCESS_GRANT + SEPARATOR;
      String postfix = SEPARATOR + getPermissionIDs(permissions) + ": configured";
      IRidIterator iterator = ridlist.listIterator();
      while( iterator.hasNext() ) {
        SecurityAudit.category.infoT(
                                    SecurityAudit.location,
                                    prefix + iterator.next().getPath() + postfix
                                    );
      }
    }
  }

  // ------------------------------------------------------------------------
  /**
   * Access granted by ACL on a resource for a given permission, because no ACL
   * was found.
   *
   * @param user TBD: Description of the incoming method parameter
   * @param permission TBD: Description of the incoming method parameter
   * @param rid TBD: Description of the incoming method parameter
   */
  public void aclGrant_NoExistingAcl(com.sapportals.portal.security.usermanagement.IUser user,
                                     IAclPermission permission,
                                     RID rid) {
    if(   SecurityAudit.category.beInfo()
       && isUserToBeLogged(user)
      ) {
      SecurityAudit.category.infoT(
                                  SecurityAudit.location,
                                  getUserID(user) + SEPARATOR
                                  + ACCESS_GRANT + SEPARATOR
                                  + rid.getPath() + SEPARATOR
                                  + getAclPermissionID(permission)
                                  );
    }
  }

  // ------------------------------------------------------------------------
  /**
   * Access granted by ACL on a resource for a given permission, because no ACL
   * was found.
   *
   * @param user TBD: Description of the incoming method parameter
   * @param permission TBD: Description of the incoming method parameter
   * @param rid TBD: Description of the incoming method parameter
   */
  public void aclGrant_NoExistingAcl(com.sapportals.portal.security.usermanagement.IUser user,
                                     IPermission permission,
                                     RID rid) {
    if(   SecurityAudit.category.beInfo()
       && isUserToBeLogged(user)
      ) {
      SecurityAudit.category.infoT(
                                  SecurityAudit.location,
                                  getUserID(user) + SEPARATOR
                                  + ACCESS_GRANT + SEPARATOR
                                  + rid.getPath() + SEPARATOR
                                  + getPermissionID(permission)
                                  );
    }
  }

  // ------------------------------------------------------------------------
  /**
   * Access granted by ACL on a resource for a given set of permissions, because
   * no ACL was found.
   *
   * @param user TBD: Description of the incoming method parameter
   * @param permissions TBD: Description of the incoming method parameter
   * @param rid TBD: Description of the incoming method parameter
   */
  public void aclGrant_NoExistingAcl(com.sapportals.portal.security.usermanagement.IUser user,
                                     IPermissionList permissions,
                                     RID rid) {
    if(   SecurityAudit.category.beInfo()
       && isUserToBeLogged(user)
      ) {
      SecurityAudit.category.infoT(
                                  SecurityAudit.location,
                                  getUserID(user) + SEPARATOR
                                  + ACCESS_GRANT + SEPARATOR
                                  + rid.getPath() + SEPARATOR
                                  + getPermissionIDs(permissions)
                                  );
    }
  }

  // ------------------------------------------------------------------------
  /**
   * Access granted by ACL on a resource for a given permission, because at
   * least one positive and no negative ACL was found for the user.
   *
   * @param user TBD: Description of the incoming method parameter
   * @param permission TBD: Description of the incoming method parameter
   * @param rid TBD: Description of the incoming method parameter
   */
  public void aclGrant_ByAcl(com.sapportals.portal.security.usermanagement.IUser user,
                             IAclPermission permission,
                             RID rid) {
    if(   SecurityAudit.category.beInfo()
       && isUserToBeLogged(user)
      ) {
      SecurityAudit.category.infoT(
                                  SecurityAudit.location,
                                  getUserID(user) + SEPARATOR
                                  + ACCESS_GRANT + SEPARATOR
                                  + rid.getPath() + SEPARATOR
                                  + getAclPermissionID(permission)
                                  );
    }
  }

  // ------------------------------------------------------------------------
  /**
   * Access granted by ACL on a resource for a given permission, because at
   * least one positive and no negative ACL was found for the user.
   *
   * @param user TBD: Description of the incoming method parameter
   * @param permission TBD: Description of the incoming method parameter
   * @param rid TBD: Description of the incoming method parameter
   */
  public void aclGrant_ByAcl(com.sapportals.portal.security.usermanagement.IUser user,
                             IPermission permission,
                             RID rid) {
    if(   SecurityAudit.category.beInfo()
       && isUserToBeLogged(user)
      ) {
      SecurityAudit.category.infoT(
                                  SecurityAudit.location,
                                  getUserID(user) + SEPARATOR
                                  + ACCESS_GRANT + SEPARATOR
                                  + rid.getPath() + SEPARATOR
                                  + getPermissionID(permission)
                                  );
    }
  }

  // ------------------------------------------------------------------------
  /**
   * Access granted by ACL on a resource for a given set of permissions, because
   * at least one positive and no negative ACL was found for the user.
   *
   * @param user TBD: Description of the incoming method parameter
   * @param permissions TBD: Description of the incoming method parameter
   * @param rid TBD: Description of the incoming method parameter
   */
  public void aclGrant_ByAcl(com.sapportals.portal.security.usermanagement.IUser user,
                             IPermissionList permissions,
                             RID rid) {
    if(   SecurityAudit.category.beInfo()
       && isUserToBeLogged(user)
      ) {
      SecurityAudit.category.infoT(
                                  SecurityAudit.location,
                                  getUserID(user) + SEPARATOR
                                  + ACCESS_GRANT + SEPARATOR
                                  + rid.getPath() + SEPARATOR
                                  + getPermissionIDs(permissions)
                                  );
    }
  }


  // ------------------------------------------------------------------------
  /**
   * An (empty) ACL was created for a resource.
   *
   * @param user TBD: Description of the incoming method parameter
   * @param owner TBD: Description of the incoming method parameter
   * @param rid TBD: Description of the incoming method parameter
   */
  public void aclCreate(com.sapportals.portal.security.usermanagement.IUser user,
                        com.sapportals.portal.security.usermanagement.IUser owner,
                        RID rid) {
    if(   SecurityAudit.category.beError()
       && isUserToBeLogged(user)
      ) {
      SecurityAudit.category.errorT(
                                   SecurityAudit.location,
                                   getUserID(user) + SEPARATOR
                                   + ACL_CREATE + SEPARATOR
                                   + rid.getPath() + SEPARATOR
                                   + "owner: " + getUserID(owner)
                                   );
    }
  }

  // ------------------------------------------------------------------------
  /**
   * An ACL was created for a resource by copying another ACL.
   *
   * @param user TBD: Description of the incoming method parameter
   * @param otherAcl TBD: Description of the incoming method parameter
   * @param rid TBD: Description of the incoming method parameter
   */
  public void aclCreate_FromOther(com.sapportals.portal.security.usermanagement.IUser user,
                                  IResourceAcl otherAcl,
                                  RID rid) {
    if(   SecurityAudit.category.beError()
       && isUserToBeLogged(user)
      ) {
      String owners;
      try {
        owners = "owner: " + getPrincipalIDs(otherAcl.getOwners());
      }
      catch( AclPersistenceException e ) {
        owners = "n/a";
      }
      SecurityAudit.category.errorT(
                                   SecurityAudit.location,
                                   getUserID(user) + SEPARATOR
                                   + ACL_CREATE + SEPARATOR
                                   + rid.getPath() + SEPARATOR
                                   + owners
                                   );
    }
  }

  // ------------------------------------------------------------------------
  /**
   * An ACL was deleted for a resource.
   *
   * @param user TBD: Description of the incoming method parameter
   * @param rid TBD: Description of the incoming method parameter
   */
  public void aclDelete(com.sapportals.portal.security.usermanagement.IUser user,
                        RID rid) {
    if(   SecurityAudit.category.beError()
       && isUserToBeLogged(user)
      ) {
      SecurityAudit.category.errorT(
                                   SecurityAudit.location,
                                   getUserID(user) + SEPARATOR
                                   + ACL_DELETE + SEPARATOR
                                   + rid.getPath() + SEPARATOR
                                   );
    }
  }

  // ------------------------------------------------------------------------
  /**
   * A new permission was created.
   *
   * @param permissionName TBD: Description of the incoming method parameter
   */
  public void aclPermissionCreate(String permissionName) {
    if( SecurityAudit.category.beWarning() ) {
      SecurityAudit.category.warningT(
                                     SecurityAudit.location,
                                     "<system>" + SEPARATOR
                                     + PERMISSION_CREATE + SEPARATOR
                                     + permissionName + SEPARATOR
                                     );
    }
  }

  // ------------------------------------------------------------------------
  /**
   * A permission was deleted.
   *
   * @param permissionName TBD: Description of the incoming method parameter
   */
  public void aclPermissionRemove(String permissionName) {
    if( SecurityAudit.category.beWarning() ) {
      SecurityAudit.category.warningT(
                                     SecurityAudit.location,
                                     "<system>" + SEPARATOR
                                     + PERMISSION_DELETE + SEPARATOR
                                     + permissionName + SEPARATOR
                                     );
    }
  }

  // ------------------------------------------------------------------------
  /**
   * A permission was added as supported permission to a specific object type.
   *
   * @param permissionName TBD: Description of the incoming method parameter
   * @param objectType TBD: Description of the incoming method parameter
   */
  public void aclSupportedPermissionAdd(String permissionName,
                                        String objectType) {
    if( SecurityAudit.category.beWarning() ) {
      SecurityAudit.category.warningT(
                                     SecurityAudit.location,
                                     "<system>" + SEPARATOR
                                     + SUPPORTEDPERMISSION_ADD + SEPARATOR
                                     + permissionName + SEPARATOR
                                     + objectType
                                     );
    }
  }

  // ------------------------------------------------------------------------
  /**
   * A permission was removed as supported permission from a specific object
   * type.
   *
   * @param permissionName TBD: Description of the incoming method parameter
   * @param objectType TBD: Description of the incoming method parameter
   */
  public void aclSupportedPermissionRemove(String permissionName,
                                           String objectType) {
    if( SecurityAudit.category.beWarning() ) {
      SecurityAudit.category.warningT(
                                     SecurityAudit.location,
                                     "<system>" + SEPARATOR
                                     + SUPPORTEDPERMISSION_REMOVE + SEPARATOR
                                     + permissionName + SEPARATOR
                                     + objectType
                                     );
    }
  }

}
