/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id: //kmgmt/bc.rf/60NW_SP_COR/src/_runtime/java/api/com/sapportals/wcm/repository/security/ResourceAclManager.java#7 $
 */

package com.sapportals.wcm.repository.security;

import com.sap.tc.logging.Location;

import com.sapportals.portal.security.usermanagement.IUMEventHandler;
import com.sapportals.portal.security.usermanagement.IUMObserver;
import com.sapportals.portal.security.usermanagement.IUMPrincipal;
import com.sapportals.portal.security.usermanagement.IUser;
import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.repository.*;
import com.sapportals.wcm.repository.enum.*;
import com.sapportals.wcm.repository.manager.IAclInheritanceController;
import com.sapportals.wcm.repository.manager.IRepositoryManager;
import com.sapportals.wcm.repository.manager.IManagerFactory;
import com.sapportals.wcm.util.cache.*;
import com.sapportals.wcm.util.acl.*;
import com.sapportals.wcm.util.logging.LoggingFormatter;
import com.sapportals.wcm.util.uri.RID;
import com.sapportals.wcm.util.usermanagement.WPUMFactory;

import java.util.*;

/**
 * Default implementation of the IResourceAclManager interface
 */
public final class ResourceAclManager
implements IResourceAclManagerRaw, IUMObserver {
  
  private final static String UM_EVENT_NAMESPACE = "wcmacl";

  private com.sap.tc.logging.Location s_log = null ;
  private com.sapportals.wcm.repository.security.SecurityAudit audit = null ;


  private final IAclManagerRaw m_aclManager;
  private final IResourceAclChangeTracker m_aclChangeTracker;// report changes of resource acls to


  /**
   * Construct
   *
   * @param aclChangeTracker TBD: Description of the incoming method parameter
   * @param configBlock TBD: Description of the incoming method parameter
   * @exception WcmException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @exception AclLoadClassException Exception raised in failure situation
   * @exception AclPersistenceException Exception raised in failure situation
   * @exception InvalidConfigException Exception raised in failure situation
   */
  public ResourceAclManager(IResourceAclChangeTracker aclChangeTracker, String configBlock)
    throws WcmException, ResourceException, AclLoadClassException, AclPersistenceException, InvalidConfigException {

    s_log = com.sap.tc.logging.Location.getLocation(
      com.sapportals.wcm.repository.security.ResourceAclManager.class);
    audit = com.sapportals.wcm.repository.security.SecurityAudit.getInstance();
    m_aclChangeTracker = aclChangeTracker;

    AclManagerFactory aclManagerFactory = AclManagerFactory.getInstance();
    IAclManager aclManager = aclManagerFactory.getAclManager(configBlock, null);  
    if( !(aclManager instanceof IAclManagerRaw) ) {
      throw new InvalidConfigException("base AclManager " + aclManager.getClass().getName() + " doesn't implement IAclManagerRaw interface");
    }
    m_aclManager = (IAclManagerRaw)aclManager;

    IUMEventHandler evHdl = WPUMFactory.getEventHandler();
    if (evHdl != null) {
      evHdl.registerForEvent(UM_EVENT_NAMESPACE, IUMEventHandler.USER_DELETED, this);
      // evHdl.registerForEvent(UM_EVENT_NAMESPACE, IUMEventHandler.USER_UPDATED, this);
      evHdl.registerForEvent(UM_EVENT_NAMESPACE, IUMEventHandler.GROUP_DELETED, this);
      // evHdl.registerForEvent(UM_EVENT_NAMESPACE, IUMEventHandler.GROUP_UPDATED, this);
      evHdl.registerForEvent(UM_EVENT_NAMESPACE, IUMEventHandler.ROLE_DELETED, this);
      // evHdl.registerForEvent(UM_EVENT_NAMESPACE, IUMEventHandler.ROLE_UPDATED, this);
    }
  }


  /**
   * Construct
   *
   * @param aclChangeTracker TBD: Description of the incoming method parameter
   * @param configBlock TBD: Description of the incoming method parameter
   * @exception WcmException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   * @exception AclLoadClassException Exception raised in failure situation
   * @exception AclPersistenceException Exception raised in failure situation
   * @exception InvalidConfigException Exception raised in failure situation
   */
  public ResourceAclManager(IResourceAclChangeTracker aclChangeTracker, String configBlock, IClusterCacheArea cacheArea)
    throws WcmException, ResourceException, AclLoadClassException, AclPersistenceException, InvalidConfigException {

    s_log = com.sap.tc.logging.Location.getLocation(
      com.sapportals.wcm.repository.security.ResourceAclManager.class);
    audit = com.sapportals.wcm.repository.security.SecurityAudit.getInstance();
    m_aclChangeTracker = aclChangeTracker;

    AclManagerFactory aclManagerFactory = AclManagerFactory.getInstance();
    IAclManager aclManager = aclManagerFactory.getAclManager(configBlock, null);  
    if( !(aclManager instanceof IAclManagerRaw) ) {
      throw new InvalidConfigException("base AclManager " + aclManager.getClass().getName() + " doesn't implement IAclManagerRaw interface");
    }
    m_aclManager = (IAclManagerRaw)aclManager;
   
    IUMEventHandler evHdl = WPUMFactory.getEventHandler();
    if (evHdl != null) {
      evHdl.registerForEvent(UM_EVENT_NAMESPACE, IUMEventHandler.USER_DELETED, this);
      // evHdl.registerForEvent(UM_EVENT_NAMESPACE, IUMEventHandler.USER_UPDATED, this);
      evHdl.registerForEvent(UM_EVENT_NAMESPACE, IUMEventHandler.GROUP_DELETED, this);
      // evHdl.registerForEvent(UM_EVENT_NAMESPACE, IUMEventHandler.GROUP_UPDATED, this);
      evHdl.registerForEvent(UM_EVENT_NAMESPACE, IUMEventHandler.ROLE_DELETED, this);
      // evHdl.registerForEvent(UM_EVENT_NAMESPACE, IUMEventHandler.ROLE_UPDATED, this);
    }
  }
  

  public IResourceAcl getAclRaw(IResource resource)
                         throws AclPersistenceException,
                                ResourceException {
    return getAclRaw(resource, -1);
  }


  public IResourceAcl getAclRaw(IResource resource,
                                long dbversion)
                         throws AclPersistenceException,
                                ResourceException {

    if( resource == null ) {
      throw new java.lang.IllegalArgumentException();
    }
    
    IAcl acl = m_aclManager.getAclRaw(getResourcePath(resource), dbversion);
    if( acl == null ) {
      return null;
    }
    
    if( acl instanceof IAclRaw ) {
      return new ResourceAclRaw(this, resource, (IAclRaw)acl); 
    } else {
      return new ResourceAcl(this, resource, acl);
    }
    
  }


  public IResourceAcl getAcl(IResource resource)
    throws AclPersistenceException, ResourceException {
    return getAcl(resource, -1);
  }

  public IResourceAcl getAcl(IResource resource, long dbversion)
    throws AclPersistenceException, ResourceException {

    if (resource == null) {
      throw new java.lang.IllegalArgumentException();
    }
    IAcl acl = m_aclManager.getAcl(getResourcePath(resource), dbversion);
    if (acl == null) {
      return null;
    }
    return new ResourceAcl(this, resource, acl);
  }

  public IResourceAcl[] getAcls(RID[] rids)
    throws AclPersistenceException, ResourceException {

    if (rids == null) {
      throw new java.lang.IllegalArgumentException();
    }

    // get array with RID string values
    String[] ids = new String[rids.length];
    for (int i = 0; i < rids.length; i++) {
      ids[i] = getRidPath(rids[i]);
    }

    // get ACLs for RIDs
    IAcl[] acls = m_aclManager.getAcls(ids);

    // construct result array
    IResourceAcl[] result = new IResourceAcl[rids.length];
    for (int i = 0; i < ids.length && i < acls.length; i++) {
      result[i] = (acls[i] == null) ? null : new ResourceAcl(this, acls[i]);
    }

    return result;
  }

  public IResourceAcl getInheritedAcl ( IResource child, RID r, long dbversion )
  throws AclPersistenceException, ResourceException {
    if ( r == null ) {
      throw new java.lang.IllegalArgumentException() ;
    }
    IResourceContext ctx = child.getContext() ;
    IRepositoryManager rm = child.getRepositoryManager() ;
    IAclInheritanceController aic = (rm instanceof IAclInheritanceController)
      ?(IAclInheritanceController)rm:null ;
    r = (aic!=null)?aic.getLegator(r, ctx):r.parent() ;
    while (r != null && !r.isRoot() && r.length() > 0) {
      IAcl acl = m_aclManager.getAcl(r.getPath(), dbversion);
      if (acl != null) {
        return new ResourceAcl(this, acl);
      }
      r = (aic!=null)?aic.getLegator(r, ctx):r.parent() ;
    }
    return null;
  }
  
  public IResourceAcl getInheritedAcl ( IResource resource )
  throws AclPersistenceException, ResourceException {
    if (resource == null) {
      throw new java.lang.IllegalArgumentException();
    }
    RID r = resource.getRID();
    long dbversion = m_aclManager.getDBVersion();

    IResourceContext ctx = resource.getContext() ;
    IRepositoryManager rm = resource.getRepositoryManager() ;
    IAclInheritanceController aic = (rm instanceof IAclInheritanceController)
      ?(IAclInheritanceController)rm:null ;
    IAcl acl ;
    do {
      r = (aic!=null)?aic.getLegator(r, ctx):r.parent() ;
      if (( r != null )
      && (( acl = m_aclManager.getAcl ( r.getPath(), dbversion )) != null )) {
        resource = ResourceFactory.getInstance().getResource ( r, ctx, false ) ;
        return new ResourceAcl ( this, resource, acl ) ;
      }
    } while (r != null && !r.isRoot() && r.length() > 0) ;
    return null;
  }

  public IResourceAcl[] getInheritedAcls(RID[] rids, IResourceContext context)
  throws AclPersistenceException, ResourceException {
    /*
    * new version may get more ACLs than needed, but makes only
    * one call to get them
    */
    if (rids == null) {
      throw new java.lang.IllegalArgumentException();
    }
    if (rids.length == 0) {
      return new IResourceAcl[0];
    }

    /*
    * collect ids of all parents of the passed RIDs in a hash table
    */
    HashMap parents = new HashMap(rids.length * 2);
    IRepositoryManager repMgr = null ;
    IManagerFactory mf = ResourceFactory.getInstance().getManagerFactory() ;
    RID rmRid = null ;
    IAclInheritanceController aic = null ;
    /*
    * best case: all rids belong to the same rep mgr which means only
    * a single resource will be created by a call to getResource()
    * instead of one per rid
    */
    for ( int i = 0 ; i < rids.length ; i++ ) {
      RID rid = rids[i] ;
      if ( repMgr == null ) {
        /*
        * a newly located rep mgr will always prefix-match the resource
        * it has been derived from
        */
        if (( repMgr = mf.getRepositoryManager ( rid )) != null ) {
          rmRid = RID.getRID ( repMgr.getPrefix()) ;
          aic = (repMgr instanceof IAclInheritanceController)?
            (IAclInheritanceController)repMgr:null ;
        }
      }
      else {
        /*
        * verify that the current rep mgr owns the resource corresponding
        * to the given rid
        */
        if ( !rmRid.isAncestorOfOrSelf ( rid )) {
          if (( repMgr = mf.getRepositoryManager ( rid )) != null ) {
            rmRid = RID.getRID ( repMgr.getPrefix()) ;
            aic = (repMgr instanceof IAclInheritanceController)?
              (IAclInheritanceController)repMgr:null ;
          }
        }
      }
      if ( repMgr != null ) {
        rid = (aic!=null)?aic.getLegator(rid, context):rid.parent() ;
        while (rid != null && !rid.isRoot() && rid.length() > 0) {
          parents.put(rid.getPath(), null);
          rid = (aic!=null)?aic.getLegator(rid, context):rid.parent() ;
        }
      }
    }    

    // put the hashed ids to an array and get their ACLs
    String[] ids = new String[parents.size()];
    int idIndex = 0;
    Set keys = parents.keySet();
    Iterator parentIterator = keys.iterator();
    while (parentIterator.hasNext()) {
      ids[idIndex++] = (String)parentIterator.next();
    }
    IAcl[] acls = m_aclManager.getAcls(ids);

    // put the found ACLs to the HashTable
    for (int i = 0; i < acls.length && i < ids.length; i++) {
      if (acls[i] != null) {
        parents.put(ids[i], acls[i]);
      }
    }

    // for each passed RID get the inherited ACL from the parents hash table
    IResourceAcl[] result = new IResourceAcl[rids.length];
    /*
    * best case again: if all rids belong to the same rep mgr then repMgr is
    * still set to the correct value and each rid is an ancestor of rmRid
    */
    for ( int i = 0 ; i < rids.length ; i++ ) {
      RID rid = rids[i] ;

      if ( repMgr == null ) {
        if (( repMgr = mf.getRepositoryManager ( rid )) != null ) {
          rmRid = RID.getRID ( repMgr.getPrefix()) ;
          aic = (repMgr instanceof IAclInheritanceController)?
            (IAclInheritanceController)repMgr:null ;
        }
      }
      else {
        if ( !rmRid.isAncestorOfOrSelf ( rid )) {
          if (( repMgr = mf.getRepositoryManager ( rid )) != null ) {
            rmRid = RID.getRID ( repMgr.getPrefix()) ;
            aic = (repMgr instanceof IAclInheritanceController)?
              (IAclInheritanceController)repMgr:null ;
          }
        }
      }
      result[i] = null;
      if ( repMgr != null ) {
        while (rid != null && !rid.isRoot() && rid.length() > 0) {
          IAcl acl = (IAcl)parents.get(rid.getPath());
          if (acl != null) {
            result[i] = new ResourceAcl(this, acl);
            break;
          }
          else {
            rid = (aic!=null)?aic.getLegator(rid, context):rid.parent() ;
          }
        }
      }
    }
    return result;
  }

  public IObjectTypeList getSupportedObjectTypes()
    throws AclPersistenceException {

    return m_aclManager.getSupportedObjectTypes();
  }

  public IObjectType getObjectType(IResource resource)
    throws AclPersistenceException, ResourceException {

    if (resource == null) {
      throw new java.lang.IllegalArgumentException();
    }
    return resource.isCollection() ? ObjectType.NODE : ObjectType.LEAF;
  }

  public IAclPermissionList getSupportedPermissions(IResource resource)
    throws AclPersistenceException, ResourceException {

    if (resource == null) {
      throw new java.lang.IllegalArgumentException();
    }
    return m_aclManager.getSupportedPermissions(getObjectType(resource));
  }

  public IAclPermission getPermission(String name)
    throws AclPersistenceException {

    if (name == null) {
      throw new java.lang.IllegalArgumentException();
    }
    return m_aclManager.getPermission(name);
  }

  public boolean isPermissionUsedInAcl(IAclPermission permission)
    throws AclPersistenceException {

    if (permission == null) {
      throw new java.lang.IllegalArgumentException();
    }
    return m_aclManager.isPermissionUsedInAcl(permission);
  }

  public boolean isReadOnly() {

    return false;
  }

  public IResourceList propagateAcl_Remove(IResource resource)
    throws AclPersistenceException, ResourceException, NoAclException, InvalidClassException, NotAuthorizedException {

    if (resource == null) {
      throw new IllegalArgumentException();
    }
    if (!resource.isCollection()) {
      throw new InvalidClassException();
    }

    IResourceList result = new ResourceList();
    recursiveRemovePropagation((ResourceAcl)getAcl(resource),
      ((ICollection)resource).getChildren(), result);
    m_aclChangeTracker.someAclsChanged(resource.getRID());

    return result.size() == 0 ? null : result;
  }

  public boolean[] checkPermissions(IResourceAcl[] acls, IUser user, IAclPermission[] permissions)
    throws AclException {

    if (acls == null || user == null || permissions == null) {
      throw new java.lang.IllegalArgumentException();
    }
    if (acls.length == 0) {
      return new boolean[0];
    }
    IAcl[] localAcls = new IAcl[acls.length];
    for (int i = 0; i < acls.length; i++) {
      localAcls[i] = ((ResourceAcl)acls[i]).getAcl();
    }
    return m_aclManager.checkPermissions(localAcls, user, permissions);
  }

  /**
   * Register a repository manager which uses this resource ACL manager
   *
   * @param manager TBD: Description of the incoming method parameter
   * @exception WcmException Exception raised in failure situation
   */
  public void registerRepositoryManager(IRepositoryManager manager)
    throws WcmException {

    ResourceAclEventHandler.getInstance().register(this, manager);
  }

  /*
   * public IResourceAclEntry createAclEntry(IUMPrincipal principal, boolean negative, IAclPermission permission, int sortIndex, boolean propagate) throws InvalidClassException
   * {
   * return new ResourceAclEntry(m_aclManager.createAclEntry(principal,negative,permission,sortIndex,propagate));
   * }
   */
  public IResourceAclEntry createAclEntry(IUMPrincipal principal, boolean negative, IAclPermission permission, int sortIndex)
    throws AclPersistenceException, UnsupportedOperationException, InvalidClassException {

    if (principal == null || permission == null) {
      throw new java.lang.IllegalArgumentException();
    }
    if (negative) {
      throw new UnsupportedOperationException();
    }
    // propagate always
    return new ResourceAclEntry(m_aclManager.createAclEntry(principal, negative, permission, sortIndex, true));
  }

  public boolean areNegativeAclEntriesSupported() {

    return false;
  }

  public IResourceAcl createAcl(IResource resource)
    throws AclPersistenceException, NotAuthorizedException, AclExistsException, ResourceException {

    if (resource == null) {
      throw new java.lang.IllegalArgumentException();
    }
    if (getAcl(resource) != null) {
      throw new AclExistsException();
    }
    IUser owner = getLoggedInUser(resource);
    IResourceAcl inheritedAcl = getInheritedAcl(resource);
    if (inheritedAcl != null && !inheritedAcl.isAuthorized(owner)) {
      throw new NotAuthorizedException();
    }
    this.audit.aclCreate(owner, owner, resource.getRID());
    IAcl acl = m_aclManager.createAcl(owner, getResourcePath(resource), getObjectType(resource));
    addInheritedPermissions(owner, acl, resource);
    m_aclChangeTracker.oneAclChanged(resource.getRID());
    return new ResourceAcl(this, resource, acl);
  }

  public boolean assignForeignAcl(IResourceAcl foreignAcl, IResource resource)
    throws AclPersistenceException, NotAuthorizedException, AclExistsException, ResourceException {

    if (foreignAcl == null || resource == null) {
      throw new java.lang.IllegalArgumentException();
    }
    if (getAcl(resource) != null) {
      throw new AclExistsException();
    }

    IUMPrincipal user = getLoggedInUser(resource);
    IResourceAcl inheritedAcl = getInheritedAcl(resource);
    if (inheritedAcl != null && !inheritedAcl.isAuthorized(user)) {
      throw new NotAuthorizedException();
    }

    // get the owners of the foreign acl
    IUMPrincipalList owners = foreignAcl.getOwners();
    if (owners == null) {
      return false;
    }
    IUMPrincipalListIterator ownerIterator = owners.iterator();
    if (ownerIterator == null || !ownerIterator.hasNext()) {
      return false;
    }
    IUMPrincipal firstOwner = ownerIterator.next();
    if (firstOwner == null) {
      return false;
    }

    // create a new acl
    IAcl newAcl = m_aclManager.createAcl(firstOwner, getResourcePath(resource), getObjectType(resource));

    // assign the owners
    while (ownerIterator.hasNext()) {
      IUMPrincipal owner = ownerIterator.next();
      if (owner != null) {
        try {
          if (!newAcl.addOwner(firstOwner, owner)) {
            m_aclManager.removeAcl(firstOwner, getResourcePath(resource));
            return false;
          }
        }
        catch (NotAuthorizedException e) {
          s_log.errorT("assignForeignAcl(445)", "unexpected error" + LoggingFormatter.extractCallstack(e));
          // will not be thrown; just to keep the exception from the method prototype
        }
      }
    }

    // check the acl entries of the foreign acl and add the matching ones to the new acl
    IResourceAclEntryList entries = foreignAcl.getEntries();
    if (entries == null) {
      IResourceAclEntryListIterator entryIterator = entries.iterator();
      while (entryIterator != null && entryIterator.hasNext()) {
        IResourceAclEntry foreignEntry = entryIterator.next();
        if (foreignEntry != null) {
          IAclPermission localPermission = getMatchingPermission(resource, foreignEntry.getPermission());
          if (localPermission != null) {
            try {
              /*
               * isPropagated was removed from interface
               * IAclEntry newEntry=m_aclManager.createAclEntry(foreignEntry.getPrincipal(),foreignEntry.isNegative(),localPermission,foreignEntry.getSortIndex(),foreignEntry.isPropagated());
               */
              IAclEntry newEntry = m_aclManager.createAclEntry(foreignEntry.getPrincipal(), foreignEntry.isNegative(), localPermission, foreignEntry.getSortIndex(), true);
              // propagate always

              if (newEntry == null) {
                m_aclManager.removeAcl(firstOwner, getResourcePath(resource));
                return false;
              }
              if (!newAcl.addEntry(firstOwner, newEntry)) {
                m_aclManager.removeAcl(firstOwner, getResourcePath(resource));
                return false;
              }
            }
            catch (InvalidClassException e) {
              s_log.errorT("assignForeignAcl(478)", "unexpected error" + LoggingFormatter.extractCallstack(e));
              // will not be thrown; just to keep the exception from the method prototype
            }
            catch (NotAuthorizedException e) {
              s_log.errorT("assignForeignAcl(482)", "unexpected error" + LoggingFormatter.extractCallstack(e));
              // will not be thrown; just to keep the exception from the method prototype
            }
            catch (AlreadyAssignedToAclException e) {
              s_log.errorT("assignForeignAcl(486)", "unexpected error" + LoggingFormatter.extractCallstack(e));
              // will not be thrown; just to keep the exception from the method prototype
            }
            catch (PermissionNotSupportedException e) {
              s_log.errorT("assignForeignAcl(490)", "unexpected error" + LoggingFormatter.extractCallstack(e));
              // will not be thrown; just to keep the exception from the method prototype
            }
          }
        }
      }
    }

    try {
      if (!addInheritedPermissions(firstOwner, newAcl, resource)) {
        try {
          m_aclManager.removeAcl(firstOwner, getResourcePath(resource));
        }
        catch (Exception e) {
          s_log.errorT("assignForeignAcl(504)", "unexpected error" + LoggingFormatter.extractCallstack(e));
        }
        return false;
      }
    }
    catch (NotAuthorizedException e) {
      s_log.errorT("assignForeignAcl(510)", "unexpected error" + LoggingFormatter.extractCallstack(e));
      // will not be thrown; just to keep the exception from the method prototype
    }

    this.audit.aclCreate_FromOther(getLoggedInUser(resource), foreignAcl, resource.getRID());
    m_aclChangeTracker.oneAclChanged(resource.getRID());
    return true;
  }

  /**
   * Remove an ACL
   *
   * @param resource TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception AclPersistenceException Exception raised in failure situation
   * @exception NotAuthorizedException Exception raised in failure situation
   * @exception ResourceException Exception raised in failure situation
   */
  public boolean removeAcl(IResource resource)
    throws AclPersistenceException, NotAuthorizedException, ResourceException {

    if (resource == null) {
      throw new java.lang.IllegalArgumentException();
    }
    if (!m_aclManager.removeAcl(getLoggedInUser(resource), getResourcePath(resource))) {
      return false;
    }
    this.audit.aclDelete(getLoggedInUser(resource), resource.getRID());
    m_aclChangeTracker.oneAclChanged(resource.getRID());
    return true;
  }

  public boolean removeAcl(IResourceAcl acl)
    throws AclPersistenceException, NotAuthorizedException, InvalidClassException, ResourceException {

    if (acl == null) {
      throw new java.lang.IllegalArgumentException();
    }
    if (!(acl instanceof ResourceAcl)) {
      throw new InvalidClassException();
    }
    if (!m_aclManager.removeAcl(getLoggedInUser(acl.getResource()), ((ResourceAcl)acl).getAcl())) {
      return false;
    }
    this.audit.aclDelete(getLoggedInUser(acl.getResource()), acl.getResource().getRID());
    m_aclChangeTracker.oneAclChanged(acl.getResource().getRID());
    return true;
  }

  public boolean addSupportedPermission(IObjectType objectType, IAclPermission permission)
    throws AclPersistenceException, PredefinedPermissionException {

    if (objectType == null || permission == null) {
      throw new java.lang.IllegalArgumentException();
    }
    this.audit.aclSupportedPermissionAdd(permission.getName(), objectType.getName());
    return m_aclManager.addSupportedPermission(objectType, permission);
  }

  public boolean removeSupportedPermission(IObjectType objectType, IAclPermission permission)
    throws AclPersistenceException, PredefinedPermissionException, PermissionUsedException {

    if (objectType == null || permission == null) {
      throw new java.lang.IllegalArgumentException();
    }
    this.audit.aclSupportedPermissionRemove(permission.getName(), objectType.getName());
    return m_aclManager.removeSupportedPermission(objectType, permission);
  }

  public IAclPermission createPermission(String name)
    throws AclPersistenceException, PermissionExistsException {
    if (name == null) {
      throw new java.lang.IllegalArgumentException();
    }
    this.audit.aclPermissionCreate(name);
    return m_aclManager.createPermission(name);
  }

  public boolean removePermission(IAclPermission permission)
    throws AclPersistenceException, PredefinedPermissionException, PermissionUsedException {

    if (permission == null) {
      throw new java.lang.IllegalArgumentException();
    }
    this.audit.aclPermissionRemove(permission.getName());
    return m_aclManager.removePermission(permission);
  }

  public boolean[] areAclsUpToDate(IResourceAcl[] acls)
    throws AclPersistenceException {

    if (acls == null) {
      throw new java.lang.IllegalArgumentException();
    }

    IAcl[] baseAcls = new IAcl[acls.length];
    for (int i = 0; i < baseAcls.length; i++) {
      baseAcls[i] = ((ResourceAcl)acls[i]).getAcl();
    }
    return m_aclManager.areAclsUpToDate(baseAcls);
  }

  /**
   * UM event
   *
   * @param namespace TBD: Description of the incoming method parameter
   * @param event TBD: Description of the incoming method parameter
   * @param id TBD: Description of the incoming method parameter
   */
  public void update(String namespace, String event, String id) {

    if (namespace == null || event == null || id == null) {
      return;
    }
    try {
      if (event.equals(IUMEventHandler.USER_DELETED)) {
        deletePrincipal(id, IUMPrincipal.IUSER);
        return;
      }
      /*
      if (event.equals(IUMEventHandler.USER_UPDATED)) {
        updatePrincipal(id, IUMPrincipal.IUSER);
        return;
      }
      */
      if (event.equals(IUMEventHandler.GROUP_DELETED)) {
        deletePrincipal(id, IUMPrincipal.IGROUP);
        return;
      }
      /*
      if (event.equals(IUMEventHandler.GROUP_UPDATED)) {
        updatePrincipal(id, IUMPrincipal.IGROUP);
        return;
      }
      */
      if (event.equals(IUMEventHandler.ROLE_DELETED)) {
        deletePrincipal(id, IUMPrincipal.IROLE);
        return;
      }
      /*
      if (event.equals(IUMEventHandler.ROLE_UPDATED)) {
        updatePrincipal(id, IUMPrincipal.IROLE);
        return;
      }
      */
    }
    catch (Exception e) {
      s_log.errorT("update(651)", "error while processing usermanagement event" + LoggingFormatter.extractCallstack(e));
    }
  }

  public long getDBVersion() {
    return m_aclManager.getDBVersion();
  }

  public IClusterCacheArea getCacheArea() {
    return m_aclManager.getCacheArea();
  }

  protected void aclChanged(IResource resource)
    throws ResourceException {

    if (resource == null) {
      throw new java.lang.IllegalArgumentException();
    }
    m_aclChangeTracker.oneAclChanged(resource.getRID());
  }

  private int getMaxSortIndex(IAcl acl)
    throws AclPersistenceException {

    int maxSortIndex = Integer.MIN_VALUE;

    IAclEntryList entries = acl.getEntries();
    if (entries != null) {
      IAclEntryListIterator entryIterator = entries.iterator();
      while (entryIterator != null && entryIterator.hasNext()) {
        IAclEntry entry = entryIterator.next();
        if (entry != null) {
          int sortIndex = entry.getSortIndex();
          if (sortIndex > maxSortIndex) {
            maxSortIndex = sortIndex;
          }
        }
      }
    }

    return maxSortIndex;
  }

  private IAclPermission getMatchingPermission(IResource resource, IAclPermission permission)
    throws AclPersistenceException, ResourceException {

    IAclPermissionList supportedPermissions = getSupportedPermissions(resource);
    if (supportedPermissions != null) {
      IAclPermissionListIterator iterator = supportedPermissions.iterator();
      while (iterator != null && iterator.hasNext()) {
        IAclPermission supportedPermission = iterator.next();
        if (supportedPermission != null) {
          String supportedPermissionName = supportedPermission.getName();
          String permissionName = permission.getName();

          if (supportedPermissionName != null
             && permissionName != null
             && supportedPermissionName.equals(permissionName)) {
            return supportedPermission;
          }
        }
      }
    }

    return null;
  }

  private boolean isMatchingPrincipalPermission(IAcl acl, IUMPrincipal principal, IAclPermission permission)
    throws AclPersistenceException {

    IAclEntryList entries = acl.getEntries();
    if (entries == null) {
      IAclEntryListIterator entryIterator = entries.iterator();
      while (entryIterator != null && entryIterator.hasNext()) {
        IAclEntry entry = entryIterator.next();
        if (entry != null) {
          IAclPermission entryPermission = entry.getPermission();
          if (entryPermission != null) {
            String entryPermissionName = entryPermission.getName();
            String permissionName = permission.getName();
            IUMPrincipal entryPrincipal = entry.getPrincipal();

            if (entryPermissionName != null
               && permissionName != null
               && entryPrincipal != null
               && entryPermissionName.equals(permissionName)
               && entryPrincipal.equals(principal)) {
              return true;
            }
          }
        }
      }
    }

    return false;
  }

  private boolean isPermissionSuitableForObjectType(IAclPermission permission, IObjectType objectType)
    throws AclPersistenceException {

    IAclPermissionList supportedPermissions = m_aclManager.getSupportedPermissions(objectType);
    if (supportedPermissions != null) {
      IAclPermissionListIterator iterator = supportedPermissions.iterator();
      while (iterator != null && iterator.hasNext()) {
        IAclPermission supportedPermission = iterator.next();
        if (supportedPermission != null) {
          String supportedPermissionName = supportedPermission.getName();
          String permissionName = permission.getName();

          if (supportedPermissionName != null
             && permissionName != null
             && supportedPermissionName.equals(permissionName)) {
            return true;
          }
        }
      }
    }

    return false;
  }

  private void deletePrincipal(String principalID, int principalType)
    throws AclPersistenceException {
    m_aclManager.deletePrincipal(principalID, principalType);
    m_aclChangeTracker.someAclsChanged();
  }

  private void updatePrincipal(String principalID, int principalType)
    throws AclPersistenceException {
    m_aclManager.updatePrincipal(principalID, principalType);
    m_aclChangeTracker.someAclsChanged();
  }

  void deleteResource(IResource resource)
    throws AclPersistenceException, NotAuthorizedException, InvalidClassException, ResourceException {
    if (resource == null) {
      throw new IllegalArgumentException();
    }

    IAcl acl = m_aclManager.getAcl(getResourcePath(resource), -1);
    if (acl != null) {
      m_aclManager.removeAcl(getLoggedInUser(resource), acl);
    }
    if (resource.isCollection()) {
      m_aclManager.removeMultipleAcls(getResourcePath(resource) + '/');
    }
    m_aclChangeTracker.someAclsChanged(resource.getRID());
  }

  void moveResource(IResource from, IResource to, ResourceAclManager destAclManager)
    throws AclPersistenceException, NotAuthorizedException, AclExistsException, InvalidClassException,
    AlreadyAssignedToAclException, PermissionNotSupportedException, LastOwnerException, ResourceException {

    if (from == null) {
      throw new IllegalArgumentException();
    }

    //manfred.baedke@greenbytes.de TODO: the following code might probably be replaced by something much more efficient
    if (destAclManager == null) {
      IAcl acl = m_aclManager.getAcl(getResourcePath(from), -1);
      if (acl != null) {
        m_aclManager.changeAclID(getLoggedInUser(from), acl, getResourcePath(to));
      }
      if (from.isCollection()) {
        m_aclManager.changeMultipleAclIDs(getResourcePath(from) + '/', getResourcePath(to) + '/');
      }
      m_aclChangeTracker.someAclsChanged(from.getRID());
      m_aclChangeTracker.someAclsChanged(to.getRID());
    }
    else {
      IResourceAcl srcAcl = this.getAcl(from);
      if (srcAcl == null) {
        return;
      }
      destAclManager.deleteResource(to);
      IResourceAcl destAcl = destAclManager.createAcl(to);

      IUMPrincipalList ownerList = destAcl.getOwners();
      IUMPrincipalListIterator ownerIterator = ownerList.iterator();
      IUMPrincipal initialOwner = ownerIterator.hasNext() ? ownerIterator.next() : null;

      IUMPrincipalList srcOwnerList = srcAcl.getOwners();
      IUMPrincipalListIterator srcOwnerIter = srcOwnerList.iterator();

      boolean keepInitialOwner = false;
      while (srcOwnerIter.hasNext()) {
        IUMPrincipal owner = srcOwnerIter.next();

        if (owner.equals(initialOwner)) {

          keepInitialOwner = true;

          ArrayList destEntries = new ArrayList();
          IResourceAclEntryList destEntryList = destAcl.getEntries(owner);
          IResourceAclEntryListIterator destEntryIter = destEntryList.iterator();
          while (destEntryIter.hasNext()) {
            IResourceAclEntry entry = destEntryIter.next();
            destEntries.add(entry);
          }

          IResourceAclEntryList srcEntryList = srcAcl.getEntries(owner);
          IResourceAclEntryListIterator srcEntryIter = srcEntryList.iterator();
          while (srcEntryIter.hasNext()) {
            IResourceAclEntry entry = srcEntryIter.next();
            if (destEntries.contains(entry)) {
              destEntries.remove(entry);
            }
            else {
              destAcl.addEntry(
                this.createAclEntry(owner, entry.isNegative(), entry.getPermission(), entry.getSortIndex()));
            }
          }

          Iterator garbage = destEntries.iterator();
          while (garbage.hasNext()) {
            IResourceAclEntry entry = (IResourceAclEntry)garbage.next();
            destAcl.removeEntry(entry);
          }
        }
        else {
          destAcl.addOwner(owner);

          IResourceAclEntryList srcEntryList = srcAcl.getEntries(owner);
          IResourceAclEntryListIterator srcEntryIter = srcEntryList.iterator();
          while (srcEntryIter.hasNext()) {
            IResourceAclEntry entry = srcEntryIter.next();
            destAcl.addEntry(this.createAclEntry(owner, entry.isNegative(), entry.getPermission(), entry.getSortIndex()));
          }
        }
      }
      if (!keepInitialOwner) {
        destAcl.removeOwner(initialOwner);
      }
    }
    this.deleteResource(from);
  }

  void renameResource(String fromUri, IResource to)
    throws AclPersistenceException, NotAuthorizedException, AclExistsException, ResourceException {
    if (to == null) {
      throw new IllegalArgumentException();
    }

    IAcl acl = m_aclManager.getAcl(fromUri, -1);
    if (acl != null) {
      m_aclManager.changeAclID(getLoggedInUser(to), acl, getResourcePath(to));
    }
    if (to.isCollection()) {
      m_aclManager.changeMultipleAclIDs(fromUri + '/', getResourcePath(to) + '/');
    }
    m_aclChangeTracker.someAclsChanged(RID.getRID(fromUri));
    m_aclChangeTracker.someAclsChanged(to.getRID());
  }

  private boolean addInheritedPermissions(IUMPrincipal caller, IAcl acl, IResource resource)
    throws AclPersistenceException, NotAuthorizedException, ResourceException {
    int sortIndexToStartWith = getMaxSortIndex(acl) + 1;

    // add inherited permissions to the new acl
    IResourceAcl inheritedAcl = getInheritedAcl(resource);
    if (inheritedAcl != null) {
      // check the acl entries of the inherited acl and add the propagated ones to the new acl (in case it does not contain
      // an entry concerning the same permission for the same principal)
      IResourceAclEntryList entries = inheritedAcl.getEntries();
      if (entries != null) {
        IResourceAclEntryListIterator entryIterator = entries.iterator();
        while (entryIterator != null && entryIterator.hasNext()) {
          IResourceAclEntry inheritedEntry = entryIterator.next();
          if (inheritedEntry != null) {
            IAclPermission inheritedPermission = inheritedEntry.getPermission();
            IUMPrincipal inheritedPrincipal = inheritedEntry.getPrincipal();
            if (true
            // inheritedEntry.isPropagated() method was removed from interface
               && isPermissionSuitableForObjectType(inheritedPermission, getObjectType(resource))
               && !isMatchingPrincipalPermission(acl, inheritedPrincipal, inheritedPermission)) {
              try {
                IAclEntry newEntry = m_aclManager.createAclEntry(inheritedPrincipal, inheritedEntry.isNegative(), inheritedPermission, ++sortIndexToStartWith, true);
                if (newEntry == null || !acl.addEntry(caller, newEntry)) {
                  return false;
                }
              }
              catch (InvalidClassException e) {
                s_log.errorT("addInheritedPermissions(929)", "unexpected error" + LoggingFormatter.extractCallstack(e));
                // will not be thrown; just to keep the exception from the method prototype
              }
              catch (AlreadyAssignedToAclException e) {
                s_log.errorT("addInheritedPermissions(933)", "unexpected error" + LoggingFormatter.extractCallstack(e));
                // will not be thrown; just to keep the exception from the method prototype
              }
              catch (PermissionNotSupportedException e) {
                s_log.errorT("addInheritedPermissions(937)", "unexpected error" + LoggingFormatter.extractCallstack(e));
                // will not be thrown; just to keep the exception from the method prototype
              }
            }
          }
        }
      }
    }

    return true;
  }

  private void recursiveRemovePropagation(ResourceAcl rsrcAcl, IResourceList children, IResourceList result)
    throws AclPersistenceException {

    if (children == null) {
      return;
    }

    IResourceListIterator iterator = children.listIterator();
    while (iterator.hasNext()) {
      IResource resource = iterator.next();
      try {
        if (getAcl(resource) != null) {
          if (!removeAcl(resource)) {
            result.add(resource);
          }
        }

        if ((resource.isCollection())
           && (resource.getLinkType().equals(LinkType.NONE))
          ) {
          recursiveRemovePropagation(rsrcAcl, ((ICollection)resource).getChildren(), result);
        }
      }
      catch (AclPersistenceException e) {
        throw e;
      }
      catch (Exception e) {
        s_log.warningT("recursiveRemovePropagation(976)", "failed to propagate ACL" + LoggingFormatter.extractCallstack(e));
        result.add(resource);
      }
    }
  }

  protected static IUser getLoggedInUser(IResource resource) {

    if (resource == null) {
      throw new java.lang.IllegalArgumentException();
    }
    return resource.getContext().getUser();
  }

  private static String getResourcePath(IResource resource)
    throws ResourceException {

    return getRidPath(resource.getRID());
  }

  private static String getRidPath(RID rid) {
    return rid.getPath();
  }
}
